<?php

declare(strict_types=1);

namespace Modules\ShipMyOrders\Services;

use App\Helpers\ValidatesXml;
use App\Integrations\Http\IntegrationHttpClient as BaseClient;
use App\Models\ApiLog;
use App\Services\ShippingProvider\ShippingProviderClient;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Client\Pool;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\View;
use Modules\ShipMyOrders\Data\ShipMyOrdersCredentialsData;
use Modules\ShipMyOrders\Data\ShipMyOrdersShipmentTrackingData;
use Modules\ShipMyOrders\Entities\ShipMyOrdersIntegrationInstance;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInventory;
use RuntimeException;
use SimpleXMLElement;
use Throwable;

/**
 * Client
 *
 * SMO API is old and outdated. Take for granted a few things:
 * 1. SMO is mimicking an RPC/SOAP integration, but not strictly.
 * 2. All requests, even ones that do not mutate data on their server appear to
 *    return `201 Created` as a response code and require a `POST` HTTP method.
 * 3. There is no HTTP-level authentication. All credentials are submitted inside the
 *    request body.
 * 4. XML is barely valid, and not namespaced or wrapped in any meaningful way.
 * 5. Many specs (read: data shapes) outlined in the SMO docs are not accurate.
 * 6. They misspelled a lot of things (e.g. "AditionalData")
 */
class ShipMyOrdersClient extends BaseClient implements ShippingProviderClient
{

    use ValidatesXml;

    const GETTRACKING = 'gettracking';
    const GETINVENTORY = 'getinventory';
    /**
     * @var ShipMyOrdersIntegrationInstance
     */
    private ShipMyOrdersIntegrationInstance $instance;

    /**
     * @var ShipMyOrdersCredentialsData
     */
    protected ShipMyOrdersCredentialsData $credentials;

    /**
     * @var string|null
     */
    protected ?string $environment;

    /**
     * @param  ShipMyOrdersIntegrationInstance  $shipMyOrdersIntegrationInstance
     */
    public function __construct(
        ShipMyOrdersIntegrationInstance $shipMyOrdersIntegrationInstance,
    ) {
        $this->instance = $shipMyOrdersIntegrationInstance;

        parent::__construct($shipMyOrdersIntegrationInstance);
    }


    /**
     * @param  array  $payload
     * @return string
     * @throws Exception
     * @throws Throwable
     */
    public function sendFulfillmentOrder(array $payload): string
    {
        $url = $this->makeUrl('createorder');

        $payload = View::make(
            'shipmyorders::createorder',
            array_merge(
                $this->getAuthenticationForTransport(),
                $payload
            )
        )->render();
        if (! $this->validateXmlPayload($payload)) {
            throw new RuntimeException(
                'The XML payload for the createOrder API call is invalid.'
            );
        }

        $response = $this->post(
            url: $url,
            body: $payload,
            bodyContentType: 'text/xml',
            tries: 3
        );

        $responseBody = $response->body();

        return json_encode(simplexml_load_string($responseBody));
    }

    /**
     * @param  array  $needingUpdates
     * @return array
     * @throws Exception
     * @throws Throwable
     */
    public function getTrackingInfo(array $needingUpdates): array
    {
        $chunks = array_chunk($needingUpdates, 50);
        $tracking = [];
        $errors = [];
        $logs = [];
        $payloads = [];
        foreach ($chunks as $chunk) {

            foreach ($chunk as $orderId) {
                $url = $this->makeUrl(self::GETTRACKING);
                $urls[] = $url;
                $payload = View::make(
                    'shipmyorders::gettracking',
                    array_merge(
                        $this->getAuthenticationForTransport(),
                        [
                            'orderId' => str_replace('#', '', $orderId),
                        ]
                    )
                )->render();
                $payloads[] = $payload;
                $logs[] = $this->logApiCall(
                    $url,
                    $payload
                );
            }

            $responses = Http::pool(function(Pool $pool) use ($urls, $payloads) {
                return collect($urls)->map(
                    fn($url, $index) => $pool
                        ->withBody($payloads[$index], 'text/xml')
                        ->post($url)
                )->all();
            });

            foreach ($responses as $index => $response){

                /** @var ApiLog $log */
                $log = $logs[$index];
                $log->update([
                    'responseStatusCode' => $response->status(),
                    'responseHeaders' => $response->headers(),
                    'responseBody' => $response->body(),
                ]);

                if($response->successful()){

                    $responseXml = new SimpleXMLElement($response->body());
                    // Tracking not ready yet, so continue
                    if ((string) $responseXml->Result == 'error') {
                        continue;
                    }

                    $shipment = $this->getShippingFromResponse($response);

                    $tracking[] = [
                        'OrderID' => $chunk[$index],
                        'Shipping' => [
                            'TrackingNumber' => $shipment->getTrackingNumber(),
                            'ShippingMethod' => $shipment->getCarrier(),
                            'ShipDate' => $shipment->getShipDate()->format('m/d/Y'),
                        ],
                    ];
                } else {
                    $errors[] = $response->reason();
                }
            }
        }

        if(count($errors) == count($needingUpdates)){
            // All the requests failed
            throw new RuntimeException(
                'Error getting tracking info from SMO: ' . implode(', ', $errors)
            );
        }

        return $tracking;
    }


    /**
     * @param  Response  $response
     * @return ShipMyOrdersShipmentTrackingData
     * @throws Exception
     */
    private function getShippingFromResponse(Response $response): ShipMyOrdersShipmentTrackingData
    {
        $responseBody = $response->body();
        $desiredXPaths = [
            $trackingNumberPath = '//*[local-name()="Tracking"]',
            $shippingMethodPath = '//*[local-name()="Carrier"]',
            $shipDatePath = '//*[local-name()="ShipDate"]',
        ];
        $parsedXml = $this->parseResponseXPaths($responseBody, $desiredXPaths);

        return new ShipMyOrdersShipmentTrackingData(
            $parsedXml[$trackingNumberPath],
            $parsedXml[$shippingMethodPath],
            Carbon::createFromFormat('m/d/Y', $parsedXml[$shipDatePath])
        );
    }

    /**
     * @throws Throwable
     */
    public function getInventory(): Collection
    {
        $url = $this->makeUrl(self::GETINVENTORY);
        $payload = View::make(
            'shipmyorders::getinventory',
            $this->getAuthenticationForTransport(),
        )->render();
        $log = $this->logApiCall(
            $url,
            $payload
        );

        $response = Http::withBody($payload, 'text/xml')
            ->post($url);

        $log->update([
            'responseStatusCode' => $response->status(),
            'responseHeaders' => $response->headers(),
            'responseBody' => $response->body(),
        ]);

        $collection = collect();

        if($response->successful()){

            $responseXml = new SimpleXMLElement($response->body());

            foreach($responseXml->children() as $item) {
                // convert item to json
                $smoInventory = new ShipMyOrdersInventory();
                $smoInventory->json_object = $item;
                $collection->push($smoInventory);
            }
        } else {
            throw new RuntimeException(
                'Error getting tracking info from SMO: ' . $response->reason()
            );
        }

        return $collection;
    }

    /**
     * @return ShipMyOrdersCredentialsData
     */
    public function getCredentials(): ShipMyOrdersCredentialsData
    {
        return $this->credentials;
    }

    /**
     * @param  string  $operation
     * @return string
     */
    public static function makeUrl(string $operation): string
    {
        $baseUrl = config('shipmyorders.api_url');
        $apiPath = config('shipmyorders.api_path');
        return $baseUrl.$apiPath.$operation;
    }

    /**
     * @throws Throwable
     */
    protected function logApiCall(string $url, string $body, ?Response $response = null): ApiLog|Model
    {
        return $this->logger->logApiCall(
            $this->instance,
            $url,
            $body,
            $response
        );
    }

    /**
     * @return array
     */
    protected function getAuthenticationForTransport(): array
    {
        return [
            'username' => $this->instance->getUsername(),
            'password' => $this->instance->getPassword(),
            'clientId' => $this->instance->getClientId(),
        ];
    }

}
