<?php

namespace Modules\ShipMyOrders\Repositories;

use App\Abstractions\AbstractRepository;
use DB;
use Illuminate\Support\Collection;
use Modules\ShipMyOrders\Data\ShipMyOrdersInvoiceData;
use Modules\ShipMyOrders\Data\ShipMyOrdersInvoiceLineData;
use Modules\ShipMyOrders\Data\ShipMyOrdersInvoiceNominalCodeMappingRuleData;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoice;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoiceLine;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoiceNominalCodeMappingRule;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Throwable;

class ShipMyOrdersInvoiceRepository extends AbstractRepository
{
    /**
     * @throws Throwable
     */
    public function saveWithRelations(#[DataCollectionOf(ShipMyOrdersInvoiceData::class)] DataCollection $data): Collection|ShipMyOrdersInvoice
    {
        $data = $data->toCollection();

        return DB::transaction(function () use ($data) {
            $shipMyOrdersInvoiceCollection = $data->map(function (ShipMyOrdersInvoiceData $invoiceData) {
                return ShipMyOrdersInvoiceData::from(new ShipMyOrdersInvoice($invoiceData->toArray()));
            });
            $shipMyOrdersInvoiceCollection = $this->save($shipMyOrdersInvoiceCollection, ShipMyOrdersInvoice::class);
            $this->saveLines($shipMyOrdersInvoiceCollection, $data);

            if ($shipMyOrdersInvoiceCollection->count() === 1) {
                return ShipMyOrdersInvoice::find($shipMyOrdersInvoiceCollection->first()['id']);
            }

            return ShipMyOrdersInvoice::whereIn('id', $shipMyOrdersInvoiceCollection->pluck('id'))->get();
        });
    }

    public function update(ShipMyOrdersInvoice $shipMyOrdersInvoice, array $data): ShipMyOrdersInvoice
    {
        if(isset($data['lines']) && count($data['lines'])) {
            foreach ($data['lines'] as $item) {
                ShipMyOrdersInvoiceLine::where('ship_my_orders_invoice_id', $shipMyOrdersInvoice->id)
                    ->where('id', $item['id'])
                    ->update(['nominal_code_id' => $item['nominal_code_id']]);
            }
        }
        return $shipMyOrdersInvoice;
    }

    private function saveLines(Collection $shipMyOrdersInvoiceCollection, $data): void
    {
        if($shipMyOrdersInvoiceCollection->isEmpty()) {
            return;
        }

        $shipMyOrdersInvoiceLineCollection = collect();

        $shipMyOrdersInvoiceCollection->each(function ($shipMyOrdersInvoice) use ($data, $shipMyOrdersInvoiceLineCollection) {
            $shipMyOrdersInvoiceData = $data->where('invoice_number', $shipMyOrdersInvoice['invoice_number'])->first();
            $shipMyOrdersInvoiceData->lines->each(function (ShipMyOrdersInvoiceLineData $shipMyOrdersInvoiceLineData) use (
                $shipMyOrdersInvoiceData,
                $shipMyOrdersInvoice,
                $shipMyOrdersInvoiceLineCollection
            ) {
                $shipMyOrdersInvoiceLineData->ship_my_orders_invoice_id = $shipMyOrdersInvoice['id'];
                $shipMyOrdersInvoiceLineCollection->add($shipMyOrdersInvoiceLineData::from(new ShipMyOrdersInvoiceLine($shipMyOrdersInvoiceLineData->toArray())));
            });
        });

        $this->save($shipMyOrdersInvoiceLineCollection, ShipMyOrdersInvoiceLine::class);
    }

    public function getUnprocessedInvoices(): Collection
    {
        return ShipMyOrdersInvoice::whereNull('processed_at')->get();
    }

    public function getProcessedInvoices(): Collection
    {
        return ShipMyOrdersInvoice::whereNotNull('processed_at')->get();
    }

    public function getNominalCodesRules(): Collection
    {
        return ShipMyOrdersInvoiceNominalCodeMappingRule::orderBy('sort_order')->get();
    }

    /**
     * @throws Throwable
     */
    public function saveNominalCodesRules(#[DataCollectionOf(ShipMyOrdersInvoiceNominalCodeMappingRuleData::class)] DataCollection $rules): Collection
    {
        return DB::transaction(function () use ($rules) {
            $currentIds = $this->createOrUpdateRules($rules);
            $this->deleteRulesNotIn($currentIds);
            return ShipMyOrdersInvoiceNominalCodeMappingRule::whereIn('id', $currentIds)->get();
        });
    }

    private function createOrUpdateRules(DataCollection $rules): array
    {
        $currentIds = [];
        foreach ($rules->toArray() as $index => $rule) {
            if(!isset($rule['sort_order'])) {
                $rule['sort_order'] = $index;
            }
            if(isset($rule['id'])) {
                $currentIds[] = $rule['id'];
                ShipMyOrdersInvoiceNominalCodeMappingRule::where('id', $rule['id'])->update($rule);
            }
            else {
                $currentIds[] = ShipMyOrdersInvoiceNominalCodeMappingRule::create($rule)->id;
            }
        }
        return $currentIds;
    }

    private function deleteRulesNotIn(array $currentIds): void
    {
        ShipMyOrdersInvoiceNominalCodeMappingRule::whereNotIn('id', $currentIds)->delete();
    }
}