<?php

namespace Modules\ShipMyOrders\Repositories;

use App\Models\Product;
use App\Models\ProductComponent;
use App\Models\Warehouse;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\ShipMyOrders\Entities\ShipMyOrdersIntegrationInstance;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInventory;

class ShipMyOrdersInventoryRepository
{
    public function save(Collection $data): void
    {
        $data = $data->map(function ($item) {
            return [
                'json_object' => json_encode($item->json_object),
            ];
        });

        ShipMyOrdersInventory::truncate();

        ShipMyOrdersInventory::query()->insert($data->toArray());
    }

    public function smoWarehouse(): ?Warehouse
    {
        return Warehouse::where('integration_instance_id', ShipMyOrdersIntegrationInstance::active()->id)->first();
    }

    public function withoutSkuProducts(): Collection
    {
        return ShipMyOrdersInventory::doesntHave('product')
            ->select('id', 'SKU', 'Desc', 'UPC', 'Stock')
            ->orderBy('SKU')
            ->get();
    }

    public function withoutShipMyOrdersInventory(): Collection
    {
        return Product::doesntHave('shipMyOrdersInventory')
            ->join('products_inventory', function(JoinClause $join) {
                $join->on('products_inventory.product_id', '=', 'products.id')
                    ->where('products_inventory.warehouse_id', $this->smoWarehouse()?->id);

            })
            ->where('products_inventory.inventory_available', '>', 0)
            ->select('products.id', 'sku', 'name')
            ->selectRaw('products_inventory.inventory_available as inventory')
            ->orderBy('sku')
            ->get();
    }

    public function inventoryDiscrepancies(): Collection
    {
        // Subquery to count product components
        $productComponentsCount = ProductComponent::query()
            ->whereHas('parentProduct', function ($query) {
                $query->where('type', Product::TYPE_KIT);
            })
            ->select('component_product_id', DB::raw('count(*) as component_count'))
            ->groupBy('component_product_id');

        return ShipMyOrdersInventory::query()
            ->join('products', 'products.SKU', '=', 'ship_my_orders_inventory.SKU')
            ->leftJoin('products_inventory', function (JoinClause $join) {
                $join->on('products_inventory.product_id', '=', 'products.id')
                    ->where('products_inventory.warehouse_id', $this->smoWarehouse()?->id);
            })
            ->leftJoinSub($productComponentsCount, 'product_components_count', function ($join) {
                $join->on('products.id', '=', 'product_components_count.component_product_id');
            })
            ->selectRaw('products.id, products.sku, products.type, COALESCE(products_inventory.inventory_available, 0) as sku_inventory, ship_my_orders_inventory.Stock as smo_inventory, product_components_count.component_count')
            ->selectRaw('ship_my_orders_inventory.Stock - products_inventory.inventory_available as discrepancy')
            ->whereRaw('COALESCE(products_inventory.inventory_available, 0) != ship_my_orders_inventory.Stock')
            ->orderBy('products.sku')
            ->get();
    }
}