<?php

namespace Modules\ShipMyOrders\Managers;

use App\Data\FinancialLineData;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Exceptions\StoredFileDoesNotExistException;
use App\Helpers;
use App\Models\FinancialLine;
use App\Repositories\FinancialLineRepository;
use App\Repositories\SalesOrderFulfillmentRepository;
use Carbon\Carbon;
use DB;
use Exception;
use Illuminate\Support\Collection;
use Modules\ShipMyOrders\Data\ShipMyOrdersInvoiceData;
use Modules\ShipMyOrders\Data\ShipMyOrdersInvoiceLineData;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoice;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoiceLine;
use Modules\ShipMyOrders\Exceptions\SmoInvoiceAlreadyExistsException;
use Modules\ShipMyOrders\Exceptions\SmoInvoiceFileEmptyException;
use Modules\ShipMyOrders\Repositories\ShipMyOrdersInvoiceRepository;
use Storage;
use Throwable;

class ShipMyOrdersInvoiceManager
{
    public function __construct(
        private readonly ShipMyOrdersInvoiceRepository $invoices,
        private readonly SalesOrderFulfillmentRepository $fulfillments,
        private readonly FinancialLineRepository $financialLines,
        private readonly ShipMyOrdersInvoiceNominalCodeMappingRuleManager $smoInvoiceRuleManager,
    )
    {
    }

    /**
     * @throws StoredFileDoesNotExistException
     * @throws SmoInvoiceFileEmptyException
     * @throws SmoInvoiceAlreadyExistsException
     * @throws Throwable
     */
    public function storeInvoice(string $storedFilename): ShipMyOrdersInvoice
    {
        if (!Storage::disk('smo-invoices')->exists($storedFilename)) {
            throw new StoredFileDoesNotExistException();
        }

        $file = Storage::disk('smo-invoices')->path($storedFilename);
        Storage::disk('smo-invoices')
            ->prepend($storedFilename, '"C1","invoice_number","date","reference","description","amount","quantity","subtotal","tax"');
        $data = Helpers::csvFileToCollection($file);

        if ($data->count() === 0) {
            Storage::disk('smo-invoices')->delete($storedFilename);
            throw new SmoInvoiceFileEmptyException();
        }

        $invoiceNumber = $data->first()['invoice_number'];

        // check if invoice is existing
        if(ShipMyOrdersInvoice::where('invoice_number', $invoiceNumber)->exists()) {
            Storage::disk('smo-invoices')->delete($storedFilename);
            throw new SmoInvoiceAlreadyExistsException();
        }

        $invoiceDate  = $data->max(function ($item) {
            return Carbon::parse($item['date']);
        });
        $invoiceTotal = $data->sum(function ($item) {
            return (double) $item['subtotal'] + floatval($item['tax']);
        });

        $shipMyOrdersInvoiceLineData = $data->map(fn($item) => $this->mapLine($item));

        $shipMyOrdersInvoiceData = ShipMyOrdersInvoiceData::collection([ShipMyOrdersInvoiceData::from([
            'invoice_number' => $invoiceNumber,
            'invoice_date' => $invoiceDate,
            'total' => $invoiceTotal,
            'lines' => $shipMyOrdersInvoiceLineData
        ])]);

        $invoice = $this->invoices->saveWithRelations($shipMyOrdersInvoiceData);

        Storage::disk('smo-invoices')->delete($storedFilename);

        $this->processInvoice($invoice->refresh());

        return $invoice->refresh();
    }

    private function mapLine(array $item): ShipMyOrdersInvoiceLineData
    {
        return ShipMyOrdersInvoiceLineData::from([
            'date' => $item['date'],
            'reference' => $item['reference'],
            'description' => $item['description'],
            'amount' => $item['amount'],
            'quantity' => $item['quantity'],
            'subtotal' => $item['subtotal'],
            'tax' => floatval($item['tax']),
            'total' => (double) $item['subtotal'] + floatval($item['tax'])
        ]);
    }

    public function processAllInvoices(): Collection
    {
        return $this->processInvoices($this->invoices->getUnprocessedInvoices());
    }

    public function unprocessAllInvoices(): Collection
    {
        return $this->unprocessInvoices($this->invoices->getProcessedInvoices());
    }

    public function processInvoices(Collection $invoices): Collection
    {
        $processedInvoices = collect();
        $invoices->each(/**
         * @throws Throwable
         */ function (ShipMyOrdersInvoice $invoice) use ($processedInvoices) {
            $processedInvoices->push($this->processInvoice($invoice));
        });
        return $processedInvoices;
    }

    public function unprocessInvoices(Collection $invoices): Collection
    {
        $processedInvoices = collect();
        $invoices->each(/**
         * @throws Throwable
         */ function (ShipMyOrdersInvoice $invoice) use ($processedInvoices) {
            $processedInvoices->push($this->unprocessInvoice($invoice));
        });
        return $processedInvoices;
    }

    /**
     * @throws Throwable
     */
    public function processInvoice(ShipMyOrdersInvoice $invoice): ShipMyOrdersInvoice
    {
        DB::transaction(function () use ($invoice) {
            $invoice->shipMyOrdersInvoiceLines->each(fn($invoiceLine) => $this->processInvoiceLine($invoiceLine));
            $invoice->processed_at = now();
            $invoice->save();
        });

        return $invoice->refresh();
    }

    /**
     * @throws Exception
     */
    public function processInvoiceLine(ShipMyOrdersInvoiceLine $invoiceLine): void
    {

        if (!$invoiceLine->nominal_code_id) {
            $nominalCode = $this->smoInvoiceRuleManager->getNominalCodeForInvoiceLine($invoiceLine);
            if($nominalCode) {
                $invoiceLine->nominal_code_id = $nominalCode->id;
            }
        }

        // look for matching fulfillment using reference
        // if match found, create cost line and link it

        $reference = $invoiceLine->reference;

        if (!$salesOrderFulfillment = $this->fulfillments->getForReference($reference))
        {
            $invoiceLine->save();
            return;
        }

        $financialLineData = FinancialLineData::from([
            'sales_order_id' => $salesOrderFulfillment->sales_order_id,
            'financial_line_type_id' => $this->financialLines->getOrCreateFinancialLineType('Fulfillment', FinancialLineClassificationEnum::COST)->id,
            'description' => $invoiceLine->description,
            'quantity' => $invoiceLine->quantity,
            'amount' => $invoiceLine->amount,
            'tax_allocation' => $invoiceLine->tax,
            'allocate_to_products' => true,
            'proration_strategy' => FinancialLineProrationStrategyEnum::REVENUE_BASED,
        ]);

        $financialLine = FinancialLine::create($financialLineData->toArray());

        $invoiceLine->sales_order_fulfillment_id = $salesOrderFulfillment->id;
        $invoiceLine->financial_line_id = $financialLine->id;
        $invoiceLine->processed_at = now();
        $invoiceLine->save();
    }

    /**
     * @throws Throwable
     */
    public function unprocessInvoice(ShipMyOrdersInvoice $invoice): ShipMyOrdersInvoice
    {
        if (!$invoice->processed_at) {
            return $invoice;
        }

        DB::transaction(function () use ($invoice) {
            $invoice->shipMyOrdersInvoiceLines->each(fn($invoiceLine) => $this->unprocessInvoiceLine($invoiceLine));
            $invoice->processed_at = null;
            $invoice->save();
        });

        return $invoice->refresh();
    }

    /**
     * @throws Exception
     */
    public function unprocessInvoiceLine(ShipMyOrdersInvoiceLine $invoiceLine): void
    {
        if ($invoiceLine->financialLine) {
            $invoiceLine->sales_order_fulfillment_id = null;
            $invoiceLine->financialLine->delete();
            $invoiceLine->financial_line_id = null;
            $invoiceLine->processed_at = null;
            $invoiceLine->save();
        }
    }
}