<?php

namespace Modules\ShipMyOrders\Http\Controllers;

use App\DataTable\DataTable;
use App\Exceptions\StoredFileDoesNotExistException;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Traits\BulkDeleteProcessor;
use App\Http\Controllers\Traits\BulkOperation;
use App\Response;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Modules\ShipMyOrders\Data\ShipMyOrdersInvoiceData;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoice;
use Modules\ShipMyOrders\Exceptions\SmoInvoiceAlreadyExistsException;
use Modules\ShipMyOrders\Exceptions\SmoInvoiceFileEmptyException;
use Modules\ShipMyOrders\Http\Resources\ShipMyOrderInvoiceResource;
use Modules\ShipMyOrders\Managers\ShipMyOrdersInvoiceManager;
use Modules\ShipMyOrders\Repositories\ShipMyOrdersInvoiceRepository;
use Throwable;

class ShipMyOrderInvoiceController extends Controller
{
    use BulkDeleteProcessor, BulkOperation, DataTable;

    protected string $model_path = ShipMyOrdersInvoice::class;

    public function __construct(private ShipMyOrdersInvoiceRepository $repository)
    {
        parent::__construct();
    }

    public function show($id)
    {
        $shipMyOrdersInvoice = ShipMyOrdersInvoice::findOrFail($id);
        $shipMyOrdersInvoice->load('shipMyOrdersInvoiceLines.salesOrderFulfillment.salesOrder');
        $shipMyOrdersInvoice->load('shipMyOrdersInvoiceLines.nominalCode');

        return $this->response->addData(ShipMyOrderInvoiceResource::make($shipMyOrdersInvoice));
    }

    /**
     * @throws StoredFileDoesNotExistException
     * @throws Throwable
     */
    public function store(Request $request): Response
    {
        $request->validate(['stored_name' => 'required']);

        try {
            $invoice = app(ShipMyOrdersInvoiceManager::class)->storeInvoice($request->get('stored_name'));
        } catch (StoredFileDoesNotExistException) {
            return $this->response->addError(
                'File not exists', Response::HTTP_BAD_REQUEST, ''
            )->error(Response::HTTP_BAD_REQUEST);
        } catch (SmoInvoiceFileEmptyException) {
            return $this->response->addError(
                'The file is empty', Response::HTTP_BAD_REQUEST, ''
            )->error(Response::HTTP_BAD_REQUEST);
        } catch (SmoInvoiceAlreadyExistsException) {
            return $this->response->addError(
                'Invoice already exists', Response::HTTP_BAD_REQUEST, ''
            )->error(Response::HTTP_BAD_REQUEST);
        }

        $invoice->load('shipMyOrdersInvoiceLines.salesOrderFulfillment.salesOrder');
        $invoice->load('shipMyOrdersInvoiceLines.nominalCode');
        return $this->response->addData(ShipMyOrderInvoiceResource::make($invoice));
    }

    public function update(ShipMyOrdersInvoice $shipMyOrdersInvoice, ShipMyOrdersInvoiceData $request): Response
    {
        $this->repository->update($shipMyOrdersInvoice, $request->toArray());

        $shipMyOrdersInvoice->refresh();
        $shipMyOrdersInvoice->load('shipMyOrdersInvoiceLines.salesOrderFulfillment.salesOrder');
        $shipMyOrdersInvoice->load('shipMyOrdersInvoiceLines.nominalCode');

        return $this->response->addData(ShipMyOrderInvoiceResource::make($shipMyOrdersInvoice));
    }

    public function destroy($smoInvoiceID)
    {
        $shipMyOrdersInvoice = ShipMyOrdersInvoice::findOrFail($smoInvoiceID);
        $shipMyOrdersInvoice->delete();

        return $this->response->setMessage(__('messages.success.delete', [
            'resource' => 'smo invoice',
            'id' => $shipMyOrdersInvoice->invoice_number,
        ]));
    }


    public function bulkDestroy(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_DELETE);
    }


    public function archive($smoInvoiceID): JsonResponse
    {
        $smoInvoice = ShipMyOrdersInvoice::with(
            'shipMyOrdersInvoiceLines.salesOrderFulfillment.salesOrder',
            'shipMyOrdersInvoiceLines.nominalCode'
        )->findOrFail($smoInvoiceID);

        if ($smoInvoice->archive()) {
            return $this->response
                ->setMessage(__('messages.success.archive', [
                    'resource' => 'smo invoice',
                    'id' => $smoInvoice->invoice_number,
                ]))
                ->addData(ShipMyOrderInvoiceResource::make($smoInvoice));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.already_archive', [
                'resource' => 'smo invoice',
                'id' => $smoInvoice->invoice_number,
            ]), 'ProductBrand'.Response::CODE_ALREADY_ARCHIVED, 'id', ['id' => $smoInvoice->id])
            ->addData(ShipMyOrderInvoiceResource::make($smoInvoice));
    }

    public function unarchived($smoInvoiceID): JsonResponse
    {
        $smoInvoice = ShipMyOrdersInvoice::with(
            'shipMyOrdersInvoiceLines.salesOrderFulfillment.salesOrder',
            'shipMyOrdersInvoiceLines.nominalCode'
        )->findOrFail($smoInvoiceID);

        if ($smoInvoice->unarchived()) {
            return $this->response
                ->setMessage(__('messages.success.unarchived', [
                    'resource' => 'smo invoice',
                    'id' => $smoInvoice->invoice_number,
                ]))
                ->addData(ShipMyOrderInvoiceResource::make($smoInvoice));
        }

        return $this->response->warning()
            ->addWarning(__('messages.failed.unarchived', [
                'resource' => 'smo invoice',
                'id' => $smoInvoice->invoice_number,
            ]), 'ProductBrand'.Response::CODE_ALREADY_UNARCHIVED, 'id', ['id' => $smoInvoice->id])
            ->addData(ShipMyOrderInvoiceResource::make($smoInvoice));
    }

    public function bulkArchive(Request $request): Response
    {
        return $this->bulkOperation($request, $this->BULK_ARCHIVE);
    }


    public function bulkUnArchive(Request $request)
    {
        return $this->bulkOperation($request, $this->BULK_UN_ARCHIVE);
    }

    public function isDeletable(Request $request): Response
    {
        // validate
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:ship_my_orders_invoices,id',
        ]);

        $ids = array_unique($request->input('ids', []));

        $result = [];
        $smoInvoices = ShipMyOrdersInvoice::with([])->whereIn('id', $ids)->select('id', 'invoice_number')->get();
        foreach ($smoInvoices as $key => $smoInvoice) {
            $isUsed = $smoInvoice->isUsed();

            $result[$key] = $smoInvoice->only('id', 'invoice_number');
            $result[$key]['deletable'] = !$isUsed;
            $result[$key]['reason'] = $isUsed ?: null;
        }

        return $this->response->addData($result);
    }

    /**
     * @throws Throwable
     */
    public function process(ShipMyOrdersInvoice $invoice): Response
    {
        $invoice = app(ShipMyOrdersInvoiceManager::class)->processInvoice($invoice);

        $invoice->load('shipMyOrdersInvoiceLines.salesOrderFulfillment.salesOrder');
        $invoice->load('shipMyOrdersInvoiceLines.nominalCode');

        return $this->response->addData(ShipMyOrderInvoiceResource::make($invoice));
    }

    public function bulkProcess(Request $request): Response
    {
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:ship_my_orders_invoices,id',
        ]);

        $data = app(ShipMyOrdersInvoiceManager::class)->processInvoices(
            ShipMyOrdersInvoice::whereIn('id', $request->input('ids'))->get()
        );

        return $this->response->addData(ShipMyOrderInvoiceResource::collection($data));
    }

    public function processAll(): Response
    {
        $data = app(ShipMyOrdersInvoiceManager::class)->processAllInvoices();

        return $this->response->addData(ShipMyOrderInvoiceResource::collection($data));
    }

    /**
     * @throws Throwable
     */
    public function unprocess(ShipMyOrdersInvoice $invoice): Response
    {
        $invoice = app(ShipMyOrdersInvoiceManager::class)->unprocessInvoice($invoice);

        $invoice->load('shipMyOrdersInvoiceLines.salesOrderFulfillment.salesOrder');
        $invoice->load('shipMyOrdersInvoiceLines.nominalCode');

        return $this->response->addData(ShipMyOrderInvoiceResource::make($invoice));
    }

    public function bulkUnprocess(Request $request): Response
    {
        $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:ship_my_orders_invoices,id',
        ]);

        $data = app(ShipMyOrdersInvoiceManager::class)->unprocessInvoices(
            ShipMyOrdersInvoice::whereIn('id', $request->input('ids'))->get()
        );

        return $this->response->addData(ShipMyOrderInvoiceResource::collection($data));
    }

    public function unprocessAll(): Response
    {
        $data = app(ShipMyOrdersInvoiceManager::class)->unprocessAllInvoices();

        return $this->response->addData(ShipMyOrderInvoiceResource::collection($data));
    }

    protected function getModel()
    {
        return ShipMyOrdersInvoice::class;
    }

    protected function getResource()
    {
        return ShipMyOrderInvoiceResource::class;
    }
}