<?php

declare(strict_types=1);

namespace Modules\ShipMyOrders\Entities;

use App\Models\ApiLog;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Warehouse;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * Class ShipMyOrdersIntegrationInstance
 */
class ShipMyOrdersIntegrationInstance extends IntegrationInstance
{
    use HasFactory;

    protected $table = 'integration_instances';

    protected static function boot()
    {
        static::creating(function (self $model) {
            $model->mergeDefaultInstanceConnectionSettings();
        });

        static::addGlobalScope('shipmyorders', function (Builder $query) {
            return $query->whereHas('integration', function (Builder $query) {
                $query->where('name', Integration::NAME_SHIPMYORDERS);
            });
        });

        parent::boot();
    }

    public static function active(): ShipMyOrdersIntegrationInstance|Model|null
    {
        return static::query()->first();
    }

    /*
     |--------------------------------------------------------------------------
     | Relations
     |--------------------------------------------------------------------------
     */
    public function apiLogs(): HasMany
    {
        return $this->hasMany(ApiLog::class, 'integration_instance_id');
    }

    public function getUsername(): string{
        return data_get($this->connection_settings, 'username');
    }

    /**
     * @return string
     */
    public function getPassword(): string{
        return data_get($this->connection_settings, 'password');
    }

    public function getClientId(): string{
        return data_get($this->connection_settings, 'clientId');
    }

    public function linkedWarehouse(): Attribute{
        return Attribute::get(fn() => Warehouse::query()->findOrFail($this->integration_settings['linked_warehouse_id']));
    }

    /**
     * Get a connection setting with dot notation
     * e.g. `foo.bar` will resolve to ShipMyOrdersIntegrationInstance->connection_settings['foo']['bar']
     *
     *
     * @return array|mixed
     */
    public function getConnectionSetting(string $settingName): mixed
    {
        return data_get($this->connection_settings, $settingName);
    }

    protected function mergeDefaultInstanceConnectionSettings(): void
    {
        $this->connection_settings = array_merge(
            (array) $this->connection_settings,
            (array) $this->integration->integration_settings_template
        );
    }
}
