<?php

namespace Modules\Qbo\Services;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Enums\HttpMethodEnum;
use Illuminate\Http\Client\Response;
use Modules\Qbo\ApiParameterObjects\QboDeletePaymentsApo;
use Modules\Qbo\Data\QboBillData;
use Modules\Qbo\Data\QboCreditMemoData;
use Modules\Qbo\Data\QboInvoiceData;
use Modules\Qbo\Data\QboJournalData;
use Modules\Qbo\Data\QboPaymentData;
use Modules\Qbo\Data\QboPurchaseOrderData;
use Modules\Qbo\Data\QboResponseData;
use Modules\Qbo\Exceptions\QboSystemFailureException;

class QboBatchClient extends QboAuthenticationClient
{
    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);
    }

    /**
     * @throws Exception
     */
    public function batch(array $body): Response
    {
        return $this->request(
            HttpMethodEnum::POST,
            '/batch',
            [
                'body' => json_encode($body),
            ]
        );
    }

    public function updateOrCreateBatchRecords(ApiDataTransformerInterface $parameters, string $qboData)
    {
        $batchRequestEntity = null;

        if ($qboData === QboBillData::class) {
            $batchRequestEntity = 'Bill';
        } elseif ($qboData === QboPurchaseOrderData::class) {
            $batchRequestEntity = 'PurchaseOrder';
        } elseif ($qboData === QboPaymentData::class) {
            $batchRequestEntity = 'Payment';
        } elseif ($qboData === QboCreditMemoData::class) {
            $batchRequestEntity = 'CreditMemo';
        } elseif ($qboData === QboInvoiceData::class) {
            $batchRequestEntity = 'Invoice';
        } elseif ($qboData === QboJournalData::class) {
            $batchRequestEntity = 'JournalEntry';
        }

        $requestParams = $parameters->transform();

        $body = [];
        $i = 1;
        foreach ($requestParams['records'] as $record) {
            $body['BatchItemRequest'][] = [
                'bId' => 'bid'.$i++,
                'operation' => @$record['Id'] ? 'update' : 'create', // TODO: Handle update or create
                $batchRequestEntity => $record,
            ];
        }

        $response = $this->batch($body);

        $dataCollection = collect();
        $syncTokenErrors = collect();
        $syncErrors = collect();

        if (isset($response->json()['BatchItemResponse'])) {
            collect($response->json()['BatchItemResponse'])->each(function (array $bill) use ($qboData, $batchRequestEntity, &$dataCollection, &$syncTokenErrors, &$syncErrors, $body) {
                if (@$bill[$batchRequestEntity]) {
                    $return = $bill[$batchRequestEntity];
                    $return = array_merge($return, [
                        'bId' => $bill['bId'],
                        'json_object' => $return,
                    ]);

                    $dataCollection->push($qboData::from($return));
                } elseif (@$bill['Fault']['Error'][0]['Message'] == 'Stale Object Error') {
                    $bid = $bill['bId'];
                    $syncTokenErrors->push(collect($body['BatchItemRequest'])->where('bId', $bid)->first());
                } elseif (isset($bill['Fault']['Error'])) {
                    $bid = $bill['bId'];
                    $return = [
                        'response' => $bill,
                        'request' => collect($body['BatchItemRequest'])->where('bId', $bid)->first()[$batchRequestEntity],
                    ];
                    $syncErrors->push($return);
                } else {
                    throw new \Exception('Error Processing Request', 1);
                }

                return null;
            });
        } elseif (! is_null($response->json())) {
            if (@$response->json()['Fault']['Error'][0]['Message'] == 'An application error has occurred while processing your request') {
                throw new QboSystemFailureException();
            } else {
                throw new \Exception('Missing BatchItemResponse:'.json_encode($response->json()), 1);
            }
        }

        return QboResponseData::from([
            'collection' => $dataCollection,
            'errors' => $syncErrors,
            'syncTokenErrors' => $syncTokenErrors,
        ]);
    }

    public function deleteBatchRecords(ApiDataTransformerInterface|QboDeletePaymentsApo $parameters): QboResponseData
    {
        $requestParams = $parameters->transform();
        $paymentIdsDeleted = collect();

        $body = [];
        $i = 1;
        foreach ($requestParams['records'] as $record) {
            $body['BatchItemRequest'][] = [
                'bId' => 'bid'.$i++,
                'operation' => 'delete',
                'Payment' => $record,
            ];
        }

        $response = $this->batch($body);

        if (isset($response->json()['BatchItemResponse'])) {
            collect($response->json()['BatchItemResponse'])->each(function (array $payment) use (&$paymentIdsDeleted, &$syncTokenErrors, &$syncErrors) {
                if (@$payment['Payment']) {
                    $paymentIdsDeleted->push($payment['Payment']['Id']);
                } else {
                    throw new \Exception('Error Processing Request', 1);
                }

                return null;
            });
        }

        return QboResponseData::from([
            'collection' => $paymentIdsDeleted,
            'errors' => collect(),
            'syncTokenErrors' => collect(),
        ]);
    }
}
