<?php

namespace Modules\Qbo\Services;

use App\Abstractions\Integrations\AbstractIntegrationClient;
use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\Concerns\OauthAuthenticationTrait;
use App\Abstractions\Integrations\Data\OauthRefreshTokenResponseData;
use App\Abstractions\Integrations\IntegrationClientInterface;
use App\Abstractions\Integrations\IntegrationHeaderDtoInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\OauthAuthenticationClientInterface;
use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use App\Models\ApiLog;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Modules\Qbo\ApiParameterObjects\QboQueryApo;
use Modules\Qbo\Data\QboHeaderData;
use Modules\Qbo\Data\QboOauthRefreshTokenResponseData;
use Modules\Qbo\Data\QboResponseData;
use Modules\Qbo\Exceptions\QboRefreshAccessTokenException;
use Modules\Qbo\Exceptions\QboSystemFailureException;
use Modules\Qbo\Exceptions\QboTimeoutException;
use Modules\Qbo\Repositories\QboIntegrationInstanceRepository;

class QboAuthenticationClient extends AbstractIntegrationClient implements IntegrationClientInterface, OauthAuthenticationClientInterface
{
    use OauthAuthenticationTrait;

    private string $client;

    private string $secret;

    private array $parameters;

    private string $path;

    private string $httpMethod;

    /**
     * @throws Exception
     */
    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);
        $this->client = config('qbo.clientId', '');
        $this->secret = config('qbo.clientSecret', '');

        if ($refreshToken = @$integrationInstance->connection_settings['tokens']['refreshToken']) {
            $this->setClientCredentials($refreshToken);
        }

        $baseUrl = config('qbo.sandbox') ? config('qbo.sandboxUrl') : config('qbo.productionUrl');

        if ($integrationInstance->refresh_token) {
            $this->setClientCredentials($integrationInstance->refresh_token);
        }

        if (@$integrationInstance->connection_settings['realmId']) {
            $this->setBaseUrl($baseUrl.'/v3/company/'.$integrationInstance->connection_settings['realmId']);
        }

        // if ($this->getAccessToken()) {
        //     if ($this->isAccessTokenExpired()) {
        //         $this->refreshAccessToken();
        //     }
        //     $this->setBaseUrl($baseUrl.'/v3/company/'.$integrationInstance->connection_settings['callbackResponse']['realmId']);
        // }
    }

    public function client(): PendingRequest
    {
        return Http::withHeaders($this->requestHeaders);
    }

    public function getRedirectUrl(): string
    {
        $params = [
            'client_id' => $this->client,
            'scope' => config('qbo.scope'),
            'redirect_uri' => config('qbo.callbackUrl'),
            'response_type' => 'code',
            'state' => config('app.url').'_'.$this->integrationInstance->id,
        ];

        return config('qbo.urlAuthorize').'?'.http_build_query($params);
    }

    public function isAccessTokenExpired(): bool
    {
        $expires = $this->integrationInstance->connection_settings['expiresAccessToken'] ?? 0;

        return (time() + 60) >= $expires; //60 seconds are added as existing job might in progress.
    }

    /**
     * @throws Exception
     */
    public function getRefreshTokenFromAuthCode(string $authCode): OauthRefreshTokenResponseData
    {
        $response = Http::withHeaders([
            'Accept' => 'application/json',
            'Content-Type' => 'application/x-www-form-urlencoded',
        ])->asForm()->post(
            config('qbo.urlAccessToken'),
            [
                'grant_type' => 'authorization_code',
                'code' => $authCode,
                'redirect_uri' => config('qbo.callbackUrl'),
                'client_id' => $this->client,
                'client_secret' => $this->secret,
            ]
        );

        if (! $response->ok()) {
            throw new Exception('Failed to refresh access token for QBO');
        }

        return QboOauthRefreshTokenResponseData::from($response->json());
    }

    /**
     * Generate access_token using refresh_token.
     *
     * @return ?string
     *
     * @throws Exception
     */
    public function getAccessToken(): ?string
    {
        if (! $this->isAccessTokenExpired()) {
            return $this->integrationInstance->connection_settings['access_token'];
        } else {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/x-www-form-urlencoded',
            ])->asForm()->post(
                config('qbo.urlAccessToken'),
                [
                    'grant_type' => 'refresh_token',
                    'refresh_token' => $this->getRefreshToken(),
                    'redirect_uri' => config('qbo.callbackUrl'),
                    'client_id' => $this->client,
                    'client_secret' => $this->secret,
                ]
            );

            if (! $response->ok() && is_null(@$response->json()['access_token'])) {
                throw new QboRefreshAccessTokenException('Failed to refresh access token for QBO');
            }

            $tokens = QboOauthRefreshTokenResponseData::from($response->json());

            app(QboIntegrationInstanceRepository::class)->updateConnectionSettings($this->integrationInstance, $tokens->toArray());

            return $tokens->access_token;
        }
    }

    /**
     * @throws Exception
     */
    public function setRequestHeaders(IntegrationHeaderDtoInterface|QboHeaderData $headerDto): void
    {
        // Request headers
        $this->requestHeaders = [
            'Authorization' => 'Bearer '.$this->getAccessToken(),
            'Accept' => 'application/json',
        ];
    }

    /**
     * @throws Exception
     */
    public function request(HttpMethodEnum $httpMethod, string $path, array $parameters = []): Response
    {
        $this->path = $path;
        $this->httpMethod = strtoupper($httpMethod->value);

        if (count($parameters) > 0) {
            $parameters = collect($parameters)->sortKeys()->toArray();
        }

        $this->parameters = array_merge($parameters, [
            // 'minorversion' => 70,
        ]);

        $this->setRequestHeaders(QboHeaderData::from([]));

        $apiLog = null;
        if ($this->isApiLoggingEnabled) {
            $apiLog = ApiLog::create([
                'integration_instance_id' => $this->integrationInstance->id,
                'url' => $this->getBaseUrl().$path,
                'requestHeaders' => $this->getRequestHeaders(),
                'requestBody' => json_encode($this->parameters),
                'responseStatusCode' => null,
                'responseHeaders' => null,
                'responseBody' => null,
            ]);
        }

        $client = $this->client();
        if (isset($parameters['body'])) {
            $client = $client->withBody($parameters['body'], isJson($parameters['body']) ? 'application/json' : 'application/text');
            unset($this->parameters['body']);
        }

        $response = $client->{$httpMethod->value}($this->getBaseUrl().$path.'?minorversion=70', $this->parameters);

        if ($this->isApiLoggingEnabled) {
            $apiLog->update([
                'responseStatusCode' => $response->status(),
                'responseHeaders' => $response->headers(),
                'responseBody' => $response->body(),
            ]);
        }

        if ($response->status() === 429) {
            $statusCode = $response->status();
            throw new ApiException('Throttling error', $statusCode);
        }

        return $response;
    }

    /**
     * @throws Exception
     */
    public function query(string $query): Response
    {
        return $this->request(
            HttpMethodEnum::POST,
            '/query',
            [
                'body' => $query,
            ]
        );
    }

    /**
     * @throws Exception
     */
    public function getData(string $entityType, string $dtoClass, QboQueryApo|ApiDataTransformerInterface $parameters): QboResponseData
    {
        // TODO: Handle where clause "WHERE DisplayName = '{$this->displayName}'".parent::transformQuery();

        $whereClauseArray = [];
        $whereClause = '';
        foreach ((@$parameters->transform()['whereClause'] ?? []) as $key => $value) {
            if (! is_array($value)) {
                $value = addslashes($value);
                $whereClauseArray[] = "$key = '$value'";

                $whereClause = '';
                if (count($whereClauseArray)) {
                    $whereClause = ' WHERE '.implode(' AND ', $whereClauseArray);
                }
            } else {
                if (count($value) > 0) {
                    foreach ($value as $_value) {
                        $_value = addslashes($_value);
                        $whereClauseArray[] = "$key = '$_value'";
                    }
                    $whereClause = ' WHERE '.implode(' OR ', $whereClauseArray);
                }
            }
        }

        try {
            // $response = $this->query("Select * from Bill Where Id = '2561' STARTPOSITION 1 MAXRESULTS 1000");
            $response = $this->query("Select * from {$entityType} $whereClause STARTPOSITION {$parameters->startPosition} MAXRESULTS {$parameters->maxResults}");
        } catch (ConnectionException $e) {
            // Check if the error message contains 'cURL error 28'
            if (str_contains($e->getMessage(), 'cURL error 28: Operation timed out after')) {
                throw new QboTimeoutException('Qbo API timed out. Please try again later.');
            }
            throw $e;
        }

        $collection = new Collection();

        if (isset($response->json()['QueryResponse'])) {
            if (! is_null(@$response->json()['QueryResponse'][$entityType]) && is_array($response->json()['QueryResponse'][$entityType])) {
                foreach (@$response->json()['QueryResponse'][$entityType] as $item) {
                    $item['json_object'] = $item;
                    $entity = $dtoClass::from($item);
                    $collection->add(item: $entity);
                }
            }
        } elseif (@$response->json()['Fault']['Error'][0]['Message'] == 'An application error has occurred while processing your request') {
            throw new QboSystemFailureException();
        } else {
            throw new Exception('Missing QueryResponse:'.json_encode($response->json()), 1);
        }

        /*
         *  NOTE: count(*) opens up an issue where the result set could change
         *  https://docs.google.com/spreadsheets/d/1PIYHPKdhZBTO7rgVrk3VdJZ_hG5YGjZGjsKYy0aYuBE/edit?usp=sharing
         *
         * If we did ascending order, then i don't think there is an issue... the only problem is descending and using the count(*) method
         */

        return QboResponseData::from([
            'collection' => $collection,
            'startPosition' => @$response->json()['QueryResponse']['startPosition'],
            'maxResults' => @$response->json()['QueryResponse']['maxResults'],
        ]);
    }
}
