<?php

namespace Modules\Qbo\Repositories;

use App\Abstractions\Integrations\AbstractIntegrationRepository;
use App\Abstractions\Integrations\Accounting\AccountingVendorRepositoryInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Models\Supplier;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Query\JoinClause;
use Modules\Qbo\Data\QboVendorData;
use Modules\Qbo\Entities\QboIntegrationInstance;
use Modules\Qbo\Entities\QboVendor;
use Spatie\LaravelData\Data;

class QboVendorRepository extends AbstractIntegrationRepository implements AccountingVendorRepositoryInterface
{
    public function getModelClassName(): string
    {
        return QboVendor::class;
    }

    public function getStartDateField(): string
    {
        return 'LastUpdatedTimeUtc';
    }

    public function getStartDateForNew(IntegrationInstanceInterface $integrationInstance): Carbon
    {
        return Carbon::now();
    }

    public function getQboVendorByDisplayName(string $supplierName): QboVendor
    {
        return QboVendor::whereHasMorph(
            'link',
            [Supplier::class],
            function ($query) use ($supplierName) {
                $query->where('name', $supplierName);
            }
        )->first();
    }

    public function getVendorsNeedingUpdate(array $supplier_ids = []): Collection
    {
        $query = Supplier::query()
            ->leftJoin('qbo_vendors', function (JoinClause $join) {
                $join->on('qbo_vendors.link_id', 'suppliers.id');
                $join->where('qbo_vendors.link_type', Supplier::class);
            })
            ->where(function (Builder $query) {
                $query->whereNull('qbo_vendors.link_id');
                $query->orWhereColumn('suppliers.updated_at', '>', 'qbo_vendors.updated_at');
                $query->orWhereNull('qbo_vendors.updated_at');

            });
        if ($supplier_ids) {
            $query->where(function (Builder $supplierQuery) use ($supplier_ids) {
                $supplierQuery->orWhereIn('suppliers.id', $supplier_ids);
            });
        }

        return $query->get(['suppliers.*']);
    }

    public function syncVendor(QboIntegrationInstance|IntegrationInstanceInterface $qboIntegrationInstance, Supplier $supplier, QboVendorData|Data $payload): QboVendor
    {
        return QboVendor::updateOrCreate(
            [
                'integration_instance_id' => $qboIntegrationInstance->id,
                'link_id' => $supplier->id,
                'link_type' => Supplier::class,
            ],
            [
                'json_object' => $payload->json_object,
                'integration_instance_id' => $qboIntegrationInstance->id,
                'link_id' => $supplier->id,
                'link_type' => Supplier::class,
            ]
        );
    }
}
