<?php

namespace Modules\Qbo\Jobs;

use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Models\Concerns\HasJobPayload;
use DateTime;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeEncrypted;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\Middleware\WithoutOverlapping;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\App;
use Modules\Qbo\Data\QboResponseData;
use Modules\Qbo\Exceptions\QboRefreshAccessTokenException;
use Modules\Qbo\Exceptions\QboSystemFailureException;
use Modules\Qbo\Exceptions\QboTimeoutException;

abstract class QboJob implements ShouldBeEncrypted, ShouldQueue
{
    use Dispatchable;
    use HasJobPayload;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    public ?array $jobPayload = [];

    public int $uniqueFor = 60 * 60;

    public int $timeout = 60 * 60;

    public int $maxExceptions = 2;

    public int $limit = 1000;

    abstract public function apiCall(): QboResponseData|ClientResponseDataInterface;

    public function __construct()
    {
    }

    public function uniqueId(): string
    {
        return App::environment('testing') ? microtime() : get_class_name_without_namespace($this);
    }

    /**
     * Initialization steps for every job.
     *
     * @throws InvalidArgumentException
     */
    protected function jobStarted(): void
    {
        if (App::environment() !== 'testing') {
            $job = $this->getJobPayload();

            if ($job) {
                $this->jobPayload = $job->payload;
            }
        }
    }

    /**
     * @throws InvalidArgumentException
     */
    protected function jobCompleted(): void
    {
        if (App::environment() !== 'testing') {
            $this->deleteJobPayload();
        }
    }

    protected function nextPage(): void
    {
        $this->release(0);
    }

    /**
     * Set the value of timeout
     *
     * @return self
     */
    public function setTimeout($timeout)
    {
        $this->timeout = $timeout;

        $this->uniqueFor = $this->timeout;

        return $this;
    }

    /**
     * Determine the time at which the job should time out.
     */
    public function retryUntil(): DateTime
    {
        return now()->addSeconds($this->timeout);
    }

    /**
     * @return string[]
     */
    public function tags()
    {
        return [
            'QboJob',
            get_class_name_without_namespace($this),
        ];
    }

    protected function handleTimeoutException(int $delay = 60): void
    {
        $this->release($delay);
    }

    /**
     * Execute the job.
     *
     * @throws Exception
     */
    public function handle(): void
    {
        //Get job payload from database.
        $this->jobStarted();

        //Make API call.
        try {
            $qboResponseDto = $this->apiCall();
        } catch (Exception $exception) {
            throw $exception;

            return;
        } catch (QboTimeoutException $e) {
            $this->handleTimeoutException();

            return;
        } catch (QboSystemFailureException $e) {
            $this->handleTimeoutException();

            return;
        } catch (QboRefreshAccessTokenException $e) {
            $this->handleTimeoutException();

            return;
        }

        //If more records exist
        if ($qboResponseDto->maxResults === $this->limit) {
            $this->saveJobPayload([
                'startPosition' => $qboResponseDto->startPosition + $this->limit,
            ]);

            $this->nextPage();
        } else {
            $this->jobCompleted();
        }
    }

    public function middleware()
    {
        if (! App::environment('testing')) {
            return [
                (new WithoutOverlapping($this->uniqueId()))->expireAfter($this->timeout)->dontRelease(),
            ];
        }

        return [];
    }
}
