<?php

namespace Modules\Qbo\Http\Controllers;

use App\DataTable\DataTable;
use App\Http\Controllers\Controller;
use App\Jobs\CacheProductListingPriceJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Repositories\SalesChannelRepository;
use App\Response;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\DTO\AmazonIntegrationInstanceDto;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Http\Requests\StoreAmazonIntegrationInstanceRequest;
use Modules\Amazon\Http\Resources\AmazonIntegrationInstanceResource;
use Modules\Amazon\Jobs\RefreshAmazonFbaInboundShipmentsJob;
use Modules\Amazon\Jobs\RefreshAmazonFinancialEventGroupsJob;
use Modules\Amazon\Managers\AmazonInboundManager;
use Modules\Amazon\Managers\AmazonIntegrationInstanceManager;
use Modules\Amazon\Repositories\AmazonIntegrationInstanceRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Modules\Amazon\Services\AmazonClient;
use Throwable;

class QboIntegrationInstanceController extends Controller
{
    use DataTable;

    private AmazonIntegrationInstanceRepository $amazonIntegrationInstanceRepository;

    private SalesChannelRepository $salesChannelRepository;

    private AmazonReportRepository $reportRepository;

    public function __construct()
    {
        $this->amazonIntegrationInstanceRepository = app(AmazonIntegrationInstanceRepository::class);
        $this->salesChannelRepository = app(SalesChannelRepository::class);
        $this->reportRepository = app(AmazonReportRepository::class);

        parent::__construct();
    }

    public function show($id): ?Response
    {
        $amazonIntegrationInstance = AmazonIntegrationInstance::query()->findOrFail($id);

        $amazonIntegrationInstance->load([
            'salesChannel',
            'integration',
        ]);

        return $this->response->addData(AmazonIntegrationInstanceResource::make($amazonIntegrationInstance));
    }

    /**
     * @throws Throwable
     */
    public function store(StoreAmazonIntegrationInstanceRequest $request)
    {
        $amazonIntegrationInstance = null;
        DB::transaction(function () use ($request, &$amazonIntegrationInstance) {
            $inputs = $request->validated();

            $integrationSettings = $inputs['integration_settings'];

            $integrationSettings['pricing'] = [
                'masterOfPrice' => [
                    'name' => 'Neither',
                    'id' => 'neither',
                ],
            ];

            $integrationSettings['inventory'] = [
                'masterOfStock' => 'Neither',
            ];

            $dto = AmazonIntegrationInstanceDto::from([
                'integration_id'       => $inputs['integration_id'],
                'name'                 => $inputs['name'],
                'connection_settings'  => [],
                'integration_settings' => $integrationSettings,
                'is_automatic_sync_enabled' => ! is_null(@$inputs['is_automatic_sync_enabled']) ? $inputs['is_automatic_sync_enabled'] : false,
            ]);

            $amazonIntegrationInstance = $this->amazonIntegrationInstanceRepository->save($dto);

            $this->salesChannelRepository->saveFromIntegrationInstance($amazonIntegrationInstance);
        });

        return $amazonIntegrationInstance;
    }

    /**
     * @throws Throwable
     */
    public function update(StoreAmazonIntegrationInstanceRequest $request, int $amazonIntegrationInstanceId): ?Response
    {
        $inputs = $request->validated();

        /** @var AmazonIntegrationInstance $amazonIntegrationInstance */
        $amazonIntegrationInstance = AmazonIntegrationInstance::query()->findOrFail($amazonIntegrationInstanceId);
        $existingIntegrationSettings = $amazonIntegrationInstance->integration_settings;
        $newIntegrationSettings = $inputs['integration_settings'];

        // integrationSettingsChanged is set to the changed portion of the payload.
        $integrationSettingsChanged = arrayRecursiveDiff(
            $newIntegrationSettings,
            $existingIntegrationSettings,
        );

        $amazonIntegrationInstance->update($inputs);

        if (isset($integrationSettingsChanged['start_date'])) {
            $amazonIntegrationInstance->setDate('start_date');
        }

        if (isset($integrationSettingsChanged['fba_inventory_tracking_start_date'])) {
            $amazonIntegrationInstance->setDate('fba_inventory_tracking_start_date');
        }

        if (isset($integrationSettingsChanged['store_id'])) {
            $amazonIntegrationInstance
                ->salesChannel->update(['store_id' => $integrationSettingsChanged['settings']['store']['id']]);
        }

        if (isset($integrationSettingsChanged['is_fba_enabled'])) {
            (new AmazonIntegrationInstanceManager($amazonIntegrationInstance))->processAmazonFbaStateChange();
        }

        /*
         *  ProductListing price and quantity need updating if the settings have changed
         *  TODO: Needs automated test
         */
        if (isset($integrationSettingsChanged['pricing']['masterOfPrice']['name'])) {
            $this->dispatchSync(new CacheProductListingPriceJob($amazonIntegrationInstance));
        }
        if (isset($integrationSettingsChanged['inventory'])) {
            $this->dispatchSync(new GenerateCacheProductListingQuantityJob($amazonIntegrationInstance));
        }

        return $this->response->addData(AmazonIntegrationInstanceResource::make($amazonIntegrationInstance));
    }

    public function destroy(int $id): ?Response
    {
        $this->amazonIntegrationInstanceRepository->deleteById($id);

        return $this->response->setStatusCode(204); // TODO - Confirm that the resulting response actually ends up as a 204. Sam observed this actually resulting in a 200
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function createWarehouseTransfer(
        Request $request,
        AmazonIntegrationInstance $amazonIntegrationInstance,
        int $amazonFbaInboundShipmentId
    ) {
        $request->validate([
            'date'         => 'required|date',
            'warehouse_id' => 'required|int|exists:warehouses,id',
        ]);

        (new AmazonInboundManager($amazonIntegrationInstance))->createWarehouseTransfer($amazonFbaInboundShipmentId,
            $request->all());
    }

    /**
     * Redirect user to Amazon for authentication.
     *
     * @param Request $request
     * @return Response
     * @throws Exception
     */
    public function authenticate(Request $request, AmazonIntegrationInstance $amazonIntegrationInstance): Response
    {
        return $this->response->addData((new AmazonClient($amazonIntegrationInstance))->getRedirectUrl());
    }

    /**
     * Handle amazon callback.  User may need to store credentials when setting up for the first time or when
     * re-authenticating, so needs to be Idempotent.
     *
     * @param  Request  $request
     * @return Redirector|Application|RedirectResponse
     * @throws Throwable
     */
    public function storeCredentials(Request $request): Redirector|Application|RedirectResponse
    {
        customlog('amazon', 'Storing Credentials');
        $request->validate([
            'state'              => 'required|string',
            'spapi_oauth_code'   => 'required|string',
            'selling_partner_id' => 'required|string',
        ]);

        $state = explode('|', $request->input('state'));
        $subdomain = $state[0];
        $amazonIntegrationInstanceId = $state[1];

        $amazonIntegrationInstance
            = $this->amazonIntegrationInstanceRepository->getById($amazonIntegrationInstanceId);

        $amazonClient = new AmazonClient($amazonIntegrationInstance);

        if ('https://'.$request->getHost() === $subdomain) {
            $refreshTokenResponse = $amazonClient->getRefreshTokenFromAuthCode($request->input('spapi_oauth_code'));

            throw_if(is_null($refreshTokenResponse), 'Amazon SP API Auth Code not working.');

            $amazonIntegrationInstanceDto = AmazonIntegrationInstanceDto::from([
                'id'                  => $amazonIntegrationInstance->id,
                'connection_settings' => [
                    'refresh_token'      => $refreshTokenResponse['refresh_token'],
                    'access_token'       => $refreshTokenResponse['access_token'],
                    'selling_partner_id' => $request->input('selling_partner_id'),
                ],
                'is_automatic_sync_enabled' => $amazonIntegrationInstance->is_automatic_sync_enabled,
            ]);

            $this->amazonIntegrationInstanceRepository->save($amazonIntegrationInstanceDto);

            $amazonIntegrationInstance->refresh();

            (new AmazonIntegrationInstanceManager($amazonIntegrationInstance))->requestInitialReports();
            (new AmazonIntegrationInstanceManager($amazonIntegrationInstance))->processAmazonFbaStateChange();
            dispatch(new RefreshAmazonFbaInboundShipmentsJob($amazonIntegrationInstance));
            dispatch(new RefreshAmazonFinancialEventGroupsJob($amazonIntegrationInstance));

            return redirect('/amazon/products?integration_instance_id='.$amazonIntegrationInstanceId);
        }

        return redirect('https://www.sku.io');
    }

    protected function getModel(): string
    {
        return AmazonIntegrationInstance::class;
    }

    protected function getResource(): string
    {
        return AmazonIntegrationInstanceResource::class;
    }
}
