<?php

namespace Modules\Ebay\Tests\Feature\Controllers;

use App\Data\CreateSkuOrderFromSalesChannelData;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Ebay\ApiDataTransferObjects\EbayGetOrdersAdt;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayOrder;
use Modules\Ebay\Entities\EbayOrderItem;
use Modules\Ebay\Jobs\RefreshEbayOrdersJob;
use Modules\Ebay\Managers\EbayOrderManager;
use Modules\Ebay\Tests\EbayMockRequests;
use Modules\Ebay\Tests\EbayTestingData;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class EbayOrderControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use EbayMockRequests;

    private EbayIntegrationInstance $ebayIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->ebayIntegrationInstance = EbayIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create();

        $this->mockGetAccessToken();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Exception
     */
    public function test_ebay_orders_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Refresh orders
        |--------------------------------------------------------------------------
        */

        $this->mockRefreshOrders();

        $this->postJson(route('ebay.orders.refresh', $this->ebayIntegrationInstance->id))->assertOk();

        Queue::assertPushed(RefreshEbayOrdersJob::class);

        (new EbayOrderManager($this->ebayIntegrationInstance))->refreshOrders(new EbayGetOrdersAdt());

        $this->assertDatabaseHas((new EbayOrder())->getTable(), [
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
            'json_object' => json_encode(EbayTestingData::getOrders()['orders'][0]),
        ]);

        $this->assertDatabaseHas((new EbayOrderItem())->getTable(), [
            'json_object' => json_encode(EbayTestingData::getOrders()['orders'][0]['lineItems'][0]),
        ]);

        /*
        |--------------------------------------------------------------------------
        | Get orders
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('ebay.orders.index', $this->ebayIntegrationInstance->id))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                '*' => [
                    'id',
                    'sku_sales_order',
                    'orderId',
                    'creationDateUtc',
                    'lastModifiedDateUtc',
                    'error_log',
                    'orderPaymentStatus',
                    'orderFulfillmentStatus',
                    'totalValue',
                    'totalCurrency',
                    'salesRecordReference',
                    'totalMarketplaceFeeValue',
                    'totalMarketplaceFeeCurrency',
                ],
            ],
        ]);

        $firstOrderId = $response->json('data')[0]['id'];

        /*
        |--------------------------------------------------------------------------
        | Show order
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('ebay.orders.show', [$this->ebayIntegrationInstance->id, $firstOrderId]))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                'id',
                'integration_instance_id',
                'sku_sales_order',
                'orderId',
                'creationDateUtc',
                'lastModifiedDateUtc',
                'error_log',
                'orderPaymentStatus',
                'orderFulfillmentStatus',
                'totalValue',
                'totalCurrency',
                'salesRecordReference',
                'totalMarketplaceFeeValue',
                'totalMarketplaceFeeCurrency',
                'archived_at',
                'created_at',
                'updated_at',
                'json_object',
            ],
        ]);

        /*
        |--------------------------------------------------------------------------
        | Create sku orders
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('ebay.orders.create-sku-orders', $this->ebayIntegrationInstance->id), CreateSkuOrderFromSalesChannelData::from([
            'create_all_orders' => true,
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'sales_order_number' => EbayTestingData::getOrders()['orders'][0]['orderId'],
            'sales_channel_order_type' => EbayOrder::class,
        ]);

        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), 2);

        // 1 order with shipping line, 2 with marketplace fee
        $this->assertDatabaseCount((new FinancialLine())->getTable(), 3);
        $this->assertDatabaseCount((new FinancialLineType())->getTable(), 2);

        /*
        |--------------------------------------------------------------------------
        | Export orders
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('ebay.orders.export', $this->ebayIntegrationInstance->id))->assertOk()->assertDownload();
    }
}
