<?php

namespace Modules\Ebay\Tests\Feature\Controllers;

use App\Data\CreateSkuProductsFromSalesChannelData;
use App\Data\SalesChannelProductImportMappingData;
use App\Data\SalesChannelProductToSkuProductMappingCollectionData;
use App\Data\SalesChannelProductToSkuProductMappingData;
use App\Enums\SalesChannelProductTemplateFieldTypeEnum;
use App\Jobs\ImportSalesChannelProductMappingsJob;
use App\Jobs\MapSalesOrderLinesToSalesChannelProductsJob;
use App\Models\Attribute;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\ProductBlemished;
use App\Models\ProductBrand;
use App\Models\ProductImage;
use App\Models\ProductListing;
use App\Models\ProductPricing;
use App\Models\SalesChannel;
use App\Models\SalesChannelFieldDefinition;
use App\Models\SalesChannelProductTemplate;
use App\Models\SalesChannelProductTemplateField;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Storage;
use Laravel\Sanctum\Sanctum;
use Modules\Ebay\ApiDataTransferObjects\EbayGetLegacyProductsAdt;
use Modules\Ebay\Data\EbayLegacyProductData;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Entities\EbayOrder;
use Modules\Ebay\Entities\EbayOrderItem;
use Modules\Ebay\Enums\EbayListingDurationCodeTypeEnum;
use Modules\Ebay\Enums\EbayListingTypeCodeTypeEnum;
use Modules\Ebay\Jobs\RefreshEbayLegacyProductsJob;
use Modules\Ebay\Managers\EbayLegacyProductManager;
use Modules\Ebay\Services\Transformer\XmlResponseTransformer;
use Modules\Ebay\Tests\EbayMockRequests;
use Modules\Ebay\Tests\EbayTestingData;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class EbayLegacyProductControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use EbayMockRequests;

    private EbayIntegrationInstance $ebayIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->ebayIntegrationInstance = EbayIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create();

        $settings = $this->ebayIntegrationInstance->integration_settings;
        $settings['inventory']['masterOfStock'] = 'sku.io';
        $this->ebayIntegrationInstance->integration_settings = $settings;
        $this->ebayIntegrationInstance->update();

        $this->mockGetAccessToken();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_ebay_legacy_products_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Refresh products
        |--------------------------------------------------------------------------
        */

        $this->mockRefreshLegacyProducts();

        $this->postJson(route('ebay.legacy-products.refresh', $this->ebayIntegrationInstance->id))->assertOk();

        Queue::assertPushed(RefreshEbayLegacyProductsJob::class);

        (new EbayLegacyProductManager($this->ebayIntegrationInstance))->refreshProducts(new EbayGetLegacyProductsAdt());

        $this->assertDatabaseHas((new EbayLegacyProduct())->getTable(), [
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
            'json_object' => json_encode(EbayLegacyProductData::from(XmlResponseTransformer::simplexmlToArray(
                simplexml_load_string(EbayTestingData::getLegacyProducts())
            )['ActiveList']['ItemArray']['Item'][0])->json_object),
        ]);

        /*
        |--------------------------------------------------------------------------
        | Get products
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('ebay.legacy-products.index', $this->ebayIntegrationInstance->id))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                '*' => [
                    'id',
                    'ItemID',
                    'SKU',
                    'ListingDuration',
                    'ListingType',
                    'Title',
                    'QuantityAvailable',
                    'CurrentPriceValue',
                    'CurrentPriceCurrency',
                    'WatchCount',
                    'StartTimeUtc',
                    'EndTimeUtc',
                    'product',
                    'created_at',
                    'updated_at',
                ],
            ],
        ]);

        $ebayProducts = $response->json('data');

        /*
        |--------------------------------------------------------------------------
        | Show product
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('ebay.legacy-products.show', [$this->ebayIntegrationInstance->id, $ebayProducts[0]['id']]))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                'id',
                'integration_instance_id',
                'product_listing_id',
                'ItemID',
                'SKU',
                'ListingDuration',
                'ListingType',
                'Title',
                'QuantityAvailable',
                'CurrentPriceValue',
                'CurrentPriceCurrency',
                'WatchCount',
                'StartTimeUtc',
                'EndTimeUtc',
                'product',
                'created_at',
                'updated_at',
                'json_object',
            ],
        ]);

        /*
        |--------------------------------------------------------------------------
        | Create sku products
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('ebay.legacy-products.create-sku-products', $this->ebayIntegrationInstance->id), CreateSkuProductsFromSalesChannelData::from([
            'create_all_products' => true,
        ])->toArray())->assertOk();

        $product = Product::first();

        $this->assertDatabaseHas((new Product())->getTable(), [
            'id' => $product->id,
            'sku' => $ebayProducts[1]['SKU'],
        ]);

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'document_type' => EbayLegacyProduct::class,
            'document_id' => $ebayProducts[1]['id'],
            'listing_sku' => $ebayProducts[1]['SKU'],
            'product_id' => $product->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Unmap product
        |--------------------------------------------------------------------------
        */

        $this->putJson(route('ebay.legacy-products.map', $this->ebayIntegrationInstance->id), SalesChannelProductToSkuProductMappingCollectionData::from([
            'mapping' => SalesChannelProductToSkuProductMappingData::collection([
                SalesChannelProductToSkuProductMappingData::from([
                    'sales_channel_listing_id' => $ebayProducts[0]['ItemID'],
                    'mapped_sku' => null,
                ]),
                SalesChannelProductToSkuProductMappingData::from([
                    'sales_channel_listing_id' => $ebayProducts[1]['ItemID'],
                    'mapped_sku' => null,
                ]),
            ]),
        ])->toArray())->assertOk();

        $this->assertDatabaseEmpty((new ProductListing())->getTable());

        /*
        |--------------------------------------------------------------------------
        | Map product
        |--------------------------------------------------------------------------
        */

        //Data to test mapping of sales order lines.
        $salesChannelOrderLine = EbayOrderItem::factory()->hasOrder()->create()->refresh();

        $salesChannelOrderLine->json_object = array_merge($salesChannelOrderLine->json_object, [
            $salesChannelOrderLine->getSalesChannelProductUniqueId() => $ebayProducts[1]['ItemID'],
        ]);
        $salesChannelOrderLine->save();

        $salesOrder = SalesOrder::factory()->create([
            'sales_channel_order_type' => EbayOrder::class,
            'sales_channel_order_id' => 1,
        ]);

        $salesOrderLine = SalesOrderLine::factory()
            ->for($salesOrder)
            ->create([
                'product_id' => null,
                'product_listing_id' => null,
                'sales_channel_line_id' => $salesChannelOrderLine->{$salesChannelOrderLine::getTableUniqueId()},
            ]);

        // $ebayProducts[1] should be the first ebay product created (it appears last in the list from the index method because it has the lowest id)

        $this->putJson(route('ebay.legacy-products.map', $this->ebayIntegrationInstance->id), SalesChannelProductToSkuProductMappingCollectionData::from([
            'mapping' => SalesChannelProductToSkuProductMappingData::collection([
                SalesChannelProductToSkuProductMappingData::from([
                    'sales_channel_listing_id' => $ebayProducts[1]['ItemID'],
                    'mapped_sku' => $ebayProducts[1]['SKU'],
                ]),
            ]),
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'document_type' => EbayLegacyProduct::class,
            'document_id' => $ebayProducts[1]['id'],
            'listing_sku' => $ebayProducts[1]['SKU'],
        ]);

        Queue::assertPushed(MapSalesOrderLinesToSalesChannelProductsJob::class);
        (end(Queue::pushedJobs()[MapSalesOrderLinesToSalesChannelProductsJob::class])['job'])->handle();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine->id,
            'product_id' => $product->id,
            'product_listing_id' => ProductListing::whereListingSku($ebayProducts[1]['SKU'])->first()->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Export products
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('ebay.legacy-products.export', $this->ebayIntegrationInstance->id))->assertOk()->assertDownload();

        /*
        |--------------------------------------------------------------------------
        | Download CSV products
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('ebay.legacy-products.export-download', $this->ebayIntegrationInstance->id), [
            'included' => [
                'ItemID',
                'Title',
                'SKU',
                'mapped_sku',
            ],
        ])->assertOk()->assertDownload();

        /*
        |--------------------------------------------------------------------------
        | Upload CSV products
        |--------------------------------------------------------------------------
        */

        $product = Product::factory()->create([
            'sku' => 'NewProduct',
        ]);

        $csvData = [
            [EbayLegacyProduct::getUniqueField(), 'Title', 'SKU', 'mapped_sku'],
            [$ebayProducts[1][EbayLegacyProduct::getUniqueField()], $ebayProducts[1]['Title'], $ebayProducts[1]['SKU'], 'NewProduct'],
            ['NonexistantItemId', 'Title', 'SKU', 'mapped_sku'],
        ];

        $path = Storage::disk('sales-channel-product-mappings')->path('test listings.csv');
        $fh = fopen($path, 'w');

        foreach ($csvData as $row) {
            fputcsv($fh, $row);
        }

        fclose($fh);

        $mappingData = SalesChannelProductImportMappingData::from([
            'original_name' => 'test.csv',
            'stored_name' => 'test listings.csv',
        ]);

        $this->postJson(route('ebay.legacy-products.import-mappings', $this->ebayIntegrationInstance->id), $mappingData->toArray())->assertOk();

        Queue::assertPushed(ImportSalesChannelProductMappingsJob::class);

        $manager = new EbayLegacyProductManager($this->ebayIntegrationInstance);

        (new ImportSalesChannelProductMappingsJob($mappingData, $manager))->handle();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'document_type' => EbayLegacyProduct::class,
            'document_id' => $ebayProducts[1]['id'],
            'sales_channel_listing_id' => $ebayProducts[1][EbayLegacyProduct::getUniqueField()],
            'listing_sku' => $ebayProducts[1][EbayLegacyProduct::getSkuField()],
            'product_id' => $product->id,
        ]);

        Queue::assertPushed(MapSalesOrderLinesToSalesChannelProductsJob::class);

        /*
        |--------------------------------------------------------------------------
        | Sync Inventory
        |--------------------------------------------------------------------------
        */

        $this->mockLegacyReviseInventoryStatus();

        $productListing = ProductListing::first();
        $productListing->quantity = 10;
        $productListing->update();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_qty' => null,
            'sales_channel_qty_last_updated' => null,
        ]);

        $this->postJson(route('ebay.legacy-products.sync-inventory', $this->ebayIntegrationInstance->id))->assertOk();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_qty' => 10,
        ]);

        $this->assertDatabaseMissing((new ProductListing())->getTable(), [
            'sales_channel_qty_last_updated' => null,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_publish_to_sales_channel(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Hydrate Template
        |--------------------------------------------------------------------------
        */

        $productBrand = ProductBrand::factory()->create([
            'name' => 'Brand',
        ]);

        $originalProduct = Product::factory()->create();

        $product = Product::factory()->withProductPricing()->create(
            [
                'brand_id' => $productBrand->id,
                'mpn' => 'MPN',
            ]
        );
        ProductBlemished::create([
            'product_id' => $product->id,
            'derived_from_product_id' => $originalProduct->id,
            'condition' => 'Slightly used',
            'reference' => 'reference',
        ]);


        $productImage1 = ProductImage::factory()->create([
            'product_id' => $product->id,
            'url' => 'https://abc123.co/image1.jpg',
            'sort_order' => 0,
        ]);
        $productImage2 = ProductImage::factory()->create([
            'product_id' => $product->id,
            'url' => 'https://abc123.co/image2.jpg',
            'sort_order' => 1,
        ]);

        $productPricing = ProductPricing::whereProductId($product->id)->firstOrFail();

        $attribute = Attribute::factory()->create([
            'name' => 'ebay_title',
            'type' => 'string',
            'validation' => [
                'limit' => 80,
            ]
        ]);

        $productAttribute = ProductAttribute::factory()->create([
            'product_id' => $product->id,
            'attribute_id' => $attribute->id,
            'value' => 'Ebay Title',
        ]);



        $salesChannelProductTemplate = SalesChannelProductTemplate::factory()->create([
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
        ]);

        // Template setup

        $templateSetup = [
            'ApplicationData' => '{{product.sku}}',
            'SKU' => '{{product.sku}}',
            'Title' => '{{product_attributes.ebay_title}}',
            'BuyItNowPrice' => '{{product_pricing.Retail}}',
            'StartPrice' => '{{product_pricing.Retail}}',
            'PrimaryCategoryID' => '35555',
            'ConditionDescription' => '{{product_blemished.condition}}',
            'ConditionID' => '1500',
            'Country' => 'US',
            'Currency' => 'USD',
            'Description' => 'This is a description',
            'Brand' => '{{product_brand}}',
            'ManufacturerPartNumber' => '{{product.mpn}}',
            'ListingDuration' => 'Days_7',
            'ListingType' => 'Chinese',
            'Location' => 'North Las Vegas, Nevada',
            'UPC' => 'Does Not Apply',
            'Quantity' => '1',
            'PaymentProfileName' => 'Default',
            'ReturnProfileName' => 'Default',
            'ShippingProfileName' => 'Default',
            'PictureURL' => '{{product_images}}',
        ];

        foreach ($templateSetup as $fieldName => $defaultValue) {
            $type = str_contains($defaultValue, '{{') ? SalesChannelProductTemplateFieldTypeEnum::DYNAMIC : SalesChannelProductTemplateFieldTypeEnum::STATIC;
            SalesChannelProductTemplateField::factory()->create([
                'sales_channel_product_template_id' => $salesChannelProductTemplate->id,
                'sales_channel_field_definition_id' => SalesChannelFieldDefinition::whereFieldName($fieldName)->firstOrFail()->id,
                'type' => $type,
                'default_value' => $defaultValue,
            ]);
        }

        $response = $this->postJson(route('ebay.legacy-products.hydrate-product-template', $this->ebayIntegrationInstance->id), [
            'product_id' => $product->id,
            'sales_channel_product_template_id' => $salesChannelProductTemplate->id,
        ])->assertOk();

        $response->assertJson([
            'data' => [
                'fields' => [
                    [
                        'name' => 'ApplicationData',
                        'value' => $product->sku,
                        'data_type' => 'string',
                        'validation_rules' => ['required|string|max:32'],
                        'sort_order' => 0,
                    ],
                    [
                        'name' => 'SKU',
                        'value' => $product->sku,
                        'data_type' => 'string',
                        'validation_rules' => ['required'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'Title',
                        'value' => $productAttribute->value,
                        'data_type' => 'string',
                        'validation_rules' => ['required|max:80'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'BuyItNowPrice',
                        'value' => $productPricing->price,
                        'data_type' => 'string',
                        'validation_rules' => ['sometimes|numeric|decimal:2'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'StartPrice',
                        'value' => $productPricing->price,
                        'data_type' => 'string',
                        'validation_rules' => ['required|numeric|decimal:2'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'PrimaryCategoryID',
                        'value' => '35555',
                        'data_type' => 'string',
                        'validation_rules' => ['required|integer|digits_between:1,10'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'ConditionDescription',
                        'value' => 'Slightly used',
                        'data_type' => 'string',
                        'validation_rules' => ['sometimes|string|max:1000'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'ConditionID',
                        'value' => '1500',
                        'data_type' => 'string',
                        'validation_rules' => ['required|integer'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'Country',
                        'value' => 'US',
                        'data_type' => 'string',
                        'validation_rules' => ['required|string|max:2'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'Currency',
                        'value' => 'USD',
                        'data_type' => 'string',
                        'validation_rules' => ['required|string|max:3'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'Description',
                        'value' => 'This is a description',
                        'data_type' => 'html',
                        'validation_rules' => ['required|string|max:500000'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'Brand',
                        'value' => $product->brand->name,
                        'data_type' => 'string',
                        'validation_rules' => ['required|string|max:65'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'ManufacturerPartNumber',
                        'value' => $product->mpn,
                        'data_type' => 'string',
                        'validation_rules' => ['required|string|max:65'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'ListingDuration',
                        'value' => 'Days_7',
                        'data_type' => 'string',
                        'validation_rules' => ['sometimes|string|in:' . implode(',', array_column(EbayListingDurationCodeTypeEnum::cases(), 'value'))],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'ListingType',
                        'value' => 'Chinese',
                        'data_type' => 'string',
                        'validation_rules' => ['required|string|in:' . implode(',', array_column(EbayListingTypeCodeTypeEnum::cases(), 'value'))],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'Location',
                        'value' => 'North Las Vegas, Nevada',
                        'data_type' => 'string',
                        'validation_rules' => ['required|string|max:45'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'UPC',
                        'value' => 'Does Not Apply',
                        'data_type' => 'string',
                        'validation_rules' => ['required|string'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'Quantity',
                        'value' => '1',
                        'data_type' => 'string',
                        'validation_rules' => ['required|integer|min:1'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'PaymentProfileName',
                        'value' => 'Default',
                        'data_type' => 'string',
                        'validation_rules' => ['required|string'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'ReturnProfileName',
                        'value' => 'Default',
                        'data_type' => 'string',
                        'validation_rules' => ['required|string'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'ShippingProfileName',
                        'value' => 'Default',
                        'data_type' => 'string',
                        'validation_rules' => ['required|string'],
                        'sort_order' => 1,
                    ],
                    [
                        'name' => 'PictureURL',
                        'value' => [
                            $productImage1->url,
                            $productImage2->url,
                        ],
                        'data_type' => 'json',
                        'validation_rules' => ['sometimes|array|max:24'],
                        'sort_order' => 1,
                    ],
                ],
                'payload' => view('ebay::trading.AddItem', [
                    'ApplicationData' => $product->sku,
                    'BuyItNowPrice' => $productPricing->price,
                    'StartPrice' => $productPricing->price,
                    'SKU' => $product->sku,
                    'Title' => $productAttribute->value,
                    'PrimaryCategoryID' => '35555',
                    'ConditionDescription' => 'Slightly used',
                    'ConditionID' => '1500',
                    'Country' => 'US',
                    'Currency' => 'USD',
                    'Description' => 'This is a description',
                    'Brand' => $product->brand->name,
                    'ManufacturerPartNumber' => $product->mpn,
                    'ListingDuration' => 'Days_7',
                    'ListingType' => 'Chinese',
                    'Location' => 'North Las Vegas, Nevada',
                    'UPC' => 'Does Not Apply',
                    'Quantity' => '1',
                    'PaymentProfileName' => 'Default',
                    'ReturnProfileName' => 'Default',
                    'ShippingProfileName' => 'Default',
                    'PictureURL' => [
                        $productImage1->url,
                        $productImage2->url,
                    ],
                ])->render(),
            ],
            'status' => 'success',
        ]);

        /*
        |--------------------------------------------------------------------------
        | Build Payload
        |--------------------------------------------------------------------------
        */

        $response = $this->postJson(route('ebay.legacy-products.build-payload', $this->ebayIntegrationInstance->id), [
            'fields' => [
                'ApplicationData' => $product->sku,
                'BuyItNowPrice' => $productPricing->price,
                'StartPrice' => $productPricing->price,
                'SKU' => $product->sku,
                'Title' => $productAttribute->value,
                'PrimaryCategoryID' => '35555',
                'ConditionDescription' => 'Slightly used',
                'ConditionID' => '1500',
                'Country' => 'US',
                'Currency' => 'USD',
                'Description' => 'This is a description',
                'Brand' => $product->brand->name,
                'ManufacturerPartNumber' => $product->mpn,
                'ListingDuration' => 'Days_7',
                'ListingType' => 'Chinese',
                'Location' => 'North Las Vegas, Nevada',
                'UPC' => 'Does Not Apply',
                'Quantity' => '1',
                'PaymentProfileName' => 'Default',
                'ReturnProfileName' => 'Default',
                'ShippingProfileName' => 'Default',
                'PictureURL' => [
                    $productImage1->url,
                    $productImage2->url,
                ]
            ],
        ])->assertOk();

        $response->assertExactJson([
            'data' => view('ebay::trading.AddItem', [
                'ApplicationData' => $product->sku,
                'BuyItNowPrice' => $productPricing->price,
                'StartPrice' => $productPricing->price,
                'SKU' => $product->sku,
                'Title' => $productAttribute->value,
                'PrimaryCategoryID' => '35555',
                'ConditionDescription' => 'Slightly used',
                'ConditionID' => '1500',
                'Country' => 'US',
                'Currency' => 'USD',
                'Description' => 'This is a description',
                'Brand' => $product->brand->name,
                'ManufacturerPartNumber' => $product->mpn,
                'ListingDuration' => 'Days_7',
                'ListingType' => 'Chinese',
                'Location' => 'North Las Vegas, Nevada',
                'UPC' => 'Does Not Apply',
                'Quantity' => '1',
                'PaymentProfileName' => 'Default',
                'ReturnProfileName' => 'Default',
                'ShippingProfileName' => 'Default',
                'PictureURL' => [
                    $productImage1->url,
                    $productImage2->url,
                ],
                ])->render(),
            'status' => 'success',
        ]);

        $payload = $response->json('data');

        /*
        |--------------------------------------------------------------------------
        | Publish to Sales Channel
        |--------------------------------------------------------------------------
        */

        $this->mockAddItem();
        $this->mockUploadSiteHostedPictures();
        $this->mockGetItem();

        $this->postJson(route('ebay.legacy-products.publish', $this->ebayIntegrationInstance->id), [
            'product_id' => $product->id,
            'payload' => $payload,
        ])->assertOk();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'document_type' => EbayLegacyProduct::class,
            'product_id' => $product->id,
        ]);

        $productListing = ProductListing::whereProductId($product->id)->firstOrFail();

        $this->assertEquals("AuctionLengthFee", $productListing->metadata['createResponse']['Fees']['Fee'][0]['Name']);
    }

    public function test_it_can_filter_products_for_specific_integration_instance(): void
    {
        $secondEbayIntegrationInstance = EbayIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create();

        EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
        ]);

        EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $secondEbayIntegrationInstance->id,
        ]);

        $response = $this->getJson(route('ebay.legacy-products.index', $this->ebayIntegrationInstance->id))->assertOk();

        $this->assertCount(1, $response->json('data'));
    }

    public function test_it_can_get_ebay_category_suggestion(): void
    {
        $this->mockGetSuggestedCategories();

        $response = $this->postJson(route('ebay.legacy-products.get-suggested-categories', $this->ebayIntegrationInstance->id), [
            'query' => 'iCarsoft Mercedes Benz Diagnostic Tool',
        ])->assertOk();

        $response->assertJsonStructure([
            'data' => [
                '*' => [
                    'Category' => [
                        'CategoryID',
                        'CategoryName',
                        'CategoryParentID',
                        'CategoryParentName',
                    ],
                    'PercentItemFound',
                ]
            ],
        ]);
    }
}
