<?php

namespace Modules\Ebay\Tests\Feature\Controllers;

use App\Abstractions\Integrations\Data\SalesChannelIntegrationSettingsData;
use App\Data\SalesChannelInventorySettingsData;
use App\Data\SalesChannelMasterOfPriceSettingsData;
use App\Data\SalesChannelPricingSettingsData;
use App\Jobs\CacheProductListingPriceJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\Store;
use App\Models\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Ebay\Data\EbayAuthorizationResponseData;
use Modules\Ebay\Data\EbayConnectionSettingsData;
use Modules\Ebay\Data\StoreEbayIntegrationData;
use Modules\Ebay\Data\UpdateEbayIntegrationData;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Entities\EbayOrder;
use Modules\Ebay\Entities\EbayOrderItem;
use Modules\Ebay\Jobs\RefreshEbayLegacyProductsJob;
use Modules\Ebay\Jobs\RefreshEbayOrdersJob;
use Modules\Ebay\Services\EbayClient;
use Modules\Ebay\Tests\EbayMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class EbayIntegrationInstanceControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use EbayMockRequests;
    use WithFaker;

    public function setUp(): void
    {
        parent::setUp();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Exception
     */
    public function test_ebay_integration_instance_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Create integration instance
        |--------------------------------------------------------------------------
        */

        $integration = Integration::where('name', Integration::NAME_EBAY)->first();

        $response = $this->postJson(route('ebay.store'), StoreEbayIntegrationData::from([
            'name' => 'Ebay',
            'integration_id' => $integration->id,
            'connection_settings' => EbayConnectionSettingsData::from([]),
            'integration_settings' => SalesChannelIntegrationSettingsData::from([
                'start_date' => Carbon::now()->subYear()->toDateTimeString(),
                'store_id' => Store::first()->id,
                'auto_link_products' => false,
                'auto_create_products' => false,
                'sales_nominal_code_id' => null,
                'cogs_nominal_code_id' => null,
                'pricing' => SalesChannelPricingSettingsData::from([
                    'masterOfPrice' => SalesChannelMasterOfPriceSettingsData::from([])
                ]),
                'inventory' => SalesChannelInventorySettingsData::from([]),
                'emailCustomers' => false,
                'proforma_marketplace_cost_percentage' => null,
                'proforma_payment_cost_percentage' => null,
                'sync_sales_order_invoices_to_accounting' => true
            ]),
        ])->toArray())->assertOk();

        $integrationInstance = EbayIntegrationInstance::find($response->json()['data']['id']);

        $this->assertEquals((new EbayClient($integrationInstance))
            ->getRedirectUrl(), $response->json('redirect_url'));

        $this->assertDatabaseHas((new IntegrationInstance())->getTable(), [
            'name' => 'Ebay',
        ]);

        $this->assertDatabaseHas((new SalesChannel())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'store_id' => Store::first()->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Update integration instance
        |--------------------------------------------------------------------------
        */

        $newStore = Store::factory()->create();

        $newIntegrationSettings = [
            'start_date' => Carbon::now()->subYear()->toDateTimeString(),
            'store_id' => $newStore->id,
            'auto_link_products' => true,
            'auto_create_products' => true,
            'sales_nominal_code_id' => null,
            'cogs_nominal_code_id' => null,
            'pricing' => SalesChannelPricingSettingsData::from([
                'masterOfPrice' => SalesChannelMasterOfPriceSettingsData::from([
                    'name' => 'sku.io',
                ]),
            ]),
            'inventory' => SalesChannelInventorySettingsData::from([
                'masterOfStock' => 'sku.io',
            ]),
            'emailCustomers' => false,
            'proforma_marketplace_cost_percentage' => null,
            'proforma_payment_cost_percentage' => null,
            'sync_sales_order_invoices_to_accounting' => true
        ];

        $this->putJson(route('ebay.update', $integrationInstance->id), UpdateEbayIntegrationData::from([
            'integration_settings' => SalesChannelIntegrationSettingsData::from($newIntegrationSettings),
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new IntegrationInstance())->getTable(), [
            'name' => 'Ebay',
            'integration_settings' => json_encode($newIntegrationSettings),
        ]);

        $this->assertDatabaseHas((new SalesChannel())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'store_id' => $newStore->id,
        ]);

        Queue::assertPushed(CacheProductListingPriceJob::class);
        Queue::assertPushed(GenerateCacheProductListingQuantityJob::class);

        /*
        |--------------------------------------------------------------------------
        | Get redirect Url
        |--------------------------------------------------------------------------
        */

        $this->get(route('ebay.get-redirect-url', $integrationInstance->id))
            ->assertOk()
            ->assertJson([
                'data' => (new EbayClient($integrationInstance))->getRedirectUrl(),
            ]);

        /*
        |--------------------------------------------------------------------------
        | Handle authorization response
        |--------------------------------------------------------------------------
        */

        $this->mockRefreshTokenFromAuthCode();

        $this->postJson(route('ebay.callback'), EbayAuthorizationResponseData::from([
            'state' => 'subdomain_' . $integrationInstance->id,
            'code' => 'code',
        ])->toArray())->assertRedirect();

        $this->assertDatabaseHas((new IntegrationInstance())->getTable(), [
            'id' => $integrationInstance->id,
            'connection_settings' => EbayConnectionSettingsData::from([
                'refresh_token' => 'refresh token',
                'access_token' => 'access token',
            ])->toJson(),
        ]);

        // Dispatches jobs to get initial products and orders
        Queue::assertPushed(RefreshEbayLegacyProductsJob::class);
        Queue::assertPushed(RefreshEbayOrdersJob::class);


        /*
        |--------------------------------------------------------------------------
        | Show integration instance
        |--------------------------------------------------------------------------
        */

        $this->get(route('ebay.show', $integrationInstance->id))
            ->assertOk()
            ->assertJsonStructure([
                'data' => [
                    'id',
                    'name',
                    'connection_settings',
                    'integration_settings',
                    'connection_status',
                    'integration_name',
                    'store',
                    'created_at',
                    'updated_at',
                ],
            ]);

        /*
        |--------------------------------------------------------------------------
        | Delete integration instance
        |--------------------------------------------------------------------------
        */

        $ebayProduct = EbayLegacyProduct::factory()->create();
        ProductListing::factory()->create([
            'document_id' => $ebayProduct->id,
            'document_type' => EbayLegacyProduct::class,
        ]);

        $ebayOrder = EbayOrder::factory(2)->has(EbayOrderItem::factory(), 'orderItems')->create();
        SalesOrder::factory()->create([
            'sales_channel_order_id' => $ebayOrder->first()->id,
            'sales_channel_order_type' => EbayOrder::class,
        ]);

        $this->assertEquals(1, EbayIntegrationInstance::count());
        $this->assertEquals(1, EbayLegacyProduct::count());
        $this->assertEquals(1, ProductListing::count());
        $this->assertEquals(2, EbayOrder::count());
        $this->assertEquals(1, SalesOrder::count());

        $this->delete(route('ebay.destroy', $integrationInstance->id));

        $this->assertEquals(0, EbayIntegrationInstance::query()->count());
        $this->assertEquals(0, EbayLegacyProduct::query()->count());
        $this->assertEquals(0, ProductListing::query()->count());
        $this->assertEquals(0, EbayLegacyProduct::query()->count());
        $this->assertEquals(0, SalesOrder::query()->count());
    }
}
