<?php

namespace Modules\Ebay\Tests\Feature\Controllers;

use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\User;
use App\Models\Warehouse;
use Laravel\Sanctum\Sanctum;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Entities\EbayProductSetting;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class EbayGlobalReportControllerTest extends TestCase
{
    use FastRefreshDatabase;

    private EbayIntegrationInstance $ebayIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();

        $this->ebayIntegrationInstance = EbayIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create([
                'name' => 'ABC Company'
            ]);

        Sanctum::actingAs(User::first());
    }

    /**
     * @throws \Throwable
     */
    public function test_it_can_get_uncovered_blemished_products(): void
    {
        $warehouse = Warehouse::first();

        $product = Product::factory()->create([
            'sku' => '123',
            'name' => 'Test Product',
            'type' => Product::TYPE_BLEMISHED,
        ]);

        $product->setInitialInventory($warehouse->id, 100, 5.00);

        EbayProductSetting::factory()->create([
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
            'product_id' => $product->id,
            'auction_until_sold_enabled' => true,
            'auction_until_sold_price_reduction_percentage' => 10,
            'auction_until_sold_minimum_price' => 10.00,
        ]);

        $ebayProduct = EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
        ]);

        $endTime = now()->subDay()->toDateTimeString();

        $jsonObject = $ebayProduct->json_object;
        $jsonObject['ItemID'] = '123';
        $jsonObject['ListingDuration'] = 'Days_7';
        $jsonObject['ListingType'] = 'Chinese';
        $jsonObject['EndTimeUtc'] = $endTime;
        $jsonObject['SellingStatus']['ListingStatus'] = 'Completed';
        $jsonObject['ListingDetails']['EndingReason'] = 'NotSold';
        $ebayProduct->json_object = $jsonObject;
        $ebayProduct->save();
        $ebayProduct->refresh();

        ProductListing::factory()->create([
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'product_id' => $product->id,
            'document_type' => EbayLegacyProduct::class,
            'document_id' => $ebayProduct->id,
        ]);

        $response = $this->getJson(route('ebay-global-reports.uncovered-blemished-products'))->assertOk();

        $response->assertJson([
            'data' => [
                [
                    'id' => $product->id,
                    'sku' => '123',
                    'name' => 'Test Product',
                    'ended_listings' => [
                        [
                            'instance_name' => 'ABC Company',
                            'ItemID' => '123',
                            'EndTimeUtc' => $endTime,
                            'EndingReason' => 'NotSold',
                        ],
                    ],
                    'auto_relist_instances' => [
                        'ABC Company'
                    ],
                ],
            ],
        ]);
    }

    public function test_it_can_get_listed_blemished_products_out_of_stock(): void
    {
        $product = Product::factory()->create([
            'sku' => '123',
            'name' => 'Test Product',
            'type' => Product::TYPE_BLEMISHED,
        ]);

        $ebayProduct = EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
        ]);

        $startTime = now()->subDay()->toDateTimeString();
        $endTime = now()->addDay()->toDateTimeString();

        $jsonObject = $ebayProduct->json_object;
        $jsonObject['ItemID'] = '123';
        $jsonObject['ListingDuration'] = 'Days_7';
        $jsonObject['ListingType'] = 'Chinese';
        $jsonObject['StartTimeUtc'] = $startTime;
        $jsonObject['EndTimeUtc'] = $endTime;
        $jsonObject['SellingStatus']['ListingStatus'] = 'Active';
        $jsonObject['SellingStatus']['BidCount'] = 2;
        $ebayProduct->json_object = $jsonObject;
        $ebayProduct->save();

        ProductListing::factory()->create([
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'product_id' => $product->id,
            'document_type' => EbayLegacyProduct::class,
            'document_id' => $ebayProduct->id,
        ]);

        $response = $this->getJson(route('ebay-global-reports.listed-blemished-products-out-of-stock'))->assertOk();

        $response->assertJson([
            'data' => [
                [
                    'id' => $product->id,
                    'sku' => '123',
                    'name' => 'Test Product',
                    'active_listings' => [
                        [
                            'instance_name' => 'ABC Company',
                            'ItemID' => '123',
                            'StartTimeUtc' => $startTime,
                            'EndTimeUtc' => $endTime,
                            'BidCount' => 2,
                        ],
                    ],
                ],
            ],
        ]);
    }
}
