<?php

declare(strict_types=1);

namespace Modules\Ebay\Services;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Enums\HttpMethodEnum;
use Exception;
use Modules\Ebay\ApiDataTransferObjects\EbayFulfillOrderAdt;
use Modules\Ebay\ApiDataTransferObjects\EbayGetOrdersAdt;
use Modules\Ebay\ApiDataTransferObjects\EbayLegacyGetOrdersAdt;
use Modules\Ebay\Data\EbayOrderData;
use Modules\Ebay\Data\EbayResponseData;

class EbayFulfillmentClient extends EbayAuthenticationClient
{
    public static function withAuthClient(EbayAuthenticationClient $authClient)
    {
        return new static(
            $authClient
        );
    }

    /**
     * @throws Exception
     */
    public function getOrder(string $orderId): EbayResponseData
    {
        $response = $this->request(HttpMethodEnum::GET, '/sell/fulfillment/v1/order/'.$orderId, []);

        return EbayResponseData::from([
            'collection' => EbayOrderData::collection([$response->json()]),
        ]);
    }

    /**
     * @throws Exception
     *
     * @see https://developer.ebay.com/api-docs/sell/fulfillment/resources/order/methods/getOrders
     */
    public function getOrders(EbayGetOrdersAdt|ApiDataTransformerInterface $parameters): EbayResponseData
    {
        $requestParams = $parameters->transform();
        $response = $this->request(HttpMethodEnum::GET, '/sell/fulfillment/v1/order', $requestParams);

        return EbayResponseData::from([
            'collection' => EbayOrderData::collection($response['orders']),
            'nextEndpoint' => @$response['next'],
        ]);
    }

    /**
     * @throws Exception
     */
    public function fulfillOrder(EbayFulfillOrderAdt|ApiDataTransformerInterface $parameters): bool
    {
        $requestBody = $parameters->transform();
        $response = $this->pushRequest(HttpMethodEnum::POST, '/sell/fulfillment/v1/order/'.$parameters->ebayOrderId.'/shipping_fulfillment', $requestBody);

        if ($response->status() !== 201) {
            throw new Exception('Failed to fulfill ebay order: '.json_encode($response->json()['errors']) ?? 'Unknown error');
        }

        return true;
    }
}
