<?php

namespace Modules\Ebay\Managers;

use Modules\Ebay\Repositories\EbayGlobalReportRepository;

class EbayGlobalReportManager
{
    public function __construct(
        private readonly EbayGlobalReportRepository $reports
    )
    {}

    public function getUncoveredBlemishedProducts(): array
    {
        $data = $this->reports->getUncoveredBlemishedProducts();

        $reportData = $data->map(function ($product) {
            return [
                'id' => $product->id,
                'sku' => $product->sku,
                'name' => $product->name,
                'ended_listings' => $product->productListings->map(function ($listing) {
                    $legacyProduct = $listing->ebayLegacyProduct;
                    return [
                        'instance_name' => $listing->salesChannel->integrationInstance->name,
                        'ItemID' => $legacyProduct->ItemID,
                        'EndTimeUtc' => $legacyProduct->EndTimeUtc,
                        'EndingReason' => $legacyProduct->EndingReason,
                        'BidCount' => $legacyProduct->BidCount,
                        'Price' => $legacyProduct->CurrentPriceValue . ' (' . $legacyProduct->CurrentPriceCurrency . ')',
                    ];
                }),
                'auto_relist_instances' => $product->ebayProductSettings
                    ->filter(fn ($setting) => (bool) $setting->auction_until_sold_enabled)
                    ->map(function ($setting) {
                        return $setting->integrationInstance->name;
                    })->values()
                ->toArray(),
                'notes' => $product->notes->sortByDesc('created_at')->pluck('note')->toArray()
            ];
        });

        return $reportData->toArray();
    }

    public function getListedBlemishedProductsOutOfStock(): array
    {
        $data = $this->reports->getListedBlemishedProductsOutOfStock();

        $reportData = $data->map(function ($product) {
            return [
                'id' => $product->id,
                'sku' => $product->sku,
                'name' => $product->name,
                'active_listings' => $product->productListings
                    ->reject(fn ($listing) => $listing->ebayLegacyProduct->ListingStatus == 'Completed')
                    ->map(function ($listing) {
                    $legacyProduct = $listing->ebayLegacyProduct;
                    return [
                        'instance_name' => $listing->salesChannel->integrationInstance->name,
                        'ItemID' => $legacyProduct->ItemID,
                        'StartTimeUtc' => $legacyProduct->StartTimeUtc,
                        'EndTimeUtc' => $legacyProduct->EndTimeUtc,
                        'BidCount' => $legacyProduct->BidCount,
                    ];
                })->values()->toArray(),
            ];
        });

        return $reportData->toArray();
    }
}