<?php

declare(strict_types=1);

namespace Modules\Ebay\Managers;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use App\Collections\SalesOrderCollection;
use App\Collections\SalesOrderLineCollection;
use App\Data\AddressData;
use App\Data\CustomerData;
use App\Data\SalesOrderLineData;
use App\DTO\SalesOrderDto;
use App\Helpers;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Repositories\WarehouseRepository;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayOrder;
use Modules\Ebay\Entities\EbayOrderItem;
use Modules\Ebay\Repositories\EbayOrderRepository;
use Modules\Ebay\Services\EbayClient;
use Modules\Ebay\Services\ShippingMethodMapper;

class EbayFulfillmentManager extends AbstractSalesChannelManager
{
    private EbayOrderRepository $ebayOrderRepository;


    private mixed $mappings;

    public function __construct(
        EbayIntegrationInstance $ebayIntegrationInstance,
        protected WarehouseRepository $warehouseRepository
    ) {
        parent::__construct($ebayIntegrationInstance, new EbayClient($ebayIntegrationInstance));
        $this->ebayOrderRepository = app(EbayOrderRepository::class);
    }

    protected function processSalesOrderLines(EbayOrder $record)
    {
        $salesOrderLineCollection = new SalesOrderLineCollection();
        // Should read from DB here, since we already processed the inserts.
        $record->orderItems()->each(function (EbayOrderItem $ebayOrderItem) use (
            $record,
            &$salesOrderLineCollection
        ) {
            $product = $ebayOrderItem->ebayProduct?->product;

            $salesOrderLineCollection->push(
                SalesOrderLineData::from([
                    'sales_order_number' => $record->id,
                    'sales_channel_line_id' => $ebayOrderItem->id,
                    'amount' => $ebayOrderItem->quantity * $ebayOrderItem->price,
                    /*
                     * We do not try to calculate a tax rate, instead we rely on the tax amount (aka tax allocation
                     * in sku).
                     */
                    'tax_allocation' => 0,
                    'description' => $ebayOrderItem->title,
                    'product_id' => $product?->id ?? null,
                    'is_product' => true,
                    'quantity' => $ebayOrderItem->quantity,
                    'product_listing_sku' => $ebayOrderItem->sku,
                    'warehouse_id' => $product ? ($this->warehouseRepository->getPriorityWarehouseIdForProduct(
                        $product,
                        $ebayOrderItem->quantity
                    )) : null,
                    'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_SALES_ORDERS),
                ])
            );
        });

        return $salesOrderLineCollection;
    }

    /*
     * todo: Shipping/delivery cost will probably need to be looked at.
     */
    protected function processShippingLines(EbayOrder $record): SalesOrderLineCollection
    {
        $salesOrderLines = new SalesOrderLineCollection();
        if ((float) $record->json_object['pricingSummary']['deliveryCost']['value'] > 0.0) {
            $salesOrderLines->push(
                SalesOrderLineData::from([
                    'sales_order_number' => $record->id,
                    'sales_channel_line_id' => null,
                    'quantity' => 1,
                    'amount' => (float) $record->json_object['pricingSummary']['deliveryCost']['value'],
                    'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_SHIPPING_SALES_ORDERS),
                    'tax_allocation' => 0, // todo: not right
                    'description' => 'Shipping',
                    'product_id' => null,
                    'is_product' => false,
                ])
            );
        }

        return $salesOrderLines;
    }

    protected function buildSalesOrderDto(
        EbayOrder $record,
        Collection|SalesOrderLineCollection $salesOrderLines,
        Collection $provinces,
        Collection $countries
    ): SalesOrderDto {
        $json_object = $record->json_object;
        $ebayOrderStatus = $this->mapOrderStatus($record->json_object['orderFulfillmentStatus']);
        $customerData = $json_object['fulfillmentStartInstructions'][0]['shippingStep']['shipTo'];
        $billingAddress = $this->buildAddressDto($customerData, $provinces, $countries);
        $shippingAddress = $this->buildAddressDto($customerData, $provinces, $countries);
        $cancellationDate = $this->getCancellationDate($record);
        $eBayRequestedShippingMethod =
            $json_object['fulfillmentStartInstructions'][0]['shippingStep']['shippingServiceCode'];
        $skuRequestedShippingMethod = ShippingMethodMapper::translateEbayShippingMethodToSkuShippingMethod(
            $eBayRequestedShippingMethod
        );
        $currency = $record->json_object['pricingSummary']['total']['currency'];

        $salesOrderDto = SalesOrderDto::from([
            'sales_order_number' => $record->id,
            'currency_code' => $currency,
            'order_date' => Carbon::parse($record->creation_date ?? null),
            'shipping_method_id' => null,
            'requested_shipping_method' => $skuRequestedShippingMethod,
            'order_status' => $ebayOrderStatus,
            'canceled_at' => $cancellationDate,
            'sales_order_lines' => $salesOrderLines,
            'sales_channel_order_type' => EbayOrder::class,
            'sales_channel_order_id' => $record->id,
            // TODO: If billing is different from shipping, needs a separate address
            'customer' => CustomerData::from([
                'name' => $customerData['fullName'],
                'email' => $customerData['email'],
                'zip' => $customerData['contactAddress']['postalCode'],
                'address1' => $customerData['contactAddress']['addressLine1'],
                'address2' => $customerData['contactAddress']['addressLine2'],
                'city' => $customerData['contactAddress']['city'],
                'province' => $customerData['contactAddress']['stateOrProvince'],
                'country' => $countries[$customerData['contactAddress']['countryCode']]->name ??
                    $customerData['contactAddress']['countryCode'],
                'company' => null,
                'fax' => null,
                'default_shipping_address' => $shippingAddress,
                'default_billing_address' => $billingAddress,
            ]),
        ]);

        return $salesOrderDto;
    }

    protected function mapOrderStatus(string $ebayOrderStatus): string
    {
        $status = match ($ebayOrderStatus) {
            'NOT_STARTED' => SalesOrder::STATUS_RESERVED,
            'IN_PROGRESS' => SalesOrder::STATUS_OPEN,
            /*
             * TODO: FULFILLED needs handling. For FULFILLED, with consideration of SKU-4065, the user
             *  would need to externally fulfill unless there is one warehouse, then maybe we can attempt to
             *  externally fulfill.
             */
            'FULFILLED' => SalesOrder::STATUS_CLOSED,
            default => SalesOrder::STATUS_DRAFT,
        };

        return $status;
    }

    private function decrementStockBulk(SalesOrderCollection $salesOrdersCollection): void
    {
        // $temp_table_name = $saveSalesOrderResult->temp_table_name;
        // Reserve inventory for each sales order line
        SalesOrderLine::query()
            ->whereHas('salesOrder', function ($query) use ($salesOrdersCollection) {
                $query
                    ->whereIn('sales_order_number', $salesOrdersCollection->pluck('sales_order_number'))
                    ->where('sales_channel_order_type', EbayOrder::class);
            })
            ->get()
            ->each(function ($salesOrderLine) {
                if ($salesOrderLine->warehouse_id && ! $salesOrderLine->salesOrder->canceled_at) {
                    $inventoryManager = new InventoryManager(
                        $salesOrderLine->warehouse_id, $salesOrderLine->product
                    );
                    $inventoryManager->takeFromStock(
                        $salesOrderLine->quantity,
                        $salesOrderLine,
                        false,
                        $salesOrderLine
                    );
                }
            });
    }

    protected function buildAddressDto(array $ebayAddress, $provinces, $countries): AddressData
    {
        $address = AddressData::from([
            'name' => $ebayAddress['fullName'],
            'email' => $ebayAddress['email'],
            'zip' => $ebayAddress['contactAddress']['postalCode'],
            'label' => 'Default Address',
            'address1' => $ebayAddress['contactAddress']['addressLine1'],
            'address2' => $ebayAddress['contactAddress']['addressLine2'],
            'city' => $ebayAddress['contactAddress']['city'],
            'province' => $provinces[$ebayAddress['contactAddress']['stateOrProvince']]->name ?? $ebayAddress['contactAddress']['stateOrProvince'],
            'province_code' => $ebayAddress['contactAddress']['stateOrProvince'],
            'country' => $countries[$ebayAddress['contactAddress']['countryCode']]->name ?? $ebayAddress['contactAddress']['countryCode'],
            'country_code' => $ebayAddress['contactAddress']['countryCode'],
        ]);

        return $address;
    }

    /**
     * Finds the first approved cancellation date in the list and returns it.
     */
    protected function getCancellationDate(EbayOrder $record): ?Carbon
    {
        $cancellationDate = null;
        if ($record->json_object['cancelStatus']['cancelState'] === 'CANCELED') {
            foreach ($record->json_object['cancelStatus']['cancelRequests'] as $request) {
                if ($request['cancelRequestState'] === 'COMPLETED') {
                    $cancellationDate = Carbon::parse($request['cancelCompletedDate']);
                }
            }
        }

        return $cancellationDate;
    }
}
