<?php

namespace Modules\Ebay\Http\Controllers;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProductController;
use App\Data\CreateSkuProductsFromSalesChannelData;
use App\Data\SalesChannelProductImportMappingData;
use App\Data\SalesChannelProductToSkuProductMappingCollectionData;
use App\Exceptions\ExportingEmptyDataException;
use App\Exceptions\SalesChannelProductMissingSkuException;
use App\Response;
use Exception;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Exceptions\EbayBadRequestException;
use Modules\Ebay\Http\Resources\EbayLegacyProductResource;
use Modules\Ebay\Jobs\RefreshEbayLegacyProductsJob;
use Modules\Ebay\Managers\EbayLegacyProductManager;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Throwable;

class EbayLegacyProductController extends AbstractSalesChannelProductController
{
    protected function getModel(): string
    {
        return EbayLegacyProduct::class;
    }

    protected function getResource(): string
    {
        return EbayLegacyProductResource::class;
    }

    // Typehint $integrationInstance with the integration implementation
    public function indexForIntegration(Request $request, EbayIntegrationInstance $integrationInstance)
    {
        return $this->abstractIndexForIntegration($request, $integrationInstance);
    }

    // Typehint $product with the integration implementation
    public function show(EbayIntegrationInstance $integrationInstance, EbayLegacyProduct $product): ?Response
    {
        return $this->abstractShow($product);
    }

    /**
     * @throws Throwable
     */
    public function createSkuProducts(CreateSkuProductsFromSalesChannelData $data, EbayIntegrationInstance $integrationInstance): Response
    {
        try {
            $result = $this->abstractCreateSkuProducts($data, $integrationInstance);
        } catch (SalesChannelProductMissingSkuException $e) {
            return $this->response->addError($e->getMessage(), 'Missing SKU', 'apiErrors')->error(400);
        }
        return $result;
    }

    public function refresh(EbayIntegrationInstance $integrationInstance): Response
    {
        return $this->abstractRefresh(new RefreshEbayLegacyProductsJob($integrationInstance, app($this->getModel())->refreshAdt()));
    }

    /**
     * @throws Throwable
     */
    public function mapProducts(SalesChannelProductToSkuProductMappingCollectionData $data, EbayIntegrationInstance $integrationInstance)
    {
        return $this->abstractMapProducts($data, $integrationInstance);
    }

    /**
     * @throws ExportingEmptyDataException
     */
    protected function exportForIntegration(Request $request, EbayIntegrationInstance $integrationInstance): BinaryFileResponse|Response|RedirectResponse
    {
        return $this->abstractExportForIntegration($request, $integrationInstance);
    }

    /**
     * @throws Exception
     */
    public function importMappings(SalesChannelProductImportMappingData $data, EbayIntegrationInstance $integrationInstance): Response
    {
        return $this->abstractImportMappings($data, $integrationInstance);
    }

    /**
     * @throws Exception
     */
    public function syncInventory(Request $request, EbayIntegrationInstance $integrationInstance): Response
    {
        return $this->abstractSyncInventory($request, $integrationInstance);
    }

    public function hydrateProductTemplate(Request $request, EbayIntegrationInstance $integrationInstance): Response
    {
        return $this->abstractHydrateProductTemplate($request, $integrationInstance);
    }

    /**
     * @throws Exception
     */
    public function getSuggestedCategories(Request $request, EbayIntegrationInstance $integrationInstance): Response
    {
        $request->validate([
            'query' => 'required|string|max:350'
        ]);

        $categorySuggestions = (new EbayLegacyProductManager($integrationInstance))->getSuggestedCategories($request->input('query'));

        return $this->response->addData($categorySuggestions)->success();
    }

    public function buildPayload(Request $request, EbayIntegrationInstance $integrationInstance): Response
    {
        return $this->abstractBuildPayload($request, $integrationInstance);
    }

    public function publish(Request $request, EbayIntegrationInstance $integrationInstance): Response
    {
        try {
            $response = $this->abstractPublish($request, $integrationInstance);
        } catch (EbayBadRequestException $e) {
            return $this->response->addError($e->getMessage(), 'Bad Ebay Request', 'apiErrors')->error(400);
        }
        return $response;
    }
}
