<?php

declare(strict_types=1);

namespace Modules\Ebay\Entities;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrderLine;
use App\Data\SalesOrderLineData;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Modules\Ebay\Database\Factories\EbayOrderItemFactory;
use Modules\Ebay\Enums\EbayOrderFulfillmentStatusEnum;
use Modules\Ebay\Enums\EbayOrderPaymentStatusEnum;

/**
 * @property int $id
 * @property int $ebay_order_id
 * @property string $lineItemId
 * @property string $legacyItemId
 * @property string $sku
 * @property string $title
 * @property float $lineItemCostValue
 * @property string $lineItemCostCurrency
 * @property int $quantity
 * @property string $soldFormat
 * @property string $listingMarketplaceId
 * @property string $purchaseMarketplaceId
 * @property string $lineItemFulfillmentStatus
 * @property float $totalValue
 * @property string $totalCurrency
 * @property float $shippingCostValue
 * @property string $shippingCostCurrency
 * @property array $json_object
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 *
 * @property-read EbayOrder $order
 * @property-read EbayLegacyProduct $product
 */
class EbayOrderItem extends AbstractSalesChannelOrderLine
{
    use HasFactory;

    public const BULK_THRESHOLD = 50;

    public string $dataTableClass = self::class;

    public string $dataTableKey = 'ebay.order_item';

    public static function getUniqueId(): string
    {
        return 'lineItemId';
    }

    public static function getTableUniqueId(): string
    {
        return 'lineItemId';
    }

    public static function getLineItemsKey(): ?string
    {
        return 'lineItems';
    }

    public static function getParentRelationId(): string
    {
        return 'ebay_order_id';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function order(): BelongsTo
    {
        return $this->belongsTo(EbayOrder::class, 'ebay_order_id');
    }

    public function product(): BelongsTo|\Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo
    {
        // TODO: Need to deal with non legacy products eventually
        return $this->belongsTo(
            EbayLegacyProduct::class,
            ['integration_instance_id', 'legacyItemId'],
            ['integration_instance_id', 'ItemID']
        );
    }

    public function legacyProduct(): BelongsTo|\Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo
    {
        // TODO: Need to deal with non legacy products eventually
        return $this->belongsTo(
            EbayLegacyProduct::class,
            ['integration_instance_id', 'legacyItemId'],
            ['integration_instance_id', 'ItemID']
        );
    }

    /**
     * @throws Exception
     */
    public function getSalesOrderLineData(): SalesOrderLineData
    {
        return SalesOrderLineData::from([
            'sales_order_number' => $this->order->orderId,
            'sales_channel_line_id' => $this->{self::getTableUniqueId()},
            'amount' => $this->getAmount(),
            'tax_allocation' => $this->getTaxAllocation(),
            'discount_allocation' => 0,
            'description' => $this->title,
            'product_id' => $this->productListingsProductId,
            'product_listing_id' => $this->productListingsId,
            'is_product' => true,
            'quantity' => $this->order->orderPaymentStatus == EbayOrderPaymentStatusEnum::FULLY_REFUNDED ? 0 : $this->quantity,
            'canceled_quantity' => $this->order->orderPaymentStatus == EbayOrderPaymentStatusEnum::FULLY_REFUNDED ? $this->quantity : 0,
            'sales_channel_listing_id' => $this->{self::getSalesChannelProductUniqueId()},
            'externally_fulfilled_quantity' => $this->order->isExternallyFulfilled() ? $this->quantity : 0,
            'warehouse_id' => $this->getWarehouseId(),
            'nominal_code_id' => $this->getNominalCodeId(),
        ]);
    }

    /**
     * @throws Exception
     */
    public function getWarehouseId(): ?int
    {
        if (! $this->legacyProduct?->productListing?->product) {
            return null;
        }

        return $this->order->warehouseRepository->getPriorityWarehouseIdForProduct(
            $this->legacyProduct?->productListing?->product,
            $this->quantity,
        );
    }

    public function getAmount(): float
    {
        return $this->quantity > 0 ? ($this->json_object['lineItemCost']['value'] / $this->quantity) : 0;
    }

    public function getTaxAllocation(): float
    {
        // TODO: https://developer.ebay.com/api-docs/sell/fulfillment/types/sel:Tax
        return 0;
    }

    public function salesChannelProductClass(): string
    {
        return EbayLegacyProduct::class;
    }

    public static function getSalesChannelProductUniqueId(): string
    {
        return 'legacyItemId';
    }

    /**
     * @return Factory<static>
     */
    public static function newFactory(): Factory
    {
        return EbayOrderItemFactory::new();
    }
}
