<?php

declare(strict_types=1);

namespace Modules\Ebay\Entities;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrder;
use App\Abstractions\Integrations\SalesChannels\SalesChannelOrderManagerInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelOrderRepositoryInterface;
use App\Data\AddressData;
use App\Data\CustomerData;
use App\Data\FinancialLineData;
use App\DTO\SalesOrderDto;
use App\DTO\SalesOrderStatusesDto;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Models\Concerns\BulkImport;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Repositories\FinancialLineRepository;
use App\Repositories\WarehouseRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Modules\Ebay\ApiDataTransferObjects\EbayGetOrdersAdt;
use Modules\Ebay\Database\Factories\EbayOrderFactory;
use Modules\Ebay\Enums\EbayOrderFulfillmentStatusEnum;
use Modules\Ebay\Enums\EbayOrderPaymentStatusEnum;
use Modules\Ebay\Managers\EbayOrderManager;
use Modules\Ebay\Repositories\EbayOrderRepository;
use Spatie\LaravelData\DataCollection;
use Throwable;

/**
 * @property int   $id
 * @property int   $integration_instance_id
 * @property string $orderId
 * @property string $legacyOrderId
 * @property string $creationDate
 * @property Carbon $creationDateUtc
 * @property string $lastModifiedDate
 * @property Carbon $lastModifiedDateUtc
 * @property string $error_log
 * @property EbayOrderPaymentStatusEnum $orderPaymentStatus
 * @property EbayOrderFulfillmentStatusEnum $orderFulfillmentStatus
 * @property float  $totalValue
 * @property string $totalCurrency
 * @property string $ebayCollectAndRemitTax
 * @property string $salesRecordReference
 * @property float  $totalMarketplaceFeeValue
 * @property string $totalMarketplaceFeeCurrency
 * @property array $json_object
 * @property ?Carbon $archived_at
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 * @property-read EbayIntegrationInstance $integrationInstance
 */
class EbayOrder extends AbstractSalesChannelOrder
{
    use BulkImport;
    use HasFactory;

    protected $table = 'ebay_orders';

    // TODO: Lower this to 50 after finishing bulk for ebay order items
    public const BULK_THRESHOLD = 50000;

    protected $guarded = [];

    public WarehouseRepository $warehouseRepository;

    protected $casts = [
        'lastModifiedDateUtc' => 'datetime',
        'creationDateUtc' => 'datetime',
        'error_log' => 'array',
        'json_object' => 'array',
        'orderPaymentStatus' => EbayOrderPaymentStatusEnum::class,
        'orderFulfillmentStatus' => EbayOrderFulfillmentStatusEnum::class,
    ];

    public function __construct(array $attributes = [])
    {
        $this->warehouseRepository = app(WarehouseRepository::class);
        parent::__construct($attributes);
    }

    public static function getIntegrationName(): string
    {
        return 'ebay';
    }

    public static function getOrderDate(): string
    {
        return 'creationDateUtc';
    }

    public static function getCurrency(): ?string
    {
        return 'totalCurrency';
    }

    public static function getUniqueId(): string
    {
        return 'orderId';
    }

    public static function getTableUniqueId(): string
    {
        return 'orderId';
    }

    public static function getLastModified(): string
    {
        return 'lastModifiedDateUtc';
    }

    public function orderItems(): HasMany
    {
        return $this->hasMany(EbayOrderItem::class, 'ebay_order_id', 'id');
    }

    public static function getItemClassName(): string
    {
        return EbayOrderItem::class;
    }



    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function availableColumns(): array
    {
        $this->availableColumns = [
            "id" => "integer",
            "orderId" => "string",
            "sku_sales_order_id" => "integer",
            "salesRecordReference" => "string",
            "orderPaymentStatus" => "string",
            "orderFulfillmentStatus" => "string",
            "totalValue" => "float",
            "totalCurrency" => "string",
            "totalMarketplaceFeeValue" => "float",
            "totalMarketplaceFeeCurrency" => "string",
            "creationDateUtc" => "datetime",
            "lastModifiedDateUtc" => "datetime",
            "error_log" => "string",
            "archived_at" => "datetime",
            "created_at" => "datetime",
            "updated_at" => "datetime",
        ];
        return $this->availableColumns;
    }

    public static function specialLabels(): array
    {
        return [
            'id' => 'ID',
            'sku_sales_order_id' => 'SKU Order',
            'creationDateUtc' => 'Order Date',
            'lastModifiedDateUtc' => 'Last Update',
        ];
    }

    /**
     * @return Factory<static>
     */
    public static function newFactory(): Factory
    {
        return EbayOrderFactory::new();
    }

    /**
     * @throws Exception
     */
    private function getOrderStatus(): string
    {
        if ($this->orderPaymentStatus == EbayOrderPaymentStatusEnum::FULLY_REFUNDED)
        {
            return SalesOrder::STATUS_CLOSED;
        }

        if ($this->orderFulfillmentStatus == EbayOrderFulfillmentStatusEnum::FULFILLED && !$this->isExternallyFulfilled())
        {
            return SalesOrder::STATUS_OPEN;
        }

        return match($this->orderFulfillmentStatus) {
            EbayOrderFulfillmentStatusEnum::FULFILLED => SalesOrder::STATUS_CLOSED,
            EbayOrderFulfillmentStatusEnum::NOT_STARTED, EbayOrderFulfillmentStatusEnum::IN_PROGRESS => SalesOrder::STATUS_OPEN,
            default => throw new Exception('Unknown ebay fulfillment status: ' . $this->orderFulfillmentStatus->value),
        };
    }

    /**
     * For now we are treating fulfilled orders as externally fulfilled, regardless of date
     *
     * https://developer.ebay.com/api-docs/sell/fulfillment/types/sel:OrderFulfillmentStatus
     *
     * @throws Exception
     */
    private function getFulfillmentStatus(): string {
        if ($this->isExternallyFulfilled()) {
            return SalesOrder::FULFILLMENT_STATUS_FULFILLED;
        }
        return match($this->orderFulfillmentStatus) {
            EbayOrderFulfillmentStatusEnum::FULFILLED => SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC,
            EbayOrderFulfillmentStatusEnum::IN_PROGRESS, EbayOrderFulfillmentStatusEnum::NOT_STARTED => SalesOrder::FULFILLMENT_STATUS_UNFULFILLED,
            default => throw new Exception('Unknown ebay fulfillment status: ' . $this->orderFulfillmentStatus->value),
        };
    }

    public function isExternallyFulfilled(): bool
    {
        return ($this->orderFulfillmentStatus == EbayOrderFulfillmentStatusEnum::FULFILLED) && $this->lastModifiedDateUtc < $this->integrationInstance->integration_settings['start_date'];
    }

    /**
     * https://developer.ebay.com/api-docs/sell/fulfillment/types/sel:OrderPaymentStatusEnum
     *
     * @throws Exception
     */
    private function getPaymentStatus(): string {
        return match($this->orderPaymentStatus) {
            EbayOrderPaymentStatusEnum::FAILED, EbayOrderPaymentStatusEnum::PENDING => SalesOrder::PAYMENT_STATUS_UNPAID,
            EbayOrderPaymentStatusEnum::FULLY_REFUNDED => SalesOrder::PAYMENT_STATUS_REFUNDED,
            EbayOrderPaymentStatusEnum::PAID => SalesOrder::PAYMENT_STATUS_PAID,
            EbayOrderPaymentStatusEnum::PARTIALLY_REFUNDED => SalesOrder::PAYMENT_STATUS_PARTIALLY_REFUNDED,
            default => throw new Exception('Unknown ebay payment status: ' . $this->orderPaymentStatus->value),
        };
    }

    /**
     * @throws Exception
     */
    public function getSalesOrderStatusesDto(): SalesOrderStatusesDto
    {
        return SalesOrderStatusesDto::from([
            'order_status' => $this->getOrderStatus(),
            'fulfillment_status' => $this->getFulfillmentStatus(),
            'payment_status' => $this->getPaymentStatus(),
        ]);
    }

    public function getCustomerDto(): ?CustomerData
    {
        $json_object = $this->json_object;

        $billingInfo = $json_object['buyer']['buyerRegistrationAddress'];
        $billingAddress = $billingInfo['contactAddress'];
        $shippingInfo = $json_object['fulfillmentStartInstructions'][0]['shippingStep']['shipTo'];
        $shippingAddress = $shippingInfo['contactAddress'];

        return CustomerData::from([
            'name' => @$billingInfo['fullName'] ?? 'Ebay Customer',
            'email' => @$billingInfo['email'] ?? '',
            'zip' => @$billingAddress['postalCode'],
            'address1' => @$billingAddress['addressLine1'],
            'address2' => @$billingAddress['addressLine2'],
            'city' => @$billingAddress['city'],
            'province' => @$billingAddress['stateOrProvince'],
            'country' => @$billingAddress['countryCode'],
            'company' => @$billingInfo['companyName'],
            'fax' => null,
            'default_shipping_address' => AddressData::from([
                'name' => @$shippingInfo['fullName'],
                'email' => @$shippingInfo['email'] ?? '',
                'zip' => @$shippingAddress['postalCode'],
                'label' => 'Default Address',
                'address1' => @$shippingAddress['addressLine1'],
                'address2' => @$shippingAddress['addressLine2'],
                'city' => @$shippingAddress['city'],
                'province' => @@$shippingAddress['stateOrProvince'],
                'province_code' => @@$shippingAddress['stateOrProvince'],
                'country_code' => @$shippingAddress['countryCode'],
                'country' => @$shippingAddress['countryCode'],
            ]),
            'default_billing_address' => AddressData::from([
                'name' => @$billingInfo['fullName'],
                'email' => @$billingInfo['email'] ?? '',
                'zip' => @$billingAddress['postalCode'],
                'label' => 'Default Address',
                'address1' => @$billingAddress['addressLine1'],
                'address2' => @$billingAddress['addressLine2'],
                'city' => @$billingAddress['city'],
                'province' => @$billingAddress['stateOrProvince'],
                'province_code' => @$billingAddress['stateOrProvince'],
                'country_code' => @$billingAddress['countryCode'],
                'country' => @$billingAddress['countryCode'],
            ]),
        ]);
    }

    /**
     * @throws Throwable
     */
    public function getSalesOrderDto(): SalesOrderDto
    {
        $orderStatuses = $this->getSalesOrderStatusesDto();

        return SalesOrderDto::from([
            'sales_order_number' => $this->orderId,
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'currency_code' => $this->getCurrencyCode($this->totalCurrency),
            'order_date' => $this->creationDateUtc,
            'shipping_method_id' => null,
            'requested_shipping_method' => $this->getRequestedShippingMethod(),
            'order_status' => $orderStatuses->order_status,
            'fulfillment_status' => $orderStatuses->fulfillment_status,
            'payment_status' => $orderStatuses->payment_status,
            'canceled_at' => null, //TODO: Check logic on this
            'sales_order_lines' => $this->getSalesOrderLines(),
            'financial_lines' => $this->getFinancialLines(),
            'sales_channel_order_type' => self::class,
            'sales_channel_order_id' => $this->id,
            'customer' => $this->getCustomerDto(),
            'last_synced_from_sales_channel_at' => $this->updated_at,
        ]);
    }

    public function getFinancialLines(): DataCollection
    {
        $financialLines = parent::getFinancialLines();

        if ($this->totalMarketplaceFeeValue && $this->totalMarketplaceFeeValue > 0)
        {
            $financialLines[] = FinancialLineData::from([
                'sales_order_id' => $this->id,
                'financial_line_type_id' => app(FinancialLineRepository::class)
                    ->getOrCreateFinancialLineType(
                        'Marketplace Fee',
                        FinancialLineClassificationEnum::COST,
                        prorationStrategy: FinancialLineProrationStrategyEnum::REVENUE_BASED,
                        allocateToProducts: true
                    )->id,
                'description' => 'Ebay Marketplace Fee',
                'quantity' => 1,
                'amount' => $this->totalMarketplaceFeeValue,
                'tax_allocation' => 0,
                'allocate_to_products' => 1,
                'proration_strategy' => FinancialLineProrationStrategyEnum::REVENUE_BASED,
            ]);
        }

        return $financialLines;
    }

    public function getShippingTotal(): float
    {
        return (float) $this->json_object['pricingSummary']['deliveryCost']['value'] ?? 0;
    }

    public function getShippingTaxTotal(): float
    {
        return 0;
    }

    private function getRequestedShippingMethod(): string
    {
        return @$this->json_object['fulfillmentStartInstructions'][0]['shippingStep']['shippingServiceCode'] ??
            SalesChannel::UNSPECIFIED_SHIPPING_METHOD;
    }

    /**
     * @throws Exception
     */
    public function manager(EbayIntegrationInstance|IntegrationInstanceInterface $integrationInstance): SalesChannelOrderManagerInterface
    {
        return new EbayOrderManager($integrationInstance);
    }

    public function refreshAdt(): ApiDataTransformerInterface
    {
        return new EbayGetOrdersAdt();
    }

    public static function repository(): SalesChannelOrderRepositoryInterface
    {
        return app(EbayOrderRepository::class);
    }


}
