<?php

declare(strict_types=1);

namespace Modules\Ebay\Entities;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProduct;
use App\Abstractions\Integrations\SalesChannels\SalesChannelProductManagerInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelProductRepositoryInterface;
use App\Collections\ProductImageCollection;
use App\DTO\ProductDto;
use App\DTO\ProductImageDto;
use App\DTO\ProductPricingDto;
use App\Exceptions\SalesChannelProductMissingSkuException;
use App\Helpers;
use App\Importers\Parsers\FieldParserFactory;
use App\Models\DataImportMapping;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\ProductPricingTier;
use App\Models\Setting;
use App\Repositories\DataImportMappingRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Arr;
use Modules\Ebay\ApiDataTransferObjects\EbayGetLegacyProductsAdt;
use Modules\Ebay\Database\Factories\EbayLegacyProductFactory;
use Modules\Ebay\Managers\EbayLegacyProductManager;
use Modules\Ebay\Repositories\EbayLegacyProductRepository;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Spatie\LaravelData\DataCollection;
use Throwable;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property string $ItemID
 * @property string $SKU
 * @property string $ListingDuration
 * @property string $ListingType
 * @property string $Title
 * @property int $QuantityAvailable
 * @property float $CurrentPriceValue
 * @property string $CurrentPriceCurrency
 * @property string $ListingStatus
 * @property string $BidCount
 * @property string $EndingReason
 * @property int $WatchCount
 * @property string $StartTime
 * @property Carbon $StartTimeUtc
 * @property ?string $EndTime
 * @property ?Carbon $EndTimeUtc
 * @property array $json_object
 * @property ?Carbon $updated_at
 * @property Carbon $created_at
 * @property-read EbayIntegrationInstance $integrationInstance
 * @property-read ProductListing $productListing
 */
class EbayLegacyProduct extends AbstractSalesChannelProduct
{
    use HasFactory;

    public const BULK_THRESHOLD = 200;

    protected $guarded = [];

    protected $casts = [
        //'StartTimeUtc' => 'datetime',
        //'EndTimeUtc' => 'datetime',
        'json_object' => 'array',
    ];

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);
    }

    public static function getIntegrationName(): string
    {
        return 'ebay';
    }

    public static function getLastModified(): ?string
    {
        return 'StartTimeUtc';
    }

    public static function getUniqueField(): string
    {
        return 'ItemID';
    }

    public function getListingUrl(): ?string
    {
        return "https://www.ebay.com/itm/{$this->ItemID}";
    }

    public function isUsed(): bool|array
    {
        $linkedOrders = EbayOrderItem::where('legacyItemId', $this->ItemID)->count();

        if ($linkedOrders) {
            return [
                'ebay.orders' => trans_choice('messages.currently_used', $linkedOrders, [
                    'resource' => 'Ebay Order',
                    'model' => 'Ebay Listing('.$this->ItemID.')',
                ]),
            ];
        }

        return false;
    }

    public function isActive(): bool
    {
        // ListingStatus is null for any initial listings downloaded, but they are active
        return $this->ListingStatus === 'Active' || !$this->ListingStatus;
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    protected static function newFactory()
    {
        return EbayLegacyProductFactory::new();
    }

    public static function getSkuField(): string
    {
        return 'SKU';
    }

    public static function getPriceField(): string
    {
        return 'CurrentPriceValue';
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws Throwable
     */
    public function getProductDto(): ProductDto
    {
        $sku = $this->applyDataMapping($this, 'sku');

        throw_if(empty($sku), new SalesChannelProductMissingSkuException(
            'The product with ItemID '.$this->ItemID.' does not have a SKU.'
        ));

        return ProductDto::from([
            'sku' => $sku,
            'name' => $this->applyDataMapping($this, 'item_name'),
            'images' => $this->getImages(),
            'pricing' => $this->getPricing(),
        ]);
    }

    public function getImages(): DataCollection
    {
        return ProductImageDto::collection([]);
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function applyDataMapping(mixed $data, string $mappedColumn): mixed
    {
        if (!$mappings = config()->get('ebay_legacy_mappings')) {
            $mappings = app(DataImportMappingRepository::class)->getListingMappings($this->integration_instance_id);
        }

        $skuMapping = $mappings->where('sku_field', $mappedColumn)->first();
        $listingField = @$skuMapping['listing_field'];

        if (is_null($listingField)) {
            switch ($mappedColumn) {
                case 'title':
                case 'item_name':
                    $listingField = 'Title';
                    break;
                case 'sku':
                    $listingField = 'SKU';
                    break;
                case 'sales_channel_listing_id':
                    $listingField = 'ItemID';
                    break;
                case 'price':
                    $listingField = 'CurrentPriceValue';
                    break;
            }
        }

        //Return value
        $returnValue = @$data[$listingField];

        //Apply mappings
        if ($returnValue && @$skuMapping['parsers'] && is_array($skuMapping['parsers'])) {
            foreach ($skuMapping['parsers'] as $parser) {
                $returnValue = FieldParserFactory::make($parser['rule'], $parser['args'], $data)->parse($returnValue);
            }
        }

        return $returnValue;
    }

    public function availableColumns(): array
    {
        $this->availableColumns = array_merge(Arr::except($this->getAllColumnsAndTypes(), [
            'integration_instance_id',
            'json_object',
        ]), [
            'product' => 'product',
        ]);

        return $this->availableColumns;
    }

    public function manager(EbayIntegrationInstance|IntegrationInstanceInterface $integrationInstance): SalesChannelProductManagerInterface
    {
        return new EbayLegacyProductManager($integrationInstance);
    }

    public function refreshAdt(): ApiDataTransformerInterface
    {
        return new EbayGetLegacyProductsAdt();
    }

    public static function repository(): SalesChannelProductRepositoryInterface
    {
        return app(EbayLegacyProductRepository::class);
    }
}
