<?php

namespace Modules\Ebay\ApiDataTransferObjects;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use Carbon\Carbon;
use Modules\Ebay\ApiDataTransferObjects\Concerns\EbayDatesTrait;

class EbayGetOrdersAdt implements ApiDataTransformerInterface
{
    use EbayDatesTrait;

    const DATES_TO_SANITIZE = [
        'last_updated_after',
        'last_updated_before',
        'created_after',
        'created_before',
    ];

    public function __construct(
        public Carbon|string|null $last_updated_after = null,
        public Carbon|string|null $last_updated_before = null,
        public ?int $daysUpdated = null,
        public Carbon|string|null $created_after = null,
        public Carbon|string|null $created_before = null,
        public ?int $daysCreated = null,
        public ?string $fulfillmentStatus = null,
        public ?string $nextEndpoint = null
    ) {
        $this->sanitizeDates();
        if (! is_null($daysUpdated)) {
            $this->setStartAndEndDateFromDays($daysUpdated, 'last_updated');
        } elseif (! is_null($daysCreated)) {
            $this->setStartAndEndDateFromDays($this->daysCreated, 'created');
        }
    }

    public function transform(): array
    {
        if ($this->nextEndpoint) {
            parse_str(parse_url($this->nextEndpoint)['query'], $params);

            return $params;
        } else {
            $params = [];
            $filters = [];
            if ($this->last_updated_after || $this->last_updated_before) {
                $filters[] = 'lastmodifieddate:['.($this->last_updated_after ? $this->getZuluDate('last_updated_after') : '').'..'.($this->last_updated_before ? $this->getZuluDate('last_updated_before') : '').']';
            } elseif ($this->created_after || $this->created_before) {
                $filters[] = 'creationdate:['.($this->created_after ? $this->getZuluDate('created_after') : '').'..'.($this->created_before ? $this->getZuluDate('created_before') : '').']';
            }
            if ($this->fulfillmentStatus) {
                $filters[] = 'orderfulfillmentstatus:['.$this->fulfillmentStatus.']';
            }
            if (count($filters)) {
                $params['filter'] = implode(',', $filters);
            }

            $params['limit'] = 200;

            return array_filter($params, fn ($param) => ! is_null($param));
        }
    }
}
