<?php

namespace Modules\Amazon\Tests\Feature\Repositories;

use App\Models\SalesOrder;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Enums\Entities\OrderStatusEnum;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class AmazonOrderRepositoryTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    private AmazonOrderRepository $amazonOrderRepository;

    protected function setUp(): void
    {
        parent::setUp();

        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()
            ->hasSalesChannel()
            ->create();

        $this->amazonOrderRepository = new AmazonOrderRepository($this->amazonIntegrationInstance);
    }

    /**
     * @throws Exception
     */
    public function test_get_active_itemless_order_ids(): void
    {
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->count(10)
            ->make()
            ->map(function ($order) {
                $order->json_object = array_merge($order->json_object, [
                    'OrderStatus' => OrderStatusEnum::STATUS_SHIPPED,
                ]);
                $order->save();
            });

        $this->assertEquals($this->amazonOrderRepository->getActiveItemlessOrderIds(10)->count(), 10);
    }

    /**
     * @throws Exception
     */
    public function test_reset_pii_data_for_fulfilled_orders(): void
    {
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->count(10)
            ->make()
            ->map(function ($order) {
                $order->json_object = array_merge($order->json_object, [
                    'OrderStatus' => OrderStatusEnum::STATUS_SHIPPED,
                    'BuyerInfo' => [
                        'BuyerEmail' => $this->faker->email(),
                        'BuyerName' => $this->faker->name(),
                    ],
                    'ShippingAddress' => encryptArray([
                        'Name' => $this->faker->name(),
                    ]),
                ]);
                $order->save();

                $salesOrder = SalesOrder::factory()->make();
                $salesOrder->sales_channel_order_id = $order->id;
                $salesOrder->sales_channel_order_type = AmazonOrder::class;
                $salesOrder->fulfilled_at = now()->subDays(31);
                $salesOrder->save();
            });

        $this->amazonOrderRepository->resetPiiDataForFulfilledOrders();

        $this->assertEquals(0, $this->amazonOrderRepository->getFulfilledOrdersForPiiReset()->count());
    }
}
