<?php

namespace Modules\Amazon\Tests\Feature\Managers;

use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\WarehouseTransfers\NotOpenWarehouseTransferException;
use App\Helpers;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\PaymentType;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Models\Setting;
use App\Models\ShippingCarrier;
use App\Models\ShippingMethod;
use App\Models\StockTake;
use App\Models\Store;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use App\Models\WarehouseTransferShipment;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceiptLine;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Storage;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Actions\CreateAmazonReport;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\Actions\RequestAmazonReport;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetReportsAdt;
use Modules\Amazon\Data\AmazonFbaLedgerReconciliationReportData;
use Modules\Amazon\Data\AmazonFbaLedgerReconciliationReportProductData;
use Modules\Amazon\Data\RequestAmazonReportData;
use Modules\Amazon\Entities\AmazonFbaInboundShipFromMapping;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonFbaInboundShipmentItem;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportCustomerReturn;
use Modules\Amazon\Entities\AmazonFbaReportInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedgerSummary;
use Modules\Amazon\Entities\AmazonFbaReportRemovalOrder;
use Modules\Amazon\Entities\AmazonFbaReportRemovalShipment;
use Modules\Amazon\Entities\AmazonFbaReportRestock;
use Modules\Amazon\Entities\AmazonFbaReportShipment;
use Modules\Amazon\Entities\AmazonFinancialEventGroup;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonLedgerDetail;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonOrderItem;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Entities\AmazonReportRequest;
use Modules\Amazon\Entities\AmazonReportSettlementData;
use Modules\Amazon\Enums\Entities\AmazonLedgerDispositionEnum;
use Modules\Amazon\Enums\Entities\FbaInboundShipmentStatusEnum;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Modules\Amazon\Enums\Entities\FbaRemovalOrderTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Jobs\SyncAmazonReportsJob;
use Modules\Amazon\Managers\AmazonFnskuProductManager;
use Modules\Amazon\Managers\AmazonInboundManager;
use Modules\Amazon\Managers\AmazonInitialInventoryManager;
use Modules\Amazon\Managers\AmazonLedgerManager;
use Modules\Amazon\Managers\AmazonRemovalOrderManager;
use Modules\Amazon\Managers\AmazonReportManager;
use Modules\Amazon\Repositories\AmazonFnskuRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Modules\Amazon\Tests\Feature\Managers\Helpers\AmazonMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonReportManagerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;
    use WithFaker;

    private Carbon $date;

    private string $order_id;

    private string $fnsku;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    private AmazonProduct $amazonProduct;

    private Product $product;

    private SalesOrder $salesOrder;

    private AmazonOrder $amazonOrder;

    private AmazonOrderItem $amazonOrderItem;

    private AmazonReport $amazonReportForLedger;

    private AmazonReport $amazonReportForEvent;

    private AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger;

    private EloquentCollection $amazonFbaReportInventoryLedgers;

    private AmazonFbaInboundShipment $amazonFbaInboundShipment;

    private Warehouse $warehouse;

    private Supplier $supplier;

    private EloquentCollection $amazonFbaReportRemovalShipments;

    private EloquentCollection $amazonFbaReportRemovalOrders;

    private AmazonFbaReportCustomerReturn $amazonFbaReportCustomerReturn;

    protected function setUp(): void
    {
        parent::setUp();

        Queue::fake();

        $this->date = Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE))->addDay();
        $this->order_id = $this->faker->uuid();
        $this->fnsku = 'fnsku_value';

        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()->has(SalesChannel::factory())->create();
        (new InitializeFbaWarehouse($this->amazonIntegrationInstance))->handle();

        $integration_settings = $this->amazonIntegrationInstance->integration_settings;
        $integration_settings['fba_inventory_tracking_start_date'] = '2000-05-01 00:00:00';
        $this->amazonIntegrationInstance->integration_settings = $integration_settings;
        $this->amazonIntegrationInstance->save();

        PaymentType::factory()->create([
            'name' => 'Amazon Payment'
        ]);

        $this->amazonIntegrationInstance->refresh();

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        $this->warehouse = $warehouse;

        /** @var Supplier $supplier */
        $supplier = Supplier::query()->first();
        $this->supplier = $supplier;
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    private function setUpProduct(): void
    {
        $this->amazonProduct = AmazonProduct::factory()->for($this->amazonIntegrationInstance)->make();
        $this->amazonProduct->json_object = array_merge($this->amazonProduct->json_object, [
            'fulfillment_channel' => $this->amazonIntegrationInstance->getFbaFulfillmentChannel(),
        ]);
        $this->amazonProduct->save();
        $this->amazonProduct->refresh();

        $this->product = Product::factory()->create([
            'sku' => $this->amazonProduct->seller_sku,
        ]);

        SupplierProduct::factory()->create([
            'product_id' => $this->product->id,
            'supplier_id' => $this->supplier->id,
        ]);

        ProductListing::factory()->create([
            'product_id' => $this->product->id,
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'sales_channel_listing_id' => $this->amazonProduct->seller_sku,
            'listing_sku' => $this->amazonProduct->seller_sku,
            'document_id' => $this->amazonProduct->id,
            'document_type' => AmazonProduct::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $this->date,
            'product_id' => $this->product->id,
            'warehouse_id' => $this->warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 50,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        // set up summary
        AmazonFbaInitialInventory::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fifo_layer_id' => null,
            'sku_product_initialized_at' => null,
            'json_object' => [
                'date' => $this->amazonIntegrationInstance->fbaInventoryTrackingStartDate()->subDay()->format('m/d/Y'),
                'fnsku' => $this->fnsku,
                'asin' => $this->amazonProduct->asin1,
                'msku' => $this->amazonProduct->seller_sku,
                'title' => $this->amazonProduct->item_name,
                'starting_warehouse_balance' => 50,
                'ending_warehouse_balance' => 50,
                'in_transit_between_warehouses' => 5,
                'location' => 'US',
                'disposition' => 'SELLABLE',
            ],
        ]);

        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->generateFnskuProducts();
        (new AmazonInitialInventoryManager($this->amazonIntegrationInstance))->reconcileAllInitialInventory();

        $this->assertDatabaseHas(InventoryMovement::class, [
            'type' => InventoryMovement::TYPE_STOCK_TAKE,
            'product_id' => $this->product->id
        ]);

        $this->assertDatabaseHas(StockTake::class, [
            'is_initial_count' => true,
        ]);
    }

    private function setUpSalesOrder(bool $withLines = true): void
    {
        $this->amazonOrder = AmazonOrder::factory()
            ->has(AmazonOrderItem::factory(1, [
                'json_object' => [
                    'SellerSKU' => $this->amazonProduct->seller_sku,
                    'ASIN' => $this->amazonProduct->asin1,
                    'OrderItemId' => $this->faker->word(),
                    'QuantityOrdered' => 1,
                    'QuantityShipped' => 1,
                ],
            ]), 'orderItems')
            ->create([
                'integration_instance_id' => $this->amazonIntegrationInstance->id,
            ])
            ->refresh()
            ->load('orderItems');

        $jsonObject = $this->amazonOrder->json_object;
        $jsonObject['FulfillmentChannel'] = 'AFN';
        $this->amazonOrder->json_object = $jsonObject;
        $this->amazonOrder->save();
        $this->amazonOrder->refresh();

        $this->amazonOrderItem = $this->amazonOrder->orderItems()->first();

        $salesOrderFactory = SalesOrder::factory();

        if ($withLines) {
            $salesOrderFactory = $salesOrderFactory->has(SalesOrderLine::factory(1, [
                'product_id' => $this->product->id,
                'quantity' => 1,
                'sales_channel_line_id' => $this->amazonOrder->orderItems->first()->OrderItemId,
                'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            ]));
        }

        $this->salesOrder = $salesOrderFactory->create([
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'order_status' => SalesOrder::STATUS_DRAFT,
            'sales_channel_order_id' => $this->amazonOrder->id,
            'sales_channel_order_type' => AmazonOrder::class,
            'order_status' => SalesOrder::STATUS_CLOSED,
            'order_date' => $this->date,
        ]);

        // TODO: Need to reserve inventory
        $this->salesOrder->approve();
    }

    private function setUpAmazonFbaInboundShipment(): void
    {
        $this->amazonFbaInboundShipment = AmazonFbaInboundShipment::factory()
            ->has(AmazonFbaInboundShipmentItem::factory(1, [
                'json_object' => [
                    'SellerSKU' => $this->amazonProduct->seller_sku,
                    'FulfillmentNetworkSKU' => $this->amazonFbaReportInventoryLedger->fnsku,
                    'QuantityShipped' => 1,
                ],
            ]))
            ->create([
                'integration_instance_id' => $this->amazonIntegrationInstance->id,
                'json_object' => [
                    'ShipmentId' => $this->amazonFbaReportInventoryLedger->reference_id,
                    'ShipmentName' => $this->faker->word(),
                    'ShipFromAddress' => [
                        'Name' => $this->faker->word(),
                    ],
                    'DestinationFulfillmentCenterId' => $this->faker->word(),
                    'ShipmentStatus' => $this->faker->randomElement(FbaInboundShipmentStatusEnum::STATUSES_ACTIVE),
                    'LabelPrepType' => $this->faker->word(),
                    'BoxContentsSource' => $this->faker->word(),
                ],
            ])->refresh();
    }

    /**
     * @throws Throwable
     */
    private function setUpWarehouseTransfer(): void
    {
        $this->setUpAmazonFbaInboundShipment();

        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->generateFnskuProducts();

        AmazonFbaInboundShipFromMapping::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'name' => $this->amazonFbaInboundShipment->ShipFromName,
            'link_type' => Warehouse::class,
            'link_id' => $this->warehouse->id,
        ]);

        (new AmazonInboundManager($this->amazonIntegrationInstance))->process([$this->amazonFbaInboundShipment->id]);
    }

    /**
     * @throws Throwable
     */
    private function setUpPurchaseOrder(): void
    {
        $this->setUpAmazonFbaInboundShipment();

        /** @var Store $store */
        $store = Store::query()->first();

        AmazonFbaInboundShipFromMapping::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'name' => $this->amazonFbaInboundShipment->ShipFromName,
            'link_type' => Supplier::class,
            'link_id' => $this->supplier->id,
        ]);

        (new AmazonInboundManager($this->amazonIntegrationInstance))->process([$this->amazonFbaInboundShipment->id]);
    }

    /**
     * @throws Exception
     */
    private function setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum $event_type, ?AmazonLedgerDispositionEnum $disposition = AmazonLedgerDispositionEnum::SELLABLE): void
    {
        $this->amazonReportForLedger = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER,
        ]);

        $quantity = 1;

        if (in_array($event_type->value, FbaInventoryLedgerReportEventTypeEnum::NegativeQuantities)) {
            $quantity = -1;
        }

        $fnsku = $this->fnsku;

        $this->amazonFbaReportInventoryLedger = AmazonFbaReportInventoryLedger::factory()->create([
            'amazon_report_id' => $this->amazonReportForLedger->id,
            'event_datetime' => $this->date->copy(),
            'json_object' => [
                'date' => $this->date->copy()->format('m/d/Y'),
                'fnsku' => $fnsku,
                'asin' => isset($this->amazonProduct) ? $this->amazonProduct->asin1 : $this->faker->word(),
                'msku' => isset($this->amazonProduct) ? $this->amazonProduct->seller_sku : $this->faker->word(),
                'event_type' => $event_type(),
                'quantity' => $quantity,
                'fulfillment_center' => 'ABC7',
                'reference_id' => $this->faker->word(),
                'disposition' => $disposition ? $disposition() : null,
                'checksum' => $this->faker->randomElements(),
                'country' => 'US'
            ],
        ])->refresh();
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    private function setUpLedgerReports(FbaInventoryLedgerReportEventTypeEnum $event_type, int $count): void
    {
        $this->amazonReportForLedger = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER,
        ]);

        $quantity = 1;

        if (in_array($event_type->value, FbaInventoryLedgerReportEventTypeEnum::NegativeQuantities)) {
            $quantity = -1;
        }

        $this->amazonFbaReportInventoryLedgers = AmazonFbaReportInventoryLedger::factory($count)->create([
            'amazon_report_id' => $this->amazonReportForLedger->id,
            'event_datetime' => $this->date->copy(),
            'json_object' => [
                'date' => $this->date->format('m/d/Y'),
                'fnsku' => $this->fnsku,
                'asin' => isset($this->amazonProduct) ? $this->amazonProduct->asin1 : $this->faker->word(),
                'msku' => isset($this->amazonProduct) ? $this->amazonProduct->seller_sku : $this->faker->word(),
                'event_type' => $event_type(),
                'quantity' => $quantity,
                'fulfillment_center' => 'ABC7',
                'reference_id' => $this->faker->word(),
                'country' => 'US',
                'disposition' => 'SELLABLE',
            ],
        ]);

        $this->amazonFbaReportInventoryLedgers = AmazonFbaReportInventoryLedger::all();

        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->generateFnskuProducts($this->amazonReportForLedger);
    }

    /**
     * @throws Exception
     */
    private function setUpCustomerReturnsReport(AmazonLedgerDispositionEnum $returnReason, bool $linkDetailsToReport = true): void
    {
        /** @var AmazonReport $amazonReportForCustomerReturn */
        $amazonReportForCustomerReturn = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_CUSTOMER_RETURNS,
            'processed_at' => $this->date->copy(),
        ]);

        $this->amazonFbaReportCustomerReturn = AmazonFbaReportCustomerReturn::factory()->create([
            'amazon_report_id' => $amazonReportForCustomerReturn->id,
            'event_datetime' => $this->date->copy(),
            'json_object' => [
                'return_date' => $this->date->copy()->toIso8601String(),
                'order_id' => isset($this->amazonOrder->AmazonOrderId) ?
                    $this->amazonOrder->AmazonOrderId : $this->faker->word(),
                'sku' => isset($this->amazonProduct) ? $this->amazonProduct->seller_sku : $this->faker->word(),
                'fnsku' => $this->amazonFbaReportInventoryLedger->fnsku,
                'quantity' => 1,
                'fulfillment_center_id' => $this->amazonFbaReportInventoryLedger->fulfillment_center,
                'reason' => $returnReason(),
                'detail' => $this->faker->word(),
                'detailed_disposition' => $returnReason(),
                'customer_comments' => $this->faker->word(),
            ],
        ])->refresh();

        if ($linkDetailsToReport) {
            (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkAllLedgersToDetailReports();
        }
    }

    /**
     * @throws Exception
     */
    private function setUpFbaRemovableShipmentReport(AmazonLedgerDispositionEnum $disposition, bool $linkDetailsToReport = true): void
    {
        /** @var AmazonReport $amazonReport */
        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_SHIPMENTS,
        ]);

        AmazonFbaReportRemovalShipment::factory()->create([
            'amazon_report_id' => $amazonReport->id,
            'event_datetime' => $this->date->copy(),
            'json_object' => [
                'shipment_date' => $this->date->copy()->toIso8601String(),
                'order_id' => isset($this->amazonOrder->AmazonOrderId) ?
                    $this->amazonOrder->AmazonOrderId : $this->faker->word(),
                'sku' => isset($this->amazonProduct) ? $this->amazonProduct->seller_sku : $this->faker->word(),
                'fnsku' => $this->faker->word(),
                'disposition' => $disposition(),
                'shipped_quantity' => 1,
                'tracking_number' => $this->faker->word(),
            ],
        ])->refresh();

        if ($linkDetailsToReport) {
            (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkAllLedgersToDetailReports();
        }
    }

    /**
     * @throws Exception
     */
    private function setUpShipmentsReport(bool $linkDetailsToReport = true): void
    {
        /** @var AmazonReport $amazonReportForShipments */
        $this->amazonReportForEvent = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_SHIPMENTS,
        ]);

        AmazonFbaReportShipment::factory()->create([
            'amazon_report_id' => $this->amazonReportForEvent->id,
            'event_datetime' => $this->date->copy(),
            'json_object' => [
                'reporting_date' => $this->date->copy()->toIso8601String(),
                'amazon_order_id' => isset($this->amazonOrder->AmazonOrderId) ?
                    $this->amazonOrder->AmazonOrderId : $this->faker->word(),
                'amazon_order_item_id' => $this->amazonOrderItem->OrderItemId ?? $this->faker->word(),
                'sku' => isset($this->amazonProduct) ? $this->amazonProduct->seller_sku : $this->faker->word(),
                'quantity_shipped' => 1,
                'fulfillment_center_id' => $this->amazonFbaReportInventoryLedger->fulfillment_center,
            ],
        ])->refresh();

        if ($linkDetailsToReport) {
            (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkAllLedgersToDetailReports();
        }
    }

    /**
     * @throws Exception
     */
    private function setUpRemovalShipmentsReport(): void
    {
        /** @var AmazonReport $amazonReportForRemovalShipments */
        $amazonReportForRemovalShipments = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_SHIPMENTS,
        ]);

        // TODO: may need to use sequence here
        $this->amazonFbaReportRemovalShipments = AmazonFbaReportRemovalShipment::factory(20)->create([
            'amazon_report_id' => $amazonReportForRemovalShipments->id,
            'event_datetime' => $this->date->copy(),
            'json_object' => [
                'shipment_date' => $this->date->copy()->toIso8601String(),
                'order_id' => $this->order_id,
                'fnsku' => $this->fnsku,
                'sku' => isset($this->amazonProduct) ? $this->amazonProduct->seller_sku : $this->faker->word(),
                'shipped_quantity' => 1,
                'removal_order_type' => FbaRemovalOrderTypeEnum::Return,
                'carrier' => 'UPS',
                'tracking_number' => $this->faker->word(),
                'disposition' => AmazonLedgerDispositionEnum::SELLABLE(),
            ],
        ]);

        $this->amazonFbaReportRemovalShipments = AmazonFbaReportRemovalShipment::all();
    }

    /**
     * @throws Exception
     */
    private function setUpRemovalOrdersReport(): void
    {
        /** @var AmazonReport $amazonReportForRemovalOrders */
        $amazonReportForRemovalOrders = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
        ]);

        $this->amazonFbaReportRemovalOrders = AmazonFbaReportRemovalOrder::factory(1)->create([
            'amazon_report_id' => $amazonReportForRemovalOrders->id,
            'json_object' => [
                'request_date' => $this->date->copy()->toIso8601String(),
                'order_id' => $this->order_id,
                'sku' => isset($this->amazonProduct) ? $this->amazonProduct->seller_sku : $this->faker->word(),
                'fnsku' => $this->fnsku,
                'order_type' => FbaRemovalOrderTypeEnum::Return(),
                'order_status' => $this->faker->word(),
                'last_updated_date' => $this->date->copy()->toIso8601String(),
                'requested_quantity' => 1,
                'disposition' => $this->faker->word(),
            ],
        ]);

        $this->amazonFbaReportRemovalOrders = AmazonFbaReportRemovalOrder::all();
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_no_adjustment_created_when_no_product_found(): void
    {
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Adjustments);

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->assertDatabaseCount((new InventoryAdjustment())->getTable(), 0);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_create_inventory_adjustment_from_ledger(): void
    {
        $this->setUpProduct();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Adjustments);

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $reason = getRecordFromArray(AmazonReport::$adjustmentReasons, 'Code', $this->amazonFbaReportInventoryLedger->reason);
        $notes = $reason['Code'].' ('.$reason['Type'].') '.$reason['Reason'].'. '.$reason['Definition'];

        $this->assertDatabaseHas((new InventoryAdjustment())->getTable(), [
            'adjustment_date' => Carbon::parse($this->date),
            'product_id' => $this->product->id,
            'quantity' => 1,
            'notes' => $notes,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);

        /** @var InventoryAdjustment $inventoryAdjustment */
        $inventoryAdjustment = InventoryAdjustment::query()
            ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
            ->latest('id')
            ->first();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => InventoryAdjustment::class,
            'sku_link_id' => $inventoryAdjustment->id,
        ]);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_create_sales_credit_return_line_from_ledger(): void
    {
        $this->setUpProduct();
        $this->setUpSalesOrder();

        // Ship first so that sales credit can happen
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Shipments);
        $this->setUpShipmentsReport();
        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::CustomerReturns);
        $this->setUpCustomerReturnsReport(AmazonLedgerDispositionEnum::DEFECTIVE);
        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);;
        $this->assertDatabaseHas((new SalesCredit())->getTable(), [
            'sales_credit_number' => 'CR-'.$this->salesOrder->sales_order_number,
            'sales_order_id' => $this->salesOrder->id,
            'credit_date' => Carbon::parse($this->amazonFbaReportInventoryLedger->date, $this->amazonIntegrationInstance->getTimezone())->setTimezone('UTC')->format('Y-m-d H:i:s'),
            'to_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'sales_credit_note' => 'Amazon FBA Customer Return',
        ]);

        $this->assertDatabaseHas((new SalesCreditReturn())->getTable(), [
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'received_at' => Carbon::parse($this->amazonFbaReportInventoryLedger->date, $this->amazonIntegrationInstance->getTimezone())->setTimezone('UTC')->format('Y-m-d H:i:s'),
        ]);

        $this->assertDatabaseHas((new SalesCreditReturnLine())->getTable(), [
            'quantity' => 1,
            'action' => SalesCreditReturnLine::ACTION_ADD_TO_STOCK,
            'notes' => $this->amazonFbaReportCustomerReturn->customer_comments ?? '',
        ]);
        /** @var SalesCreditReturnLine $salesCreditReturnLine */
        $salesCreditReturnLine = SalesCreditReturnLine::query()->first();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => SalesCreditReturnLine::class,
            'sku_link_id' => $salesCreditReturnLine->id,
        ]);

        /*
         * +1 active amazon, type return
         */

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'inventory_movement_date' => Carbon::parse($this->date),
            'product_id' => $this->product->id,
            'quantity' => 1,
            'type' => InventoryMovement::TYPE_RETURN,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'link_id' => $salesCreditReturnLine->id,
            'link_type' => SalesCreditReturnLine::class,
            'reference' => $salesCreditReturnLine->salesCreditLine->salesCredit->sales_credit_number,
        ]);

        $salesCreditReturnLine->delete();

        $this->assertEquals(
            0,
            AmazonFbaReportInventoryLedger::query()
                ->where('sku_link_type', SalesCreditReturnLine::class)
                ->count()
        );
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_throws_exception_for_customer_return_with_no_sales_order_without_fulfillment_and_no_adjustment_but_amazon_order_after_start_date(): void
    {
        $this->setUpProduct();
        $this->setUpSalesOrder();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::CustomerReturns);
        $this->setUpCustomerReturnsReport(AmazonLedgerDispositionEnum::DEFECTIVE);

        //        try {
        //            (new AmazonLedgerManager($this->amazonIntegrationInstance))->createAuditTrailFromAmazonLedger($this->amazonReportForLedger);
        //        } catch (Exception $e)
        //        {
        //            $this->assertEquals(
        //                "Sales order for Customer Return Ledger (ID: {$this->amazonFbaReportInventoryLedger->id}) does not have a fulfillment, which means something processed out of order.",
        //                $e->getMessage()
        //            );
        //        }

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->assertDatabaseHas((new InventoryAdjustment())->getTable(), [
            'adjustment_date' => Carbon::parse($this->date),
            'product_id' => $this->product->id,
            'quantity' => 1,
            'notes' => "Customer Return Ledger (ID: {$this->amazonFbaReportInventoryLedger->id}) does not have a corresponding fulfillment, so processing as an adjustment instead of Sales Credit Return.  This is likely due to bad Amazon data causing the fulfillment to be missing a corresponding ledger.",
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);
        /** @var InventoryAdjustment $inventoryAdjustment */
        $inventoryAdjustment = InventoryAdjustment::query()
            ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
            ->latest('id')
            ->first();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => InventoryAdjustment::class,
            'sku_link_id' => $inventoryAdjustment->id,
        ]);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_mark_audit_trail_error_for_customer_return_with_no_amazon_order(): void
    {
        $this->setUpProduct();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::CustomerReturns);
        $this->setUpCustomerReturnsReport(AmazonLedgerDispositionEnum::DEFECTIVE);

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->assertDatabaseHas((new InventoryAdjustment())->getTable(), [
            'product_id' => $this->product->id,
            'quantity' => 1,
            'notes' => "Customer Return Ledger (ID: {$this->amazonFbaReportInventoryLedger->id}) does not have a corresponding fulfillment, so processing as an adjustment instead of Sales Credit Return.  This is likely due to a Return on a Sales Order that occurred prior to Initial Count.",
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);
    }

    /**
     * @throws Exception
     * @throws Throwable
     *
     * Receipts
     */
    public function test_it_can_create_warehouse_shipment_receipt_from_ledger(): void
    {
        $this->setUpProduct();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Receipts);
        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->generateFnskuProducts($this->amazonReportForLedger);
        $this->setUpWarehouseTransfer();

        $ledgers = (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);
        $ledger = $ledgers->first();

        $this->assertDatabaseHas((new WarehouseTransferShipmentLine())->getTable(), [
            'quantity' => 1,
        ]);

        /** @var WarehouseTransferShipmentReceiptLine $warehouseTransferShipmentReceiptLine */
        $warehouseTransferShipmentReceiptLine = $ledger->skuLink;

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => WarehouseTransferShipmentReceiptLine::class,
            'sku_link_id' => $warehouseTransferShipmentReceiptLine->id,
        ]);

        // Check inventory movements

        /*
         * -1 active warehouse, type transfer
         * +1 reserved warehouse, type transfer
         * -1 reserved warehouse, type transfer
         * +1 in_transit amazon, type transfer
         * -1 in_transit amazon, type transfer
         * +1 active amazon, type transfer
         */

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $this->product->id,
            'quantity' => -1,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'warehouse_id' => $this->warehouse->id,
            'link_id' => $warehouseTransferShipmentReceiptLine->shipmentLine->id,
            'link_type' => WarehouseTransferShipmentLine::class,
            'reference' => $warehouseTransferShipmentReceiptLine->shipmentReceipt->shipment->warehouseTransfer->warehouse_transfer_number,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $this->product->id,
            'quantity' => 1,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'link_id' => $warehouseTransferShipmentReceiptLine->shipmentLine->id,
            'link_type' => WarehouseTransferShipmentLine::class,
            'reference' => $warehouseTransferShipmentReceiptLine->shipmentReceipt->shipment->warehouseTransfer->warehouse_transfer_number,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $this->product->id,
            'quantity' => -1,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'link_id' => $warehouseTransferShipmentReceiptLine->id,
            'link_type' => WarehouseTransferShipmentReceiptLine::class,
            'reference' => $warehouseTransferShipmentReceiptLine->shipmentReceipt->shipment->warehouseTransfer->warehouse_transfer_number,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'product_id' => $this->product->id,
            'quantity' => 1,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'link_id' => $warehouseTransferShipmentReceiptLine->id,
            'link_type' => WarehouseTransferShipmentReceiptLine::class,
            'reference' => $warehouseTransferShipmentReceiptLine->shipmentReceipt->shipment->warehouseTransfer->warehouse_transfer_number,
        ]);

        $warehouseTransferShipmentReceiptLine->delete();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => null,
            'sku_link_id' => null,
            'errorLog' => null,
            'last_reconciliation_attempted_at' => null,
            'reconciled_at' => null,
        ]);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_create_purchase_order_shipment_receipt_from_ledger(): void
    {
        $this->setUpProduct();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Receipts);
        $this->setUpPurchaseOrder();

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->assertDatabaseCount((new PurchaseOrderShipmentReceipt())->getTable(), 1);
        $this->assertDatabaseCount((new PurchaseOrderShipmentReceiptLine())->getTable(), 1);
        $this->assertDatabaseCount((new PurchaseOrderShipment())->getTable(), 1);
        $this->assertDatabaseCount((new PurchaseOrderShipmentLine())->getTable(), 1);

        /** @var PurchaseOrderShipmentReceiptLine $purchaseOrderShipmentReceiptLine */
        $purchaseOrderShipmentReceiptLine = PurchaseOrderShipmentReceiptLine::query()->first();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => PurchaseOrderShipmentReceiptLine::class,
            'sku_link_id' => $purchaseOrderShipmentReceiptLine->id,
        ]);

        $purchaseOrderShipmentReceiptLine->delete();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => null,
            'sku_link_id' => null,
        ]);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_create_sales_order_fulfillment_line_from_ledger(): void
    {
        $this->setUpProduct();
        $this->setUpSalesOrder();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Shipments);
        $this->setUpShipmentsReport();

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->assertDatabaseHas((new SalesOrderFulfillmentLine())->getTable(), [
            'quantity' => 1,
        ]);

        /** @var SalesOrderFulfillmentLine $salesOrderFulfillmentLine */
        $salesOrderFulfillmentLine = SalesOrderFulfillmentLine::query()->first();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => SalesOrderFulfillmentLine::class,
            'sku_link_id' => $salesOrderFulfillmentLine->id,
        ]);

        /*
         * -1 active amazon, type sale
         * +1 reserved amazon, type sale
         * -1 reserved amazon, type sale
         */
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'inventory_movement_date' => Carbon::parse($this->date),
            'product_id' => $this->product->id,
            'quantity' => -1,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'link_id' => $salesOrderFulfillmentLine->salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'reference' => $this->salesOrder->sales_order_number,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'inventory_movement_date' => Carbon::parse($this->date),
            'product_id' => $this->product->id,
            'quantity' => 1,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'link_id' => $salesOrderFulfillmentLine->salesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'reference' => $this->salesOrder->sales_order_number,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'inventory_movement_date' => Carbon::parse($this->date),
            'product_id' => $this->product->id,
            'quantity' => -1.0,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'link_id' => $salesOrderFulfillmentLine->id,
            'link_type' => SalesOrderFulfillmentLine::class,
            'reference' => $salesOrderFulfillmentLine->salesOrderFulfillment->getReference(),
        ]);
        $this->assertDatabaseHas(SalesOrderLineLayer::class, [
            'sales_order_line_id' => $salesOrderFulfillmentLine->salesOrderLine->id,
            'quantity' => 1,
        ]);

        $this->salesOrder->delete();

        $this->assertEquals(
            0,
            AmazonFbaReportInventoryLedger::query()
                ->where('sku_link_type', SalesOrderFulfillmentLine::class)
                ->count()
        );
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_create_warehouse_transfer_shipment_line_from_ledger(): void
    {
        $this->setUpProduct();
        $this->setUpSalesOrder();
        $this->setUpLedgerReports(FbaInventoryLedgerReportEventTypeEnum::VendorReturns, 1);
        $this->setUpRemovalOrdersReport();

        $removalOrders = (new AmazonRemovalOrderManager($this->amazonIntegrationInstance))->process(
            [$this->amazonFbaReportRemovalOrders->first()->id],
            $this->warehouse->id,
        );

        $warehouseTransfer = $removalOrders->first()->skuLink->warehouseTransfer;

        $this->assertDatabaseHas((new AmazonFbaReportRemovalOrder())->getTable(), [
            'order_id' => $this->order_id,
            'sku_link_id' => $warehouseTransfer->warehouseTransferLines()->first()->id,
            'sku_link_type' => WarehouseTransferLine::class,
        ]);

        $this->setUpRemovalShipmentsReport();

        $removalOrderIds = $this->amazonFbaReportRemovalShipments->flatten()->pluck('id');
        $this->amazonFbaReportInventoryLedgers->each(function ($amazonFbaReportInventoryLedger) use (&$removalOrderIds) {
            (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkLedgerToDetailReports($amazonFbaReportInventoryLedger, [$removalOrderIds->shift()]);
        });
        $ledgers = (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);
        $ledger = $ledgers->first();

        $shippingMethod = ShippingMethod::whereHas('shippingCarrier', function ($query) {
            $query->where('name', 'UPS');
        })
            ->where('name', 'Unspecified')
            ->firstOrFail();

        $this->assertDatabaseHas((new WarehouseTransferShipment())->getTable(), [
            'shipped_at' => Carbon::parse($this->date),
            'shipping_method_id' => $shippingMethod->id,
            'tracking_number' => $this->amazonFbaReportRemovalShipments->first()->tracking_number,
        ]);

        $this->assertDatabaseHas((new WarehouseTransferShipment())->getTable(), [
            'shipped_at' => Carbon::parse($this->date),
            'shipping_method_id' => $shippingMethod->id,
            'tracking_number' => $this->amazonFbaReportRemovalShipments->last()->tracking_number,
        ]);

        $this->assertDatabaseCount((new WarehouseTransferShipment())->getTable(), 1);

        $this->assertDatabaseHas((new WarehouseTransferShipmentLine())->getTable(), [
            'quantity' => 1,
        ]);

        $this->assertDatabaseCount((new WarehouseTransferShipmentLine())->getTable(), 1);

        /** @var WarehouseTransferShipmentLine $warehouseTransferShipmentLine */
        $warehouseTransferShipmentLine = $ledger->skuLink;

        /*
         * -1 active amazon, type transfer
         * +1 reserved amazon, type transfer
         * -1 reserved amazon, type transfer
         * +1 in_transit warehouse, type transfer
         */
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'inventory_movement_date' => Carbon::parse($this->date),
            'product_id' => $this->product->id,
            'quantity' => -1,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'link_id' => $warehouseTransferShipmentLine->id,
            'link_type' => WarehouseTransferShipmentLine::class,
            'reference' => $this->order_id,
        ]);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'inventory_movement_date' => Carbon::parse($this->date),
            'product_id' => $this->product->id,
            'quantity' => 1,
            'type' => InventoryMovement::TYPE_TRANSFER,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_IN_TRANSIT,
            'warehouse_id' => $this->warehouse->id,
            'link_id' => $warehouseTransferShipmentLine->id,
            'link_type' => WarehouseTransferShipmentLine::class,
            'reference' => $this->order_id,
        ]);

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedgers->first()->id,
            'sku_link_type' => WarehouseTransferShipmentLine::class,
            'sku_link_id' => $warehouseTransferShipmentLine->id,
        ]);

        /*
         * Need to delete the warehouse transfer and...
         * Delete the removal order link to the warehouse transfer line
         * Delete the amazon ledger link to the warehouse transfer shipment line
         */

        $warehouseTransfer->delete();

        $this->assertEquals(
            0,
            AmazonFbaReportRemovalOrder::query()
                ->where('sku_link_type', WarehouseTransferLine::class)
                ->count()
        );

        $this->assertEquals(
            0,
            AmazonFbaReportInventoryLedger::query()
                ->where('sku_link_type', WarehouseTransferShipmentLine::class)
                ->count()
        );
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_save_ledger_report(): void
    {
        $filename = 'inventory_ledger_report.tsv';

        $this->mockReport($filename);

        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER(),
            'filename' => $filename,
        ]);

        (new AmazonReportManager($this->amazonIntegrationInstance))->process($amazonReport);

        $this->assertDatabaseCount((new AmazonFbaReportInventoryLedger())->getTable(), 500);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_save_amazon_fba_shipment_report(): void
    {
        $filename = 'amazon_fba_shipment_report.tsv';

        $this->mockReport($filename);

        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_SHIPMENTS(),
            'filename' => $filename,
        ]);

        (new AmazonReportManager($this->amazonIntegrationInstance))->process($amazonReport);

        $this->assertDatabaseCount((new AmazonFbaReportShipment())->getTable(), 174);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_save_fba_restock_report(): void
    {
        $filename = 'fba_restock_report.tsv';

        $this->mockReport($filename);

        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_RESTOCK(),
            'filename' => $filename,
        ]);

        (new AmazonReportManager($this->amazonIntegrationInstance))->process($amazonReport);

        $this->assertDatabaseCount((new AmazonFbaReportRestock())->getTable(), 633);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_save_fba_removal_orders_report(): void
    {
        $filename = 'amazon_removal_orders_report.tsv';

        $this->mockReport($filename);

        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS(),
            'filename' => $filename,
        ]);

        (new AmazonReportManager($this->amazonIntegrationInstance))->process($amazonReport);

        $this->assertDatabaseCount((new AmazonFbaReportRemovalOrder())->getTable(), 500);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_save_settlement_report(): void
    {
        $filename = 'amazon_settlement_report.tsv';

        $this->mockReport($filename);

        $financialEventGroupStart = Carbon::parse('2022-12-27 15:24:06');
        $financialEventGroupEnd = Carbon::parse('2023-01-10 15:24:05');

        AmazonFinancialEventGroup::factory()->for($this->amazonIntegrationInstance)->create([
            'json_object' => [
                'FinancialEventGroupId' => $this->faker->unique()->numberBetween(),
                'ProcessingStatus' => 'Open',
                'OriginalTotal' => [
                    'CurrencyCode' => 'USD',
                    'CurrencyAmount' => '42180.44',
                ],
                'ConvertedTotal' => [
                    'CurrencyCode' => 'USD',
                    'CurrencyAmount' => $this->faker->randomFloat(),
                ],
                'BeginningBalance' => [
                    'CurrencyCode' => 'USD',
                    'CurrencyAmount' => $this->faker->randomFloat(),
                ],
                'FinancialEventGroupStart' => $financialEventGroupStart->toIso8601ZuluString(),
                'FinancialEventGroupEnd' => $financialEventGroupEnd->toIso8601ZuluString(),
            ],
            'FinancialEventGroupStart_datetime' => $financialEventGroupStart,
            'FinancialEventGroupEnd_datetime' => $financialEventGroupEnd,
        ]);

        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::SETTLEMENT_REPORT(),
            'filename' => $filename,
        ]);

        (new AmazonReportManager($this->amazonIntegrationInstance))->process($amazonReport);

        // We are asserting that the settlement report is 499 records because 2 lines are taken up by headers.  The total lines in the file is 501.
        $this->assertDatabaseCount((new AmazonReportSettlementData())->getTable(), 499);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_save_fba_inventory_report(): void
    {
        $filename = 'amazon_fba_inventory_report.tsv';

        $this->mockReport($filename);

        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_INVENTORY(),
            'filename' => $filename,
        ]);

        (new AmazonReportManager($this->amazonIntegrationInstance))->process($amazonReport);

        $this->assertDatabaseCount((new AmazonFbaReportInventory())->getTable(), 500);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_save_fba_inventory_ledger_summary(): void
    {
        $filename = 'inventory_ledger_summary.tsv';

        $this->mockReport($filename);

        //Create AmazonReport
        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY(),
            'filename' => $filename,
        ]);

        //Parse file to create amazon_products
        $records = tsvToArray(file_get_contents(Storage::disk('amazon_reports')->path($filename)), [
            'ignoreMissingColumns' => false,
            'trimColumns' => true,
        ]);
        $records = collect($records)->map(function ($record) {
            $record['seller_sku'] = $record['msku'];
            $record['asin1'] = $record['asin'];
            $record['item_name'] = $record['title'];

            return array_merge([
                'json_object' => $record,
            ], $record);
        });

        //Process Ledger Summary file
        try {
            (new AmazonReportManager($this->amazonIntegrationInstance))->process($amazonReport);
        } catch (Throwable $e) {
        }
        // 216 minus the 3 records that have a 0 total inventory quantity
        $this->assertDatabaseCount((new AmazonFbaInitialInventory())->getTable(), 213);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_save_fba_report_customer_returns_report(): void
    {
        $filename = 'amazon_fba_report_customer_returns_report.tsv';

        $this->mockReport($filename);

        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_CUSTOMER_RETURNS(),
            'filename' => $filename,
        ]);

        (new AmazonReportManager($this->amazonIntegrationInstance))->process($amazonReport);

        $this->assertDatabaseCount((new AmazonFbaReportCustomerReturn())->getTable(), 589);
    }

    /**
     * @throws Throwable
     */
    public function test_link_ledgers_to_detail_reports_for_customer_returns(): void
    {
        $this->setUpProduct();
        $this->setUpSalesOrder();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::CustomerReturns);
        $this->setUpCustomerReturnsReport(AmazonLedgerDispositionEnum::SELLABLE, false);

        $unlinkedRecordsCount = AmazonFbaReportInventoryLedger::whereDoesntHave('ledgerDetails')->count();

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkAllLedgersToDetailReports();

        $this->assertEquals($unlinkedRecordsCount, AmazonFbaReportInventoryLedger::whereHas('customerReturn')->count());
    }

    /**
     * @throws Throwable
     */
    public function test_link_ledgers_to_detail_reports_for_vendor_returns(): void
    {
        $this->setUpProduct();
        $this->setUpSalesOrder();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::VendorReturns);
        $this->setUpFbaRemovableShipmentReport(AmazonLedgerDispositionEnum::SELLABLE, false);

        $unlinkedRecordsCount = AmazonFbaReportInventoryLedger::whereDoesntHave('ledgerDetails')->count();

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkAllLedgersToDetailReports();

        $this->assertEquals($unlinkedRecordsCount, AmazonFbaReportInventoryLedger::whereHas('removalShipment')->count());
    }

    /**
     * @throws Throwable
     */
    public function test_link_ledgers_to_detail_reports_for_shipments(): void
    {
        $this->setUpProduct();
        $this->setUpSalesOrder();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Shipments);
        $this->setUpShipmentsReport(false);

        $unlinkedRecordsCount = AmazonFbaReportInventoryLedger::whereDoesntHave('ledgerDetails')->count();

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkAllLedgersToDetailReports();

        $this->assertEquals($unlinkedRecordsCount, AmazonFbaReportInventoryLedger::whereHas('shipments')->count());
    }

    /**
     * @throws Throwable
     */
    public function test_link_detail_reports_to_ledgers_for_shipments(): void
    {
        $this->setUpProduct();
        $this->setUpSalesOrder();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Shipments);
        $this->setUpShipmentsReport(false);

        $unlinkedRecordsCount = AmazonFbaReportInventoryLedger::whereDoesntHave('ledgerDetails')->count();

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkAllLedgersToDetailReports();

        $this->assertEquals($unlinkedRecordsCount, AmazonFbaReportInventoryLedger::whereHas('shipments')->count());
    }

    /**
     * @throws Throwable
     */
    public function test_report_cannot_be_requested_if_unacceptable_date_range(): void
    {
//        // Successful request
//        $reportRequest = (new RequestAmazonReport(RequestAmazonReportData::from([
//            'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
//            'report_type' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY,
//            'data_start_date' => Carbon::now()->subDays(3)->subSecond(),
//            'createImmediately' => false,
//        ])))->handle();
//
//        $this->assertCount(
//            1,
//            $reportRequest->createdRequests
//        );

        // Unsuccessful due to being unable to make end date before start date
        $reportRequest = (new RequestAmazonReport(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
            'report_type' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY,
            'data_start_date' => Carbon::now()->subDay(),
            'data_end_date' => Carbon::now()->subDays(3),
            'createImmediately' => false,
        ])))->handle();

        $this->assertEmpty($reportRequest->createdRequests);
        $this->assertEmpty($reportRequest->createdReports);

        // Successful due to end date too recent but able to make it after start date
        $reportRequest = (new RequestAmazonReport(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
            'report_type' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY,
            'data_start_date' => Carbon::now()->subDays(4),
            'data_end_date' => Carbon::now()->subDay(),
            'createImmediately' => false,
        ])))->handle();

        $this->assertCount(
            1,
            $reportRequest->createdRequests
        );

        // For splittable reports, last report is date is adjusted
        $reportRequest = (new RequestAmazonReport(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
            'report_type' => AmazonReportTypeEnum::FBA_REPORT_SHIPMENTS,
            'data_start_date' => Carbon::now()->subDays(90),
            'createImmediately' => false,
        ])))->handle();

        // Current date must be at least 72 hours greater than the requested report end time minus 1 second
        $this->assertGreaterThan(
            $reportRequest->createdRequests[2]['dataEndTime']->addHours(72)->subSecond(),
            Carbon::now()
        );
    }

    public function test_was_detail_report_processed(): void
    {
        $date = Carbon::parse('2023-05-05');

        $reportType = AmazonReportTypeEnum::FBA_REPORT_SHIPMENTS;

        AmazonReport::factory()->create([
            'reportType' => $reportType,
            'processingStatus' => 'DONE',
            'processed_at' => Carbon::now(),
            'dataStartTime' => Carbon::parse('2023-05-05 05:00:00'),
            'dataEndTime' => Carbon::parse('2023-05-06 00:00:00'),
        ]);

        $this->assertFalse(app(AmazonReportRepository::class)->wasReportTypeProcessedForDate($reportType, $date));

        AmazonReport::query()->delete();

        AmazonReport::factory()->create([
            'reportType' => $reportType,
            'processingStatus' => 'DONE',
            'processed_at' => Carbon::now(),
            'dataStartTime' => Carbon::parse('2023-05-05 00:00:00'),
            'dataEndTime' => Carbon::parse('2023-05-06 00:00:00'),
        ]);

        $this->assertTrue(app(AmazonReportRepository::class)->wasReportTypeProcessedForDate($reportType, $date));

        AmazonReport::query()->delete();

        AmazonReport::factory()->create([
            'reportType' => $reportType,
            'processingStatus' => 'DONE',
            'processed_at' => Carbon::now(),
            'dataStartTime' => Carbon::parse('2023-05-05 00:00:00'),
            'dataEndTime' => Carbon::parse('2023-05-05 23:59:58'),
        ]);

        $this->assertFalse(app(AmazonReportRepository::class)->wasReportTypeProcessedForDate($reportType, $date));

        AmazonReport::query()->delete();

        AmazonReport::factory()->create([
            'reportType' => $reportType,
            'processingStatus' => 'DONE',
            'processed_at' => Carbon::now(),
            'dataStartTime' => Carbon::parse('2023-05-05 00:00:00'),
            'dataEndTime' => Carbon::parse('2023-05-05 23:59:59'),
        ]);

        $this->assertTrue(app(AmazonReportRepository::class)->wasReportTypeProcessedForDate($reportType, $date));

        AmazonReport::query()->delete();
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_clear_amazon_audit_trail_for_product(): void
    {
        $this->markTestSkipped('TODO: Revisit with new desired logic');
        $this->setUpProduct();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::CustomerReturns);

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->assertDatabaseHas((new InventoryAdjustment())->getTable(), [
            'adjustment_date' => Carbon::parse($this->date),
            'product_id' => $this->product->id,
            'quantity' => 1,
            'notes' => "Sales order for Customer Return Ledger (ID: {$this->amazonFbaReportInventoryLedger->id}) could not be determined due to no detail report match.",
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);
        /** @var InventoryAdjustment $inventoryAdjustment */
        $inventoryAdjustment = InventoryAdjustment::query()
            ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
            ->latest('id')
            ->first();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => InventoryAdjustment::class,
            'sku_link_id' => $inventoryAdjustment->id,
        ]);

        /** @var AmazonFnskuProduct $amazonFnskuProduct */
        $amazonFnskuProduct = AmazonFnskuProduct::query()->first();

        $product = $amazonFnskuProduct->product;

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->clearAuditTrailFromAmazonLedger($product);

        $this->assertEquals(
            0,
            InventoryAdjustment::query()
                ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
                ->count()
        );

        $this->assertEquals(
            0,
            InventoryMovement::query()
                ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
                ->count()
        );

        $this->assertEquals(
            0,
            FifoLayer::query()
                ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
                ->count()
        );

        $this->assertEquals(
            0,
            AmazonFbaReportInventoryLedger::query()
                ->whereNotNull('sku_link_id')
                ->count()
        );

        $this->assertEquals(
            0,
            AmazonFbaInitialInventory::query()
                ->whereNotNull('fifo_layer_id')
                ->orWhereNotNull('sku_product_initialized_at')
                ->count()
        );

        $this->assertEquals(
            0,
            AmazonFnskuProduct::query()
                ->whereHas('amazonFbaInitialInventory', function ($query) {
                    $query->whereHas('fifoLayer');
                })
                ->count()
        );
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_clear_detail_reports_from_amazon_ledger(): void
    {
        $this->setUpProduct();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::CustomerReturns);
        $this->setUpCustomerReturnsReport(AmazonLedgerDispositionEnum::SELLABLE);

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->assertDatabaseHas((new AmazonFbaReportCustomerReturn())->getTable(), [
            'fnsku' => $this->amazonFbaReportInventoryLedger->fnsku,
        ]);

        // Inventory adjustment because no order was set up above
        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'sku_link_type' => InventoryAdjustment::class,
        ]);
        $this->assertDatabaseHas((new AmazonLedgerDetail())->getTable(), [
            'detail_type' => AmazonFbaReportCustomerReturn::class,
        ]);
        $this->assertDatabaseHas((new AmazonReport())->getTable(), [
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_CUSTOMER_RETURNS->value,
            'processed_at' => $this->date,
        ]);

        /** @var AmazonFbaReportCustomerReturn $customerReturnsReport */
        $customerReturnsReport = AmazonFbaReportCustomerReturn::query()->first();

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->resetDetailReport($customerReturnsReport->amazonReport);

        $this->assertDatabaseEmpty((new AmazonFbaReportCustomerReturn())->getTable());
        $this->assertDatabaseMissing((new AmazonFbaReportInventoryLedger())->getTable(), [
            'sku_link_type' => InventoryAdjustment::class,
        ]);
        $this->assertDatabaseMissing((new AmazonLedgerDetail())->getTable(), [
            'detail_type' => AmazonFbaReportCustomerReturn::class,
        ]);
        $this->assertDatabaseHas((new AmazonReport())->getTable(), [
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_CUSTOMER_RETURNS->value,
            'processed_at' => null,
        ]);
    }

    /**
     * If sales orders are missing but in the ledger report post start date, we can download them.  The movements
     * themselves are after the start date anyway.
     *
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws NotOpenWarehouseTransferException
     */
    public function test_it_can_download_missing_sales_orders_needed_for_ledger_fulfillment(): void
    {
        $this->setUpProduct();
        // We don't set up the sales order and let the createAuditTrailFromAmazonLedger automatically download it
        //$this->setUpSalesOrder();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Shipments);
        $this->setUpShipmentsReport();

        /** @var AmazonFbaReportShipment $amazonFbaReportShipment */
        $amazonFbaReportShipment = AmazonFbaReportShipment::query()->first();
        $amazonOrderIdFromShipmentReport = $amazonFbaReportShipment->amazon_order_id;

        Http::fake([
            config('amazon.url_access_token') => Http::response([
                'access_token' => $this->faker->sha1(),
            ]),
            '*/tokens/2021-03-01/restrictedDataToken*' => Http::response([
                'restrictedDataToken' => $this->faker->sha1(),
            ]),
            "*/orders/v0/orders/$amazonOrderIdFromShipmentReport" => Http::response([
                'payload' => [
                    'BuyerInfo' => [],
                    'AmazonOrderId' => $amazonOrderIdFromShipmentReport,
                    'EarliestShipDate' => Carbon::parse($this->faker->dateTimeBetween('-1 year'))->toIso8601String(),
                    'SalesChannel' => 'Amazon.com',
                    'OrderStatus' => 'Shipped',
                    'NumberOfItemsShipped' => $this->faker->unique()->numberBetween(),
                    'OrderType' => 'StandardOrder',
                    'IsPremiumOrder' => false,
                    'IsPrime' => false,
                    'FulfillmentChannel' => 'AFN',
                    'NumberOfItemsUnshipped' => $this->faker->unique()->numberBetween(),
                    'HasRegulatedItems' => false,
                    'IsReplacementOrder' => 'false',
                    'IsSoldByAB' => false,
                    'LatestShipDate' => Carbon::parse($this->faker->dateTimeBetween('-1 year'))->toIso8601String(),
                    'ShipServiceLevel' => 'Standard',
                    'IsISPU' => false,
                    'MarketplaceId' => 'ATVPDKIKX0DER',
                    'PurchaseDate' => $this->amazonIntegrationInstance->fbaInventoryTrackingStartDate()->subMonth(),
                    'ShippingAddress' => [],
                    'IsAccessPointOrder' => false,
                    'SellerOrderId' => $this->faker->unique()->numberBetween(),
                    'PaymentMethod' => 'Other',
                    'IsBusinessOrder' => false,
                    'OrderTotal' => [
                        'CurrencyCode' => 'USD',
                        'Amount' => $this->faker->randomFloat(2, 0, 1000),
                    ],
                    'PaymentMethodDetails' => [],
                    'IsGlobalExpressEnabled' => false,
                    'LastUpdateDate' => Carbon::parse($this->faker->dateTimeBetween('-1 year'))->toIso8601String(),
                    'ShipmentServiceLevelCategory' => 'Standard',
                ],
            ]),
            "*/orders/v0/orders/$amazonOrderIdFromShipmentReport/orderItems" => Http::response([
                'payload' => [
                    'OrderItems' => [
                        [
                            'TaxCollection' => [
                                'Model' => 'MarketplaceFacilitator',
                                'ResponsibleParty' => 'Amazon Services, Inc.',
                            ],
                            'ProductInfo' => [
                                'NumberOfItems' => '1',
                            ],
                            'BuyerInfo' => [],
                            'ItemTax' => [
                                'CurrencyCode' => 'USD',
                                'Amount' => '0.78',
                            ],
                            'QuantityShipped' => 1,
                            'ItemPrice' => [
                                'CurrencyCode' => 'USD',
                                'Amount' => '12.95',
                            ],
                            'ASIN' => $this->amazonProduct->asin1,
                            'SellerSKU' => $this->amazonProduct->seller_sku,
                            'Title' => 'Tecmate Optimate Cable O-01, Weatherproof Battery Lead, powersport',
                            'IsGift' => 'false',
                            'IsTransparency' => false,
                            'QuantityOrdered' => 1,
                            'PromotionDiscountTax' => [
                                'CurrencyCode' => 'USD',
                                'Amount' => '0.00',
                            ],
                            'PromotionDiscount' => [
                                'CurrencyCode' => 'USD',
                                'Amount' => '0.00',
                            ],
                            'OrderItemId' => $amazonFbaReportShipment->amazon_order_item_id,
                        ],
                    ],
                ],
            ]),
        ]);
#dd(ProductListing::all()->toArray(), Product::all()->toArray());
        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_order_number' => $amazonOrderIdFromShipmentReport,
        ]);

        $this->assertDatabaseHas((new SalesOrderFulfillmentLine())->getTable(), [
            'quantity' => 1,
        ]);

        /** @var SalesOrderFulfillmentLine $salesOrderFulfillmentLine */
        $salesOrderFulfillmentLine = SalesOrderFulfillmentLine::query()->first();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'sku_link_type' => SalesOrderFulfillmentLine::class,
            'sku_link_id' => $salesOrderFulfillmentLine->id,
        ]);

        $this->assertEquals(
            1,
            InventoryMovement::query()
                ->where('link_type', SalesOrderFulfillmentLine::class)
                ->where('link_id', $salesOrderFulfillmentLine->id)
                ->whereDate('inventory_movement_date', '>=', $this->amazonIntegrationInstance->fbaInventoryTrackingStartDate())
                ->count()
        );

        $this->assertEquals(
            0,
            InventoryMovement::query()
                ->whereDate('inventory_movement_date', '<', $this->amazonIntegrationInstance->fbaInventoryTrackingStartDate())
                ->count()
        );
    }

    /**
     * If inbound is missing we can download it.  The inbound itself has no date (See https://github.com/amzn/selling-partner-api-docs/issues/2706).
     * If the user marked the inbound shipment already as is_before_initial_count, we make an adjustment with an appropriate note.
     *
     * @throws Throwable
     */
    public function test_it_can_download_missing_inbounds_needed_for_ledger_receipt(): void
    {
        $this->setUpProduct();
        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Receipts);

        /** @var AmazonFbaReportInventoryLedger $amazonLedgerReport */
        $amazonLedgerReport = AmazonFbaReportInventoryLedger::query()->first();
        $inboundIdFromLedger = $amazonLedgerReport->reference_id;

        // We don't set up the inbound or warehouse transfer and let the createAuditTrailFromAmazonLedger automatically download it for checking date
        //$this->setUpWarehouseTransfer();

        Http::fake([
            config('amazon.url_access_token') => Http::response([
                'access_token' => $this->faker->sha1(),
            ]),
            '*/fba/inbound/v0/shipments?*' => Http::response([
                'payload' => [
                    'ShipmentData' => [
                        [
                            'ShipmentId' => $inboundIdFromLedger,
                            'ShipmentName' => 'FBA 2022-02-05 04:27:56 - 2',
                            'ShipFromAddress' => [
                                'Name' => $this->faker->name(),
                                'AddressLine1' => $this->faker->streetAddress(),
                                'City' => $this->faker->city(),
                                'StateOrProvinceCode' => 'CA',
                                'CountryCode' => $this->faker->countryCode(),
                                'PostalCode' => $this->faker->postcode(),
                            ],
                            'DestinationFulfillmentCenterId' => $this->faker->word(),
                            'ShipmentStatus' => 'CLOSED',
                            'LabelPrepType' => 'SELLER_LABEL',
                            'AreCasesRequired' => true,
                            'BoxContentsSource' => 'FEED',
                        ],
                    ],
                ],
            ]),
            "*/fba/inbound/v0/shipments/$inboundIdFromLedger/items" => Http::response([
                'payload' => [
                    'ItemData' => [
                        [
                            'ShipmentId' => $inboundIdFromLedger,
                            'SellerSKU' => $this->amazonProduct->seller_sku,
                            'FulfillmentNetworkSKU' => $amazonLedgerReport->fnsku,
                            'QuantityShipped' => 1,
                            'QuantityReceived' => 1,
                            'QuantityInCase' => 1,
                            'PrepDetailsList' => [
                                [
                                    'PrepInstruction' => 'Labeling',
                                    'PrepOwner' => 'SELLER',
                                ],
                            ],
                        ],
                    ],
                ],
            ]),
        ]);

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'errorLog' => 'Inbound Shipment Not Processed Yet in SKUio',
        ]);

        $this->assertDatabaseHas((new AmazonFbaInboundShipment())->getTable(), [
            'ShipmentId' => $inboundIdFromLedger,
        ]);

        $this->assertDatabaseHas((new AmazonFbaInboundShipmentItem())->getTable(), [
            'SellerSKU' => $this->amazonProduct->seller_sku,
            'FulfillmentNetworkSKU' => $amazonLedgerReport->fnsku,
        ]);

        /** @var AmazonFbaInboundShipment $amazonInboundShipment */
        $amazonInboundShipment = AmazonFbaInboundShipment::query()->first();

        $amazonInboundShipment->is_before_initial_count = true;
        $amazonInboundShipment->save();

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($this->amazonReportForLedger);

        /** @var InventoryAdjustment $inventoryAdjustment */
        $inventoryAdjustment = InventoryAdjustment::query()
            ->where('notes', "Receipt Ledger (ID: $amazonLedgerReport->id) was received inbound shipment $amazonInboundShipment->ShipmentId which was marked by User as sent before initial count.")
            ->where('quantity', 1)
            ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
            ->first();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $this->amazonFbaReportInventoryLedger->id,
            'errorLog' => null,
            'sku_link_id' => $inventoryAdjustment->id,
            'sku_link_type' => InventoryAdjustment::class,
        ]);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_type' => InventoryAdjustment::class,
            'link_id' => $inventoryAdjustment->id,
            'warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);
    }

    // TODO: Remove
//    /**
//     * @throws Throwable
//     */
//    public function test_it_can_loosely_match_ledger_report_to_shipment_detail_report(): void
//    {
//        $this->setUpProduct();
//        $this->setUpLedgerReport(FbaInventoryLedgerReportEventTypeEnum::Shipments);
//
//        /** @var AmazonFbaReportInventoryLedger $ledger */
//        $ledger = AmazonFbaReportInventoryLedger::query()->first();
//
//        $ledger->update(['json_object->msku' => 'wrongsku']);
//        $ledger->save();
//        $ledger->refresh();
//
//        $this->setUpShipmentsReport();
//
//        (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkAllLedgersToDetailReports();
//
//        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
//            'id' => $ledger->id,
//            'detail_type' => null,
//            'detail_id' => null,
//        ]);
//
//        (new AmazonLedgerManager($this->amazonIntegrationInstance))->linkLedgerToShipmentReportLoosely($ledger);
//
//        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
//            'id' => $ledger->id,
//            'detail_type' => AmazonFbaReportShipment::class,
//        ]);
//    }

    /**
     * @throws Exception
     */
    public function test_refresh_Settlement_reports(): void
    {
        Queue::fake([
            SyncAmazonReportsJob::class,
        ]);

        Http::fake([
            config('amazon.url_access_token') => Http::response([
                'access_token' => $this->faker->sha1(),
            ]),
            '*reports?*' => Http::response([
                'reports' => [
                    [
                        'reportType' => AmazonReportTypeEnum::SETTLEMENT_REPORT(),
                        'processingEndTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                        'processingStatus' => 'DONE',
                        'marketplaceIds' => [
                            'ATVPDKIKX0DER',
                        ],
                        'reportDocumentId' => fake()->url(),
                        'reportId' => fake()->randomNumber(),
                        'dataEndTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                        'createdTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                        'processingStartTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                        'dataStartTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                    ],
                    [
                        'reportType' => AmazonReportTypeEnum::SETTLEMENT_REPORT(),
                        'processingEndTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                        'processingStatus' => 'DONE',
                        'marketplaceIds' => [
                            'ATVPDKIKX0DER',
                        ],
                        'reportDocumentId' => fake()->url(),
                        'reportId' => fake()->randomNumber(),
                        'dataEndTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                        'createdTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                        'processingStartTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                        'dataStartTime' => fake()->dateTime()->format('Y-m-d\TH:i:sP'),
                    ],
                ],
            ]),
        ]);

        (new AmazonReportManager($this->amazonIntegrationInstance))->refreshSettlementReports(new AmazonGetReportsAdt(
            reportType: AmazonReportTypeEnum::SETTLEMENT_REPORT
        ));

        $this->assertDatabaseCount((new AmazonReport())->getTable(), 2);
    }

    /**
     * Test to create fnsku with null product if amazon product is not found
     *
     * @throws Throwable
     **/
    public function test_create_fnsku_with_null_product_if_amazon_product_is_not_found(): void
    {
        // create a ledger summary record but no amazon product record
        $fnsku = 'fnsku_without_product';

        // set up summary
        AmazonFbaInitialInventory::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fifo_layer_id' => null,
            'sku_product_initialized_at' => null,
            'json_object' => [
                'date' => $this->amazonIntegrationInstance->fbaInventoryTrackingStartDate()->format('m/d/Y'),
                'fnsku' => $fnsku,
                'asin' => $this->faker->word(),
                'msku' => $this->faker->word(),
                'title' => $this->faker->sentence(),
                'starting_warehouse_balance' => 50,
                'ending_warehouse_balance' => 50,
                'location' => 'US',
                'disposition' => 'SELLABLE',
            ],
        ]);
        $this->assertCount(1, app(AmazonFnskuRepository::class)->getFnskusNeedingGeneration($this->amazonIntegrationInstance));
        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->generateFnskuProducts();
        $this->assertCount(1, AmazonFnskuProduct::all());
    }


    /**
     * @throws Exception
     */
    public function test_amazon_ledger_reconciliation_report(): void
    {
        $integrationSettings = $this->amazonIntegrationInstance->integration_settings;
        $integrationSettings['fba_inventory_tracking_start_date'] = Carbon::parse('2024-01-01', 'PST')->setTimezone('UTC')->toDateTimeString();
        $this->amazonIntegrationInstance->integration_settings = $integrationSettings;
        $this->amazonIntegrationInstance->save();

        AmazonFbaReportInventoryLedger::factory(5)
            ->state(new Sequence(
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'json_object' => [
                        'date' => '2024-01-02',
                        'fnsku' => 'X000000000',
                        'msku' => 'msku1',
                        'event_type' => 'CustomerReturns'
                    ],
                    'reconciled_at' => Carbon::now(),
                ],
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'json_object' => [
                        'date' => '2024-01-01',
                        'fnsku' => 'X000000000',
                        'msku' => 'msku1',
                        'event_type' => 'CustomerReturns'
                    ],
                ],
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'json_object' => [
                        'date' => '2024-01-05',
                        'fnsku' => 'X000000001',
                        'msku' => 'msku2',
                        'event_type' => 'CustomerReturns'
                    ],
                ],
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'json_object' => [
                        'date' => '2024-01-08',
                        'fnsku' => 'X000000008',
                        'msku' => 'msku3',
                        'event_type' => 'CustomerReturns'
                    ],
                    'reconciled_at' => Carbon::now(),
                ],
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'json_object' => [
                        'date' => '2024-01-09',
                        'fnsku' => 'X000000009',
                        'msku' => 'msku4',
                        'event_type' => 'CustomerReturns'
                    ],
                ]
            ))
            ->create();

        $currentDate = Carbon::now($this->amazonIntegrationInstance->getTimezone());

        $this->assertEquals(
            AmazonFbaLedgerReconciliationReportData::from([
                'products' => AmazonFbaLedgerReconciliationReportProductData::collection([
                    [
                        'fnsku' => 'X000000008',
                        'fnskuProductId' => null,
                        'msku' => 'msku3',
                        'sku' => null,
                        'daysSinceStartLedgerData' => 7,
                        'daysSinceStartReconciled' => 7,
                        'latestLedgerDate' => Carbon::parse('2024-01-01')->addDays(7)->toDateString(),
                        'reconciliationDate' => Carbon::parse('2024-01-01')->addDays(7)->toDateString(),
                    ],
                    [
                        'fnsku' => 'X000000000',
                        'fnskuProductId' => null,
                        'msku' => 'msku1',
                        'sku' => null,
                        'daysSinceStartLedgerData' => 1,
                        'daysSinceStartReconciled' => 1,
                        'latestLedgerDate' => Carbon::parse('2024-01-01')->addDays(1)->toDateString(),
                        'reconciliationDate' => Carbon::parse('2024-01-01')->addDays(1)->toDateString(),
                    ],
                    [
                        'fnsku' => 'X000000001',
                        'fnskuProductId' => null,
                        'msku' => 'msku2',
                        'sku' => null,
                        'daysSinceStartLedgerData' => 4,
                        'daysSinceStartReconciled' => 0,
                        'latestLedgerDate' => Carbon::parse('2024-01-01')->addDays(4)->toDateString(),
                        'reconciliationDate' => null,
                    ],
                    [
                        'fnsku' => 'X000000009',
                        'fnskuProductId' => null,
                        'msku' => 'msku4',
                        'sku' => null,
                        'daysSinceStartLedgerData' => 8,
                        'daysSinceStartReconciled' => 0,
                        'latestLedgerDate' => Carbon::parse('2024-01-01')->addDays(8)->toDateString(),
                        'reconciliationDate' => null,
                    ],
                ]),
                'startDate' => '2024-01-01',
                'currentDate' => $currentDate->toDateString(),
                'daysSinceStartCurrent' => $currentDate->diffInDays('2024-01-01'),
                'reconciliationDate' => '2024-01-01',
                'reconciledDaysSinceStart' => 0,
                'ledgersReconciled' => 2,
                'ledgersUnreconciled' => 3,
                'topErrors' => [],
            ]),
            (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconciliationReport()
        );
    }

    /**
     * @throws Throwable
     */
    public function test_manually_requested_reports(): void
    {
        $integrationSettings = $this->amazonIntegrationInstance->integration_settings;
        $integrationSettings['fba_inventory_tracking_start_date'] = Carbon::parse('2000-01-01 00:00:00')->toDateTimeString();
        $this->amazonIntegrationInstance->integration_settings = $integrationSettings;
        $this->amazonIntegrationInstance->save();

        $startDate = Carbon::parse('2024-01-01 00:00:00');
        $endDate1 = Carbon::parse('2024-01-05 00:00:00');
        $endDate2 = Carbon::parse('2024-01-10 00:00:00');

        $reportRequest1 = AmazonReportRequest::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY,
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'dataStartTime' => $startDate,
            'dataEndTime' => $endDate1,
            'marketplaceIds' => ['ATVPDKIKX0DER'],
        ]);

        $this->mockCreateReport();

        $report1 = (new CreateAmazonReport($reportRequest1))->handle();

        $report1->filename = 'report1data.tsv';
        $this->mockReport($report1->filename);
        $report1->save();

        app(AmazonReportRepository::class)->saveReport($report1);

        $reportRequest2 = AmazonReportRequest::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY,
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'dataStartTime' => $startDate,
            'dataEndTime' => $endDate2,
            'marketplaceIds' => ['ATVPDKIKX0DER'],
            'requested_by_user_id' => User::first()->id,
        ]);

        $report2 = (new CreateAmazonReport($reportRequest2))->handle();

        $report2->filename = 'report2data.tsv';
        $this->mockReport($report2->filename);
        $report2->save();

        app(AmazonReportRepository::class)->saveReport($report2);

        $this->assertDatabaseCount((new AmazonReport())->getTable(), 2);
        $this->assertDatabaseCount((new AmazonFbaReportInventoryLedgerSummary())->getTable(), 3);
    }
}
