<?php

namespace Modules\Amazon\Tests\Feature\Managers;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\Entities\AmazonFbaReportRemovalOrder;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\FbaRemovalOrderTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Managers\AmazonRemovalOrderManager;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonRemovalOrderManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    private Carbon $date;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    private AmazonProduct $amazonProduct;

    private Warehouse $warehouse;

    private EloquentCollection $amazonFbaReportRemovalOrders;

    protected function setUp(): void
    {
        parent::setUp();

        $this->date = Carbon::parse($this->faker->date());

        Queue::fake();

        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()->hasSalesChannel()->create();
        (new InitializeFbaWarehouse($this->amazonIntegrationInstance))->handle();
        $this->amazonIntegrationInstance->refresh();

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::query()->first();

        $this->warehouse = $warehouse;
    }

    /**
     * @throws Exception
     */
    private function setUpRemovalOrdersReport(): void
    {
        /** @var AmazonReport $amazonReportForRemovalOrders */
        $amazonReportForRemovalOrders = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
        ]);

        $removalOrderId1 = $this->faker->word();
        $removalOrderId2 = $this->faker->word();

        AmazonProduct::factory(20)->create()->each(function (AmazonProduct $amazonProduct) use ($amazonReportForRemovalOrders, $removalOrderId1, $removalOrderId2) {
            $jsonObject = $amazonProduct->json_object;
            $jsonObject['fulfillment_channel'] = $this->amazonIntegrationInstance->getFbaFulfillmentChannel();
            $amazonProduct->json_object = $jsonObject;
            $amazonProduct->save();

            $amazonProduct->refresh();

            $product = Product::factory()->create([
                'sku' => $amazonProduct->seller_sku,
            ]);

            ProductListing::factory()->create([
                'product_id' => $product->id,
                'document_id' => $amazonProduct->id,
                'document_type' => AmazonProduct::class,
            ]);

            AmazonFbaReportRemovalOrder::factory()->create([
                'amazon_report_id' => $amazonReportForRemovalOrders->id,
                'json_object' => [
                    'request_date' => $this->date->toIso8601String(),
                    'order_id' => $this->faker->randomElement([$removalOrderId1, $removalOrderId2]),
                    'sku' => $product->sku,
                    'fnsku' => $this->faker->unique()->word(),
                    'order_type' => FbaRemovalOrderTypeEnum::Return(),
                    'order_status' => $this->faker->word(),
                    'last_updated_date' => $this->date->toIso8601String(),
                    'requested_quantity' => 1,
                    'disposition' => $this->faker->word(),
                ],
            ]);
        });

        $this->amazonFbaReportRemovalOrders = AmazonFbaReportRemovalOrder::all();
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_create_warehouse_transfer_from_removal_order(): void
    {
        $this->setUpRemovalOrdersReport();
        $removalOrders = (new AmazonRemovalOrderManager($this->amazonIntegrationInstance))->process(
            [$this->amazonFbaReportRemovalOrders->first()->id],
            $this->warehouse->id
        );

        $warehouseTransfer = $removalOrders->first()->skuLink->warehouseTransfer;

        $this->assertDatabaseHas((new WarehouseTransfer())->getTable(), [
            'warehouse_transfer_number' => $this->amazonFbaReportRemovalOrders->first()->order_id,
            'from_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'to_warehouse_id' => $this->warehouse->id,
            'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_OPEN,
        ]);
        $this->assertDatabaseHas((new WarehouseTransferLine())->getTable(), [
            'warehouse_transfer_id' => $warehouseTransfer->id,
            'quantity' => $this->amazonFbaReportRemovalOrders->first()->requested_quantity,
        ]);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_process_and_unprocess_removal_orders(): void
    {
        $this->setUpRemovalOrdersReport();

        $integrationSettings = $this->amazonIntegrationInstance->integration_settings;
        $integrationSettings['automatically_create_warehouse_transfers_from_removal_orders'] = true;
        $integrationSettings['removal_order_warehouse_id'] = $this->warehouse->id;
        $this->amazonIntegrationInstance->integration_settings = $integrationSettings;
        $this->amazonIntegrationInstance->save();

        $removalOrderIds = $this->amazonFbaReportRemovalOrders->pluck('id')->toArray();
        (new AmazonRemovalOrderManager($this->amazonIntegrationInstance))->process($removalOrderIds);

        $this->assertDatabaseHas((new WarehouseTransfer())->getTable(), [
            'warehouse_transfer_number' => $this->amazonFbaReportRemovalOrders->first()->order_id,
            'from_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'to_warehouse_id' => $this->warehouse->id,
            'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_OPEN,
        ]);

        $this->assertDatabaseCount((new WarehouseTransfer())->getTable(), 2);
        $this->assertDatabaseCount((new WarehouseTransferLine())->getTable(), 20);

        (new AmazonRemovalOrderManager($this->amazonIntegrationInstance))->unprocess($removalOrderIds);

        $this->assertDatabaseEmpty((new WarehouseTransfer())->getTable(), 0);
        $this->assertDatabaseEmpty((new WarehouseTransferLine())->getTable(), 0);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_mark_removal_order_item_as_unknown(): void
    {
        $this->setUpRemovalOrdersReport();

        (new AmazonRemovalOrderManager($this->amazonIntegrationInstance))->markItemAsUnknown($this->amazonFbaReportRemovalOrders->first());

        $this->assertDatabaseHas((new AmazonFbaReportRemovalOrder())->getTable(), [
            'id' => $this->amazonFbaReportRemovalOrders->first()->id,
            'is_unknown_item' => true,
        ]);

        $integrationSettings = $this->amazonIntegrationInstance->integration_settings;
        $integrationSettings['automatically_create_warehouse_transfers_from_removal_orders'] = true;
        $integrationSettings['removal_order_warehouse_id'] = $this->warehouse->id;
        $this->amazonIntegrationInstance->integration_settings = $integrationSettings;
        $this->amazonIntegrationInstance->save();

        $removalOrderIds = $this->amazonFbaReportRemovalOrders->pluck('id')->toArray();
        (new AmazonRemovalOrderManager($this->amazonIntegrationInstance))->process($removalOrderIds);

        $this->assertDatabaseHas((new WarehouseTransfer())->getTable(), [
            'warehouse_transfer_number' => $this->amazonFbaReportRemovalOrders->first()->order_id,
            'from_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
            'to_warehouse_id' => $this->warehouse->id,
            'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_OPEN,
        ]);

        $this->assertDatabaseCount((new WarehouseTransferLine()), 19);

        (new AmazonRemovalOrderManager($this->amazonIntegrationInstance))->unmarkItemAsUnknown($this->amazonFbaReportRemovalOrders->first());
        (new AmazonRemovalOrderManager($this->amazonIntegrationInstance))->process($removalOrderIds);

        $this->assertDatabaseCount((new WarehouseTransferLine()), 20);
    }
}
