<?php

namespace Modules\Amazon\Tests\Feature\Managers;

use App\Exceptions\SalesChannelProductMappingException;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\Supplier;
use Exception;
use Http;
use Illuminate\Foundation\Testing\WithFaker;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Managers\AmazonProductManager;
use Modules\Amazon\Tests\Feature\Managers\Helpers\AmazonMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Queue;
use Tests\TestCase;
use Throwable;

class AmazonProductManagerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;
    use WithFaker;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    private AmazonProductManager $amazonProductManager;

    protected function setUp(): void
    {
        parent::setUp();

        Queue::fake();

        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create();

        $this->amazonProductManager = new AmazonProductManager($this->amazonIntegrationInstance);

        Http::preventStrayRequests();
    }

    /**
     * A basic unit test example.
     *
     * @throws Exception
     * @throws Throwable
     */
    public function test_create_amazon_sku_products_with_bulk(): void
    {
        //Set number of records
        $numberOfRecords = 200;

        //Create Amazon Products
        AmazonProduct::factory()
            ->for($this->amazonIntegrationInstance)
            ->count($numberOfRecords)
            ->create();

        //Create SKU Products
        (new AmazonProductManager($this->amazonIntegrationInstance))->createSkuProducts([], true);

        //Supplier Check
        $this->assertDatabaseCount((new Supplier())->getTable(), 1);

        //Product Check
        $this->assertDatabaseCount((new Product())->getTable(), $numberOfRecords);

        //Product Listing Check
        $this->assertEquals(ProductListing::where('document_type', AmazonProduct::class)->whereNotNull('document_id')->count(), $numberOfRecords);
    }

    /**
     * A basic unit test example.
     *
     * @throws Exception
     * @throws Throwable
     */
    public function test_create_sku_products(): void
    {
        //Set number of records
        $numberOfRecords = ProductListing::BULK_THRESHOLD - 1;

        //Create Amazon Products
        AmazonProduct::factory()
            ->for($this->amazonIntegrationInstance)
            ->count($numberOfRecords)
            ->create();

        //Create SKU Products
        (new AmazonProductManager($this->amazonIntegrationInstance))->createSkuProducts([], true);

        //Supplier Check
        $this->assertDatabaseCount((new Supplier())->getTable(), 1);

        //Product Check
        $this->assertDatabaseCount((new Product())->getTable(), $numberOfRecords);

        //Product Listing Check
        $this->assertEquals(ProductListing::where('document_type', AmazonProduct::class)->count(), $numberOfRecords);
    }

    public function test_get_catalog_item(): void
    {
        $this->mockGetCatalogItems();
        $asin1 = 'B07B52Z39D';

        $amazonProduct = AmazonProduct::factory()
            ->for($this->amazonIntegrationInstance)
            ->create();

        //Same ASIN used in mock request
        $amazonProduct->json_object = array_merge($amazonProduct->json_object, [
            'asin1' => $asin1,
        ]);
        $amazonProduct->update();
        $amazonProduct = $amazonProduct->fresh();

        //Test
        $this->amazonProductManager->getCatalogItem($amazonProduct);

        $this->assertEquals(1, AmazonProduct::where('asin1', $asin1)->whereNotNull('catalog_data_last_sync')->count());
    }

    public function test_import(): void
    {
        $filename = '1023389019331.tsv';

        $this->mockProductReport($filename);

        $amazonReport = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::PRODUCTS(),
            'filename' => $filename,
        ]);

        $this->amazonProductManager->import($amazonReport);

        $this->assertDatabaseCount((new AmazonProduct())->getTable(), 1330);
    }

    /**
     * @throws Throwable
     */
    public function test_amazon_non_fba_products_create_sku_products_disallow_product_type(): void
    {
        //Set number of records
        $numberOfRecords = 1;

        //Create Amazon Products
        AmazonProduct::factory()
            ->for($this->amazonIntegrationInstance)
            ->count($numberOfRecords)
            ->make()
            ->map(function (AmazonProduct $amazonProduct) {
                $amazonProduct->json_object = array_merge($amazonProduct->json_object, [
                    'fulfillment_channel' => AmazonProductFulfillmentChannelEnum::AMAZON_NA(),
                ]);
                $amazonProduct->save();
            });

        //Create one product with bundle
        $randomAmazonProduct = AmazonProduct::query()->first();
        Product::create([
            'sku' => $randomAmazonProduct->seller_sku,
            'type' => Product::TYPE_BUNDLE,
        ]);

        $this->expectException(SalesChannelProductMappingException::class);

        //Create SKU Products
        (new AmazonProductManager($this->amazonIntegrationInstance))->createSkuProducts([], true);
    }

    /**
     * @throws Throwable
     */
    public function test_amazon_fba_products_create_sku_products_disallow_product_type(): void
    {
        //Set number of records
        $numberOfRecords = 1;

        //Create Amazon Products
        AmazonProduct::factory()
            ->for($this->amazonIntegrationInstance)
            ->count($numberOfRecords)
            ->make()
            ->map(function (AmazonProduct $amazonProduct) {
                $amazonProduct->json_object = array_merge($amazonProduct->json_object, [
                    'fulfillment_channel' => AmazonProductFulfillmentChannelEnum::AMAZON_NA(),
                ]);
                $amazonProduct->save();
            });

        //Create one product with bundle
        $randomAmazonProduct = AmazonProduct::query()->first();
        Product::create([
            'sku' => $randomAmazonProduct->seller_sku,
            'type' => Product::TYPE_MATRIX,
        ]);

        $this->expectException(SalesChannelProductMappingException::class);

        //Create SKU Products
        (new AmazonProductManager($this->amazonIntegrationInstance))->createSkuProducts([], true);
    }
}
