<?php

namespace Feature\Managers;

use App\Exceptions\DropshipWithOpenOrdersException;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\WarehouseTransfers\NotOpenWarehouseTransferException;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedgerSummary;
use Modules\Amazon\Entities\AmazonFbaReportShipment;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonFulfillmentOrder;
use Modules\Amazon\Entities\AmazonFulfillmentOrderItem;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonLedgerDetail;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Modules\Amazon\Managers\AmazonLedgerManager;
use Modules\Amazon\Tests\Feature\Managers\Helpers\AmazonMockRequests;
use Modules\Amazon\Tests\Seeders\SeedLedgers;
use Modules\Amazon\Tests\Seeders\SeedLedgerSummaries;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonLedgerManagerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;
    use WithFaker;

    private InventoryMovement $seedingMovement;

    /**
     * @throws DropshipWithOpenOrdersException
     */
    protected function setUp(): void
    {
        parent::setUp();

        Queue::fake();

        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()->has(SalesChannel::factory())->create();
        (new InitializeFbaWarehouse($this->amazonIntegrationInstance))->handle();
        $this->amazonIntegrationInstance = $this->amazonIntegrationInstance->refresh();
    }

    /**
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws NotOpenWarehouseTransferException
     */
    public function testLedgers(): void
    {
        $this->seedLedgersForCaseWithUnknownEventsAndInTransitBetweenWarehousesAdjustment();
        $fnskuProduct = AmazonFnskuProduct::where('fnsku', 'B00KBTQ7D4')->first();
        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers(null, $fnskuProduct);
        $this->assertTrue(true, 'No exception was thrown');
//        dd(
//            InventoryMovement::query()
//                ->where('inventory_status', 'active')
//                ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
//                ->where('id', '!=', $this->seedingMovement->id)
//                ->sum('quantity'),
//            InventoryMovement::query()
//                ->where('inventory_status', 'in_transit')
//                ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
//                ->where('id', '!=', $this->seedingMovement->id)
//                ->sum('quantity'),
//            AmazonFbaReportInventoryLedgerSummary::query()
//                ->where('date', '01/31/2024')
//                ->first()
//                ->toArray()
//        );
    }

    /**
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws NotOpenWarehouseTransferException
     */
    public function test_it_can_unreconcile_ledgers(): void
    {
        // Need two fnsku products with ledgers to reconcile, then test unreconcile

        $fnskuProduct1 = AmazonFnskuProduct::factory()
            ->hasProduct()
            ->create([
                'integration_instance_id' => $this->amazonIntegrationInstance->id,
            ]);
        $product1 = $fnskuProduct1->product;

        $fnskuProduct2 = AmazonFnskuProduct::factory()
            ->hasProduct()
            ->create([
                'integration_instance_id' => $this->amazonIntegrationInstance->id,
            ]);
        $product2 = $fnskuProduct2->product;

        $ledgers1 = AmazonFbaReportInventoryLedger::factory()
            ->count(3)
            ->withJsonObjectOverrides([
                'fnsku' => $fnskuProduct1->fnsku,
                'country' => $fnskuProduct1->location,
                'disposition' => $fnskuProduct1->disposition,
                'event_type' => FbaInventoryLedgerReportEventTypeEnum::Adjustments,
                'quantity' => 1,
            ])
            ->create([
                'integration_instance_id' => $this->amazonIntegrationInstance->id,
            ])->each->refresh();

        $date = now();

        $ledgers1->each(function (AmazonFbaReportInventoryLedger $ledger) use ($fnskuProduct1, $date) {
            $newDate = $date->addDay()->toIso8601String();
            $ledger->json_object = array_merge($ledger->json_object, [
                'date' => $newDate,
            ]);
            $ledger->event_datetime = $newDate;
            $ledger->save();
            $ledger->refresh();
        });

        $ledgers2 = AmazonFbaReportInventoryLedger::factory()
            ->count(3)
            ->withJsonObjectOverrides([
                'fnsku' => $fnskuProduct2->fnsku,
                'country' => $fnskuProduct2->location,
                'disposition' => $fnskuProduct2->disposition,
                'event_type' => FbaInventoryLedgerReportEventTypeEnum::Adjustments,
                'quantity' => 1,
            ])
            ->create([
                'integration_instance_id' => $this->amazonIntegrationInstance->id,
            ])->each->refresh();

        $date = now()->addHour();

        $ledgers2->each(function (AmazonFbaReportInventoryLedger $ledger) use ($fnskuProduct1, $date) {
            $newDate = $date->addDay()->toIso8601String();
            $ledger->json_object = array_merge($ledger->json_object, [
                'date' => $newDate,
            ]);
            $ledger->event_datetime = $newDate;
            $ledger->save();
            $ledger->refresh();
        });

        $manager = new AmazonLedgerManager($this->amazonIntegrationInstance);
        $manager->reconcileLedgers();

        $this->assertCount(6, AmazonFbaReportInventoryLedger::where('sku_link_type', InventoryAdjustment::class)->get());

        $manager->unreconcileLedgers(null, $ledgers1->pluck('id')->toArray());

        $this->assertCount(3, AmazonFbaReportInventoryLedger::where('sku_link_type', InventoryAdjustment::class)->get());
    }

    /**
     * @throws Throwable
     */
    private function seedLedgersForCaseWithUnknownEventsAndInTransitBetweenWarehousesAdjustment(): void
    {
        $product = Product::factory()->create([
            'sku' => 'TM-431'
        ]);

        $product->setInitialInventory($this->amazonIntegrationInstance->warehouse->id, 1000, 5.00);

        $this->seedingMovement = InventoryMovement::latest()->first();

        $fnskuProduct = AmazonFnskuProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => 'B00KBTQ7D4',
            'disposition' => 'SELLABLE',
            'location' => 'US',
            'product_id' => $product->id,
        ]);

        $initialInventory = AmazonFbaInitialInventory::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);
        $initialInventory->json_object = array_merge($initialInventory->json_object, [
            'asin' => $fnskuProduct->fnsku,
            'msku' => '1727569-FBA',
            'fnsku' => $fnskuProduct->fnsku,
            'disposition' => $fnskuProduct->disposition,
            'location' => $fnskuProduct->location,
            'starting_warehouse_balance' => 0,
            'in_transit_between_warehouses' => 23,
        ]);
        $initialInventory->save();
        $initialInventory->refresh();

        SeedLedgers::seed();
        SeedLedgerSummaries::seed();
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_reconcile_ledger_for_non_amazon_order_shipments(): void
    {
        $fnskuProduct = AmazonFnskuProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);

        // need sales order and fulfillment

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $fnskuProduct->product_id,
            ])
            ->create([
            'sales_channel_id' => SalesChannel::first()->id,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $salesOrderFulfillment = SalesOrderFulfillment::factory()
            ->hasSalesOrderFulfillmentLines(1, [
                'sales_order_line_id' => $salesOrderLine->id,
            ])
            ->create([
                'sales_order_id' => $salesOrder->id,
            ]);

        $salesOrderFulfillmentLine = $salesOrderFulfillment->salesOrderFulfillmentLines->first();

        $amazonFulfillmentOrder = AmazonFulfillmentOrder::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'sales_order_fulfillment_id' => $salesOrderFulfillment->id,
        ]);

        $jsonObject = $amazonFulfillmentOrder->json_object;
        $jsonObject['sellerFulfillmentOrderId'] = $salesOrderFulfillment->fulfillment_number;
        $amazonFulfillmentOrder->json_object = $jsonObject;
        $amazonFulfillmentOrder->save();

        $amazonFulfillmentOrderItem = AmazonFulfillmentOrderItem::factory()->create([
            'amazon_fulfillment_order_id' => $amazonFulfillmentOrder->id,
        ]);

        $jsonObject = $amazonFulfillmentOrderItem->json_object;
        $jsonObject['fulfillmentNetworkSku'] = $fnskuProduct->fnsku;
        $amazonFulfillmentOrderItem->json_object = $jsonObject;
        $amazonFulfillmentOrderItem->save();

        $amazonOrder = AmazonOrder::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);

        $jsonObject = $amazonOrder->json_object;
        $jsonObject['SalesChannel'] = 'Non-Amazon US';
        $amazonOrder->json_object = $jsonObject;
        $amazonOrder->save();
        $amazonOrder->refresh();

        $amazonLedger = AmazonFbaReportInventoryLedger::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);

        $jsonObject = $amazonLedger->json_object;
        $jsonObject['event_type'] = FbaInventoryLedgerReportEventTypeEnum::Shipments;
        $jsonObject['fnsku'] = $fnskuProduct->fnsku;
        $jsonObject['country'] = $fnskuProduct->location;
        $jsonObject['disposition'] = $fnskuProduct->disposition;
        $amazonLedger->json_object = $jsonObject;
        $amazonLedger->save();
        $amazonLedger->refresh();

        $amazonFbaReportShipment = AmazonFbaReportShipment::factory()->create();

        $jsonObject = $amazonFbaReportShipment->json_object;
        $jsonObject['amazon_order_id'] = $amazonOrder->AmazonOrderId;
        $jsonObject['sales_channel'] = 'Non-Amazon';
        $jsonObject['merchant_order_id'] = $salesOrderFulfillment->fulfillment_number;
        $amazonFbaReportShipment->json_object = $jsonObject;
        $amazonFbaReportShipment->save();
        $amazonFbaReportShipment->refresh();

        AmazonLedgerDetail::factory()->create([
            'amazon_fba_report_inventory_ledger_id' => $amazonLedger->id,
            'detail_id' => $amazonFbaReportShipment->id,
            'detail_type' => AmazonFbaReportShipment::class,
        ]);

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcile($amazonLedger);

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $amazonLedger->id,
            'event_type' => FbaInventoryLedgerReportEventTypeEnum::Shipments,
            'fnsku' => $fnskuProduct->fnsku,
            'country' => $fnskuProduct->location,
            'disposition' => $fnskuProduct->disposition,
            'sku_link_id' => $salesOrderFulfillmentLine->id,
            'sku_link_type' => SalesOrderFulfillmentLine::class,
        ]);
    }
}
