<?php

namespace Modules\Amazon\Tests\Feature\Managers;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\PurchaseOrder;
use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferShipmentReceiptLine;
use Carbon\Carbon;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonFbaInboundShipmentItem;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFbaInboundShipFromMapping;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;
use Modules\Amazon\Enums\Entities\FbaInboundShipmentStatusEnum;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Jobs\CreateAmazonRefreshFbaInboundShipmentItemsJobs;
use Modules\Amazon\Jobs\RefreshAmazonFbaInboundShipmentItemsJob;
use Modules\Amazon\Managers\AmazonInboundManager;
use Modules\Amazon\Managers\AmazonLedgerManager;
use Modules\Amazon\Managers\AmazonReportManager;
use Modules\Amazon\Repositories\AmazonFbaInboundShipmentRepository;
use Modules\Amazon\Tests\AmazonMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonInboundManagerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;
    use WithFaker;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();

        Queue::fake();

        $this->mockGetAccessToken();

        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()->hasSalesChannel()->create();
        (new InitializeFbaWarehouse($this->amazonIntegrationInstance))->handle();
        $this->amazonIntegrationInstance->refresh();
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_create_warehouse_transfer_from_inbound(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        /** @var Warehouse $sourceWarehouse */
        $sourceWarehouse = Warehouse::query()->first();

        /** @var AmazonProduct $amazonProduct */
        $amazonProduct = AmazonProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'json_object' => [
                'seller_sku' => 'SKU1',
                'fulfillment_channel' => AmazonProductFulfillmentChannelEnum::AMAZON_NA,
            ],
        ])
            ->refresh();

        /** @var Product $product */
        $product = Product::factory()->create([
            'sku' => 'SKU1',
        ]);

        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product->id,
            'adjustment_date' => Carbon::now()->format('Y-m-d'),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 2,
            'warehouse_id' => $sourceWarehouse->id,
            'quantity' => 5,
            'reason' => 'Test',
        ])->assertOk();

        ProductListing::factory()->create([
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'listing_sku' => 'SKU1',
            'document_id' => $amazonProduct->id,
            'document_type' => AmazonProduct::class,
            'product_id' => $product->id,
        ]);

        $amazonInboundShipment = AmazonFbaInboundShipment::factory()->has(
            AmazonFbaInboundShipmentItem::factory(1, [
                'json_object' => [
                    'SellerSKU' => 'SKU1',
                    'FulfillmentNetworkSKU' => 'FNSKU1',
                    'QuantityShipped' => 3,
                ],
            ]))
            ->create([
                'integration_instance_id' => $this->amazonIntegrationInstance->id,
                'json_object' => [
                    'ShipmentId' => 'FBA16TDCRLV9',
                    'ShipmentStatus' => 'WORKING',
                    'ShipFromAddress' => [
                        'Name' => $sourceWarehouse->name,
                    ]
                ],
            ])
            ->refresh();

        AmazonFbaInboundShipFromMapping::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'name' => $sourceWarehouse->name,
            'link_type' => Warehouse::class,
            'link_id' => $sourceWarehouse->id,
        ]);

        /** @var AmazonFbaInboundShipmentItem $amazonFbaInboundShipmentItem */
        $amazonFbaInboundShipmentItem = AmazonFbaInboundShipmentItem::query()->first();

        AmazonFnskuProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => $amazonFbaInboundShipmentItem->FulfillmentNetworkSKU,
            'product_id' => $product->id,
            'location' => 'US',
            'disposition' => 'SELLABLE',
        ]);

        $response = $this->getJson('/api/amazon/'.$this->amazonIntegrationInstance->id.'/inbound/shipments/unlinked');

        $response->assertOk()
            ->assertJsonStructure([
                'data' => [
                    '*' => [
                        'id',
                        'ShipmentId',
                    ],
                ],
            ]);

        $amazonFbaInboundShipmentId = $response->json('data.0.id');

        (new AmazonInboundManager($this->amazonIntegrationInstance))->process([$amazonFbaInboundShipmentId]);

        $this->assertEquals(
            1,
            WarehouseTransfer::query()
                ->where('from_warehouse_id', $sourceWarehouse->id)
                ->where('to_warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
                ->where('warehouse_transfer_number', 'FBA16TDCRLV9')
                ->where('transfer_status', WarehouseTransfer::TRANSFER_STATUS_OPEN)
                ->count()
        );

        $this->assertEquals(
            1,
            AmazonFbaInboundShipment::query()
                ->whereHas('skuLink', function ($query) {
                    $query->where('warehouse_transfer_number', 'FBA16TDCRLV9');
                })
                ->count()
        );

        $this->assertEquals(
            1,
            InventoryMovement::query()
                ->where('warehouse_id', $sourceWarehouse->id)
                ->where('quantity', -3)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->count()
        );

        $this->assertEquals(
            0,
            InventoryMovement::query()
                ->where('warehouse_id', $sourceWarehouse->id)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                ->sum('quantity')
        );

        $this->assertEquals(
            1,
            InventoryMovement::query()
                ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
                ->where('quantity', 3)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
                ->count()
        );

        /*
         * 1. Mock data from ReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER() report.  Make sure it creates an
         * AmazonFbaReportInventoryLedger record with event_type="Receipts".  There is no detail report needed for
         * "Receipts" - DONE
         * 2. CreateAuditTrailFromAmazonLedger should be called upon "processing" the ledger report.  This will create
         * WarehouseTransferShipmentReceipts for the Warehouse Transfer created above.  It will also perform the
         * following inventory movements:
         * - subtract qty from destination warehouse "In Transit"
         * - add qty to destination warehouse "Available"
         * It should do so using TransferProcessor->receiveShipment() method.
         * 3. The skuLink relationship should be created between the AmazonFbaReportInventoryLedger and the WarehouseTransferShipmentReceiptLine - DONE
         */
        $amazonReportForLedger = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER,
        ]);

        $fnsku = $this->faker->word();

        $date = Carbon::parse($this->faker->date());
        AmazonFbaReportInventoryLedger::factory()->create([
            'amazon_report_id' => $amazonReportForLedger->id,
            'event_datetime' => $date->copy(),
            'json_object' => [
                'date' => $date->copy()->format('m/d/Y'),
                'msku' => $amazonProduct->seller_sku,
                'fnsku' => $fnsku,
                'event_type' => FbaInventoryLedgerReportEventTypeEnum::Receipts(),
                'quantity' => 3,
                'fulfillment_center' => $this->faker->word(),
                'reference_id' => $amazonInboundShipment->ShipmentId,
                'country' => 'US',
                'disposition' => 'SELLABLE',
                'checksum' => $this->faker->randomElements(),
            ],
        ])->refresh();

        // For audit trail to be possible, we need to initiate the inventory.

        AmazonFnskuProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => $fnsku,
            'product_id' => $product->id,
            'location' => 'US',
            'disposition' => 'SELLABLE',
        ]);

        (new AmazonLedgerManager($this->amazonIntegrationInstance))->reconcileLedgers($amazonReportForLedger);
        $this->assertEquals(
            1,
            InventoryMovement::query()
                ->where('type', InventoryMovement::TYPE_TRANSFER)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
                ->where('quantity', -3)
                ->count(), 'Subtract qty from destination warehouse "In Transit"');

        $this->assertEquals(
            1,
            InventoryMovement::query()
                ->where('type', InventoryMovement::TYPE_TRANSFER)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('quantity', 3)
                ->count(), 'Subtract qty from destination warehouse "In Transit"');

        $this->assertEquals(
            1,
            AmazonFbaReportInventoryLedger::query()
                ->where('sku_link_type', WarehouseTransferShipmentReceiptLine::class)
                ->count(), 'The skuLink relationship should be created between the AmazonFbaReportInventoryLedger and the WarehouseTransferShipmentReceiptLine');

        WarehouseTransfer::query()->first()->delete();

        $this->assertEquals(
            1,
            AmazonFbaInboundShipment::query()
                ->whereNull('sku_link_id')
                ->whereNull('sku_link_type')
                ->count(), 'AmazonFbaInboundShipment should clear the relationship when the warehouse transfer is deleted');
    }

    /**
     * @throws Exception
     */
    public function test_it_can_delete_sku_link_for_cancelled_inbound()
    {
        $inbound = AmazonFbaInboundShipment::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);

        $purchaseOrder = PurchaseOrder::factory()->create([
            'purchase_order_number' => $inbound->ShipmentId,
        ]);

        $jsonObject = $inbound->json_object;
        $jsonObject['ShipmentStatus'] = FbaInboundShipmentStatusEnum::CANCELLED;
        $inbound->update([
            'sku_link_id' => $purchaseOrder->id,
            'sku_link_type' => PurchaseOrder::class,
            'json_object' => $jsonObject
        ]);

        $manager = new AmazonInboundManager($this->amazonIntegrationInstance);
        $manager->processCancellations();

        $this->assertDatabaseEmpty((new PurchaseOrder())->getTable());
    }

    /*
     * TODO: Pending tests
     *  Test that when a ledger report comes in with eventType = "Receipts" that the warehouse transfer created gets
     *  received and the expected inventory movements occur and the relationship between the Ledger and the Warehouse
     *  Transfer Shipment Receipt Line exists (via sku_link in the ledgers table).
     */
}
