<?php

namespace Modules\Amazon\Tests\Feature\Managers;

use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\FinancialLine;
use App\Models\NominalCode;
use App\Models\PaymentType;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Services\Accounting\AccountingTransactionManager;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Queue;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetFinancialEventsAdt;
use Modules\Amazon\Entities\AmazonFinancialAdjustmentEvent;
use Modules\Amazon\Entities\AmazonFinancialEventGroup;
use Modules\Amazon\Entities\AmazonFinancialEventTypeNominalCodeMapping;
use Modules\Amazon\Entities\AmazonFinancialRefundEvent;
use Modules\Amazon\Entities\AmazonFinancialShipmentEvent;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonOrderItem;
use Modules\Amazon\Entities\AmazonReportSettlementData;
use Modules\Amazon\Entities\AmazonReportSettlementTypeMapping;
use Modules\Amazon\Enums\Entities\AmazonFinancialEventGroupAccountingStatusEnum;
use Modules\Amazon\Enums\Entities\AmazonFinancialEventGroupProcessingStatusEnum;
use Modules\Amazon\Enums\Entities\OrderStatusEnum;
use Modules\Amazon\Managers\AmazonFinanceManager;
use Modules\Amazon\Managers\AmazonOrderManager;
use Modules\Amazon\Repositories\AmazonFinancialEventGroupRepository;
use Modules\Amazon\Tests\Feature\Managers\Helpers\AmazonMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonFinanceManagerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;
    use WithFaker;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();

        Event::fake();
        Queue::fake();
        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()->hasSalesChannel()->create();
        (new InitializeFbaWarehouse($this->amazonIntegrationInstance))->handle();
        $this->amazonIntegrationInstance->refresh();
    }

    /**
     * @throws Exception
     */
    public function test_it_can_get_financial_event_groups(): void
    {
        $this->mockRefreshFinancialEventGroups();

        $manager = new AmazonFinanceManager($this->amazonIntegrationInstance);

        $manager->refreshFinancialEventGroups();

        $this->assertDatabaseHas((new AmazonFinancialEventGroup())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_get_financial_event_groups_needing_accounting_transaction(): void
    {
        /** @var AmazonIntegrationInstance $amazonIntegrationInstance */
        $amazonIntegrationInstance = AmazonIntegrationInstance::factory([
            'name' => 'AmazonWithBatch',
            'integration_settings' => [
                'batch_sales_order_invoices' => true,
            ],
        ])->hasSalesChannel()->create();

        /** @var SalesChannel $salesChannel */
        $salesChannel = $amazonIntegrationInstance->salesChannel;

        /** @var AmazonOrder $amazonOrder */
        $amazonOrder = AmazonOrder::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
        ])->create()->refresh();

        /** @var AmazonOrder $amazonOrder2 */
        $amazonOrder2 = AmazonOrder::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
        ])->create()->refresh();

        $salesOrder = SalesOrder::factory([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_order_id' => $amazonOrder->id,
            'sales_channel_order_type' => AmazonOrder::class,
        ])->create();

        SalesOrder::factory([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_order_id' => $amazonOrder2->id,
            'sales_channel_order_type' => AmazonOrder::class,
        ])->create();

        app(AccountingTransactionManager::class)->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withSalesOrders($salesOrder->id)
                ->where('type', AccountingTransaction::TYPE_SALES_ORDER_INVOICE)
                ->where('is_batchable', true)
                ->count()
        );

        /** @var AmazonFinancialEventGroup $amazonFinancialEventGroup */
        $amazonFinancialEventGroup = AmazonFinancialEventGroup::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'json_object' => [
                'ProcessingStatus' => AmazonFinancialEventGroupProcessingStatusEnum::Closed,
            ],
        ])->create()->refresh();

        $test = AmazonReportSettlementData::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'amazon_financial_event_group_id' => $amazonFinancialEventGroup->id,
            'json_object' => [
                'order_id' => $amazonOrder->AmazonOrderId,
                'transaction_type' => 'Order',
                'amount_type' => 'ItemPrice',
                'amount_description' => 'Principal',
            ],
        ])->create();

        AmazonReportSettlementData::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'amazon_financial_event_group_id' => $amazonFinancialEventGroup->id,
            'json_object' => [
                'order_id' => $amazonOrder2->AmazonOrderId,
                'transaction_type' => 'Order',
                'amount_type' => 'ItemPrice',
                'amount_description' => 'Principal',
            ],
        ])->create();

        AmazonReportSettlementTypeMapping::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'transaction_type' => 'Order',
            'amount_type' => 'ItemPrice',
            'amount_description' => 'Principal',
            'nominal_code_id' => NominalCode::query()->where('name', 'Sales')->first()->id,
        ])->create();

        $this->assertEquals(1, app(AmazonFinancialEventGroupRepository::class)->getFinancialEventGroupsNeedingAccountingTransactions($amazonIntegrationInstance, [])->count());
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_create_accounting_transaction_for_financial_event_group(): void
    {
        /** @var AmazonIntegrationInstance $amazonIntegrationInstance */
        $amazonIntegrationInstance = AmazonIntegrationInstance::factory([
            'name' => 'AmazonWithBatch',
            'integration_settings' => [
                'batch_sales_order_invoices' => true,
            ],
        ])->hasSalesChannel()->create();

        /** @var SalesChannel $salesChannel */
        $salesChannel = $amazonIntegrationInstance->salesChannel;

        /** @var AmazonOrder $amazonOrder */
        $amazonOrder = AmazonOrder::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
        ])->create()->refresh();

        /** @var AmazonOrder $amazonOrder2 */
        $amazonOrder2 = AmazonOrder::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
        ])->create()->refresh();

        $salesOrder = SalesOrder::factory([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_order_id' => $amazonOrder->id,
            'sales_channel_order_type' => AmazonOrder::class,
        ])->create();

        SalesOrder::factory([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_order_id' => $amazonOrder2->id,
            'sales_channel_order_type' => AmazonOrder::class,
        ])->create();

        app(AccountingTransactionManager::class)->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withSalesOrders($salesOrder->id)
                ->where('type', AccountingTransaction::TYPE_SALES_ORDER_INVOICE)
                ->where('is_batchable', true)
                ->count()
        );

        /** @var AmazonFinancialEventGroup $amazonFinancialEventGroup */
        $amazonFinancialEventGroup = AmazonFinancialEventGroup::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'json_object' => [
                'FinancialEventGroupId' => $this->faker->uuid(),
                'ProcessingStatus' => AmazonFinancialEventGroupProcessingStatusEnum::Closed,
                'OriginalTotal' => [
                    'CurrencyAmount' => 72.50,
                    'CurrencyCode' => 'CAD',
                ],
                'ConvertedTotal' => [
                    'CurrencyAmount' => 50.00,
                    'CurrencyCode' => 'USD',
                ],
            ],
        ])->create()->refresh();

        AmazonReportSettlementData::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'amazon_financial_event_group_id' => $amazonFinancialEventGroup->id,
            'json_object' => [
                'order_id' => $amazonOrder->AmazonOrderId,
                'transaction_type' => 'Order',
                'amount_type' => 'ItemPrice',
                'amount_description' => 'Principal',
                'amount' => 25.00,
            ],
        ])->create();

        AmazonReportSettlementData::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'amazon_financial_event_group_id' => $amazonFinancialEventGroup->id,
            'json_object' => [
                'order_id' => $amazonOrder->AmazonOrderId,
                'transaction_type' => 'Order',
                'amount_type' => 'ItemFees',
                'amount_description' => 'Commission',
                'amount' => -2.50,
            ],
        ])->create();

        AmazonReportSettlementData::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'amazon_financial_event_group_id' => $amazonFinancialEventGroup->id,
            'json_object' => [
                'order_id' => $amazonOrder2->AmazonOrderId,
                'transaction_type' => 'Order',
                'amount_type' => 'ItemPrice',
                'amount_description' => 'Principal',
                'amount' => 50.00,
            ],
        ])->create();

        AmazonReportSettlementData::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'amazon_financial_event_group_id' => $amazonFinancialEventGroup->id,
            'json_object' => [
                'order_id' => $amazonOrder->AmazonOrderId,
                'transaction_type' => 'Order',
                'amount_type' => 'ItemFees',
                'amount_description' => 'Commission',
                'amount' => -5.00,
            ],
        ])->create();

        AmazonReportSettlementData::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'amazon_financial_event_group_id' => $amazonFinancialEventGroup->id,
            'json_object' => [
                'transaction_type' => 'other-transaction',
                'amount_type' => 'FBA Inventory Reimbursement',
                'amount_description' => 'WAREHOUSE_DAMAGE',
                'amount' => 5.00,
            ],
        ])->create();

        AmazonReportSettlementTypeMapping::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'transaction_type' => 'Order',
            'amount_type' => 'ItemPrice',
            'amount_description' => 'Principal',
            'nominal_code_id' => NominalCode::query()->where('name', 'Sales')->first()->id,
        ])->create();

        AmazonReportSettlementTypeMapping::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'transaction_type' => 'Order',
            'amount_type' => 'ItemFees',
            'amount_description' => 'Commission',
            'nominal_code_id' => NominalCode::query()->where('name', 'Cost of Goods Sold')->first()->id,
        ])->create();

        $mappingToDelete = AmazonReportSettlementTypeMapping::factory([
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'transaction_type' => 'other-transaction',
            'amount_type' => 'FBA Inventory Reimbursement',
            'amount_description' => 'WAREHOUSE_DAMAGE',
            'nominal_code_id' => NominalCode::query()->where('name', 'Cost of Goods Sold')->first()->id,
        ])->create();

        (new AmazonFinanceManager($amazonIntegrationInstance))->createAccountingTransactions();

        $this->assertEquals(
            1,
            AccountingTransaction::query()
                ->where('type', AccountingTransaction::TYPE_BATCH_SALES_ORDER_INVOICE)
                ->where('link_id', $amazonFinancialEventGroup->id)
                ->where('link_type', AmazonFinancialEventGroup::class)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::query()
                ->where('link_type', 'summarized')
                ->where('description', 'Order: ItemFees - Commission')
                ->where('nominal_code_id', NominalCode::query()->where('name', 'Cost of Goods Sold')->first()->id)
                ->where('amount', -7.50)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::query()
                ->where('link_type', 'summarized')
                ->where('description', 'Order: ItemPrice - Principal')
                ->where('nominal_code_id', NominalCode::query()->where('name', 'Sales')->first()->id)
                ->where('amount', 75)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::query()
                ->where('link_type', 'summarized')
                ->where('description', 'other-transaction: FBA Inventory Reimbursement - WAREHOUSE_DAMAGE')
                ->where('nominal_code_id', NominalCode::query()->where('name', 'Cost of Goods Sold')->first()->id)
                ->where('amount', 5)
                ->count()
        );

        $batchAccountingTransaction = AccountingTransaction::query()->where('type', AccountingTransaction::TYPE_BATCH_SALES_ORDER_INVOICE)->first();

        $this->assertEquals(
            2,
            AccountingTransaction::query()
                ->where('type', AccountingTransaction::TYPE_SALES_ORDER_INVOICE)
                ->where('parent_id', $batchAccountingTransaction->id)
                ->count()
        );

        // Testing the account statuses for failing to pass conditions

        $batchAccountingTransaction->delete();

        $this->assertEquals(
            0,
            AccountingTransaction::query()
                ->where('type', AccountingTransaction::TYPE_SALES_ORDER_INVOICE)
                ->where('parent_id', $batchAccountingTransaction->id)
                ->count()
        );

        AccountingTransaction::query()->where('type', AccountingTransaction::TYPE_SALES_ORDER_INVOICE)->delete();

        (new AmazonFinanceManager($amazonIntegrationInstance))->createAccountingTransactions();

        $this->assertEquals(
            0,
            AccountingTransaction::query()
                ->where('type', AccountingTransaction::TYPE_BATCH_SALES_ORDER_INVOICE)
                ->where('link_id', $amazonFinancialEventGroup->id)
                ->where('link_type', AmazonFinancialEventGroup::class)
                ->count()
        );

        $this->assertEquals(
            1,
            AmazonFinancialEventGroup::query()
                ->where('accounting_status', AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_SALES_ORDER_INVOICE)
                ->count()
        );

        $salesOrder->delete();

        (new AmazonFinanceManager($amazonIntegrationInstance))->createAccountingTransactions();

        $this->assertEquals(
            0,
            AccountingTransaction::query()
                ->where('type', AccountingTransaction::TYPE_BATCH_SALES_ORDER_INVOICE)
                ->where('link_id', $amazonFinancialEventGroup->id)
                ->where('link_type', AmazonFinancialEventGroup::class)
                ->count()
        );

        $this->assertEquals(
            1,
            AmazonFinancialEventGroup::query()
                ->where('accounting_status', AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_SALES_ORDER)
                ->count()
        );

        $amazonOrder->delete();

        (new AmazonFinanceManager($amazonIntegrationInstance))->createAccountingTransactions();

        $this->assertEquals(
            0,
            AccountingTransaction::query()
                ->where('type', AccountingTransaction::TYPE_BATCH_SALES_ORDER_INVOICE)
                ->where('link_id', $amazonFinancialEventGroup->id)
                ->where('link_type', AmazonFinancialEventGroup::class)
                ->count()
        );

        $this->assertEquals(
            1,
            AmazonFinancialEventGroup::query()
                ->where('accounting_status', AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_AMAZON_ORDER)
                ->count()
        );

        $mappingToDelete->delete();

        (new AmazonFinanceManager($amazonIntegrationInstance))->createAccountingTransactions();

        $this->assertEquals(
            0,
            AccountingTransaction::query()
                ->where('type', AccountingTransaction::TYPE_BATCH_SALES_ORDER_INVOICE)
                ->where('link_id', $amazonFinancialEventGroup->id)
                ->where('link_type', AmazonFinancialEventGroup::class)
                ->count()
        );

        $this->assertEquals(
            1,
            AmazonFinancialEventGroup::query()
                ->where('accounting_status', AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_MAPPINGS)
                ->count()
        );

        AmazonReportSettlementData::query()->delete();

        (new AmazonFinanceManager($amazonIntegrationInstance))->createAccountingTransactions();

        $this->assertEquals(
            0,
            AccountingTransaction::query()
                ->where('type', AccountingTransaction::TYPE_BATCH_SALES_ORDER_INVOICE)
                ->where('link_id', $amazonFinancialEventGroup->id)
                ->where('link_type', AmazonFinancialEventGroup::class)
                ->count()
        );

        $this->assertEquals(
            1,
            AmazonFinancialEventGroup::query()
                ->where('accounting_status', AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_SETTLEMENT_DATA)
                ->count()
        );
    }

    public function test_it_can_refresh_amazon_financial_events()
    {
        $nominalCode = NominalCode::query()->first();

        $this->mockRefreshFinancialEvents();

        /** @var AmazonIntegrationInstance $amazonIntegrationInstance */
        $amazonIntegrationInstance = AmazonIntegrationInstance::factory([
            'name' => 'AmazonWithBatch',
            'integration_settings' => [
                'batch_sales_order_invoices' => true,
                'start_date' => now()->subYears(10),
            ],
        ])->hasSalesChannel()->create();

        foreach ([
            'Amazon RefundCommission',
            'Amazon Commission',
            'Amazon FBAPerUnitFulfillmentFee',
        ] as $name) {
            AmazonFinancialEventTypeNominalCodeMapping::factory()
                ->for($amazonIntegrationInstance)
                ->for($nominalCode)
                ->create([
                    'name' => $name,
                ]);
        }

        PaymentType::factory()->create([
            'name' => 'Amazon Payment',
        ]);

        //Create amazon order for testing
        $amazonOrder = AmazonOrder::factory()->make([
            'integration_instance_id' => $amazonIntegrationInstance->id,
        ]);
        $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
            'AmazonOrderId' => '112-6644981-7758668',
            'OrderStatus' => OrderStatusEnum::STATUS_SHIPPED->value,
        ]);
        $amazonOrder->save();

        //Create amazon order line
        $amazonOrderLine = AmazonOrderItem::factory()->make([
            'amazon_order_id' => $amazonOrder->id,
        ]);
        $amazonOrderLine->json_object = array_merge($amazonOrderLine->json_object, [
            'OrderItemId' => '98357958304641',
        ]);
        $amazonOrderLine->save();

        //Create Amazon Orders
        (new AmazonOrderManager($amazonIntegrationInstance))->createSkuOrders([], true);

        //Get Financial lines
        $amazonFinanceManager = (new AmazonFinanceManager($amazonIntegrationInstance));
        $amazonFinanceManager->refreshFinancialEvents(new AmazonGetFinancialEventsAdt());
        $amazonFinanceManager->createFinancialLines();

        $this->assertEquals(1, AmazonFinancialShipmentEvent::query()->count());
        $this->assertEquals(1, AmazonFinancialRefundEvent::query()->count());
        $this->assertDatabaseCount(AmazonFinancialAdjustmentEvent::class, 2);

        AmazonFinancialShipmentEvent::all()->each(function (AmazonFinancialShipmentEvent $amazonFinancialShipmentEvent) use ($amazonOrder, $nominalCode) {
            $this->assertDatabaseHas((new FinancialLine())->getTable(), [
                'sales_order_id' => $amazonOrder->salesOrder->id,
                'external_source_type' => AmazonFinancialShipmentEvent::class,
                'external_source_id' => $amazonFinancialShipmentEvent->id,
                'nominal_code_id' => $nominalCode->id,
            ]);
        });

        AmazonFinancialRefundEvent::all()->each(function (AmazonFinancialRefundEvent $amazonFinancialRefundEvent) use ($amazonOrder, $nominalCode) {
            $this->assertDatabaseHas((new FinancialLine())->getTable(), [
                'sales_order_id' => $amazonOrder->salesOrder->id,
                'external_source_type' => AmazonFinancialRefundEvent::class,
                'external_source_id' => $amazonFinancialRefundEvent->id,
                'nominal_code_id' => $nominalCode->id,
            ]);
        });
    }
}
