<?php

namespace Modules\Amazon\Tests\Feature\Controllers;

use App\Data\CreateSkuProductsFromSalesChannelData;
use App\Data\SalesChannelProductImportMappingData;
use App\Data\SalesChannelProductToSkuProductMappingCollectionData;
use App\Data\SalesChannelProductToSkuProductMappingData;
use App\Jobs\ImportSalesChannelProductMappingsJob;
use App\Jobs\MapSalesOrderLinesToSalesChannelProductsJob;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\User;
use App\Models\Warehouse;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Storage;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonOrderItem;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Entities\AmazonReportRequest;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Jobs\AmazonInitializeSkuProductsJob;
use Modules\Amazon\Jobs\CreateAmazonFnSkusToSkuProductsJob;
use Modules\Amazon\Jobs\CreateAmazonSearchCatalogItemsJobsJob;
use Modules\Amazon\Jobs\CreateAuditTrailFromAmazonLedgerJob;
use Modules\Amazon\Jobs\ImportProductMapping;
use Modules\Amazon\Jobs\SearchAmazonCatalogItemsJob;
use Modules\Amazon\Managers\AmazonFnskuProductManager;
use Modules\Amazon\Managers\AmazonProductManager;
use Modules\Amazon\Managers\AmazonReportManager;
use Modules\Amazon\Repositories\AmazonProductRepository;
use Modules\Amazon\Tests\AmazonMockRequests;
use Modules\Amazon\Tests\AmazonTestingData;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use SellingPartnerApi\ApiException;
use Tests\TestCase;
use Throwable;

class AmazonProductControllerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->has(Warehouse::factory(1, ['type' => Warehouse::TYPE_AMAZON_FBA]))
            ->create();

        $this->mockGetAccessToken();
        $this->mockGetRestrictedDataToken();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws GuzzleException
     * @throws ApiException
     * @throws Throwable
     */
    public function test_amazon_products_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Refresh products
        |--------------------------------------------------------------------------
        */

        $this->mockCreateReport();
        $this->mockGetReport();

        $this->postJson(route('amazon.products.refresh', $this->amazonIntegrationInstance->id))->assertOk();

        $this->assertDatabaseEmpty((new AmazonReportRequest())->getTable());

        $this->assertDatabaseHas((new AmazonReport())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'reportType' => AmazonReportTypeEnum::PRODUCTS,
        ]);

        $amazonReport = AmazonReport::where('marketplaceIds', json_encode(['ATVPDKIKX0DER']))->first();

        $this->mockGetReportDocument();
        $this->mockGetDocument();

        (new AmazonReportManager($this->amazonIntegrationInstance))->sync($amazonReport);

        // Not doing an assertDatabaseHas() here because of slight differences in json encoding
        $this->assertEquals(AmazonTestingData::getProducts()[0], AmazonProduct::first()->json_object);

        $this->amazonAssertionsForCatalogProducts();

        /*
        |--------------------------------------------------------------------------
        | Get products
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('amazon.products.index', $this->amazonIntegrationInstance->id), [
            'sortObjs' => [
                'column' => 'id',
                'ascending' => true,
            ],
        ])->assertOk();

        $response->assertJsonStructure([
            'data' => [
                '*' => [
                    'seller_sku',
                    'item_name',
                    'asin1',
                    'product',
                    'price',
                    'quantity',
                    'brand',
                    'product_type',
                    'sales_rank',
                    'style',
                    'created_at',
                ],
            ],
        ]);

        $amazonProducts = $response->json('data');


        // Set up initial inventory
        $initialInventory = AmazonFbaInitialInventory::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);
        $jsonObject = $initialInventory->json_object;
        $jsonObject['fnsku'] = 'FNSKU1';
        $jsonObject['msku'] = $amazonProducts[0]['seller_sku'];
        $initialInventory->json_object = $jsonObject;
        $initialInventory->save();
        $initialInventory->refresh();

        $initialInventory = AmazonFbaInitialInventory::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);
        $jsonObject = $initialInventory->json_object;
        $jsonObject['fnsku'] = 'FNSKU2';
        $jsonObject['msku'] = $amazonProducts[1]['seller_sku'];
        $initialInventory->json_object = $jsonObject;
        $initialInventory->save();
        $initialInventory->refresh();

        /*
        |--------------------------------------------------------------------------
        | Show product
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('amazon.products.show', [$this->amazonIntegrationInstance->id, $amazonProducts[0]['id']]))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                'id',
                'product',
                'item_name',
                'seller_sku',
                'price',
                'quantity',
                'image_url',
                'product_id_type',
                'item_note',
                'item_condition',
                'asin1',
                'product_id',
                'fulfillment_channel',
                'merchant_shipping_group',
                'status',
                'json_object',
                'catalog_data',
                'brand',
                'manufacturer',
                'model',
                'part_number',
                'product_type',
                'sales_rank',
                'color',
                'package_length',
                'package_width',
                'package_height',
                'package_dimensions_unit',
                'package_weight',
                'package_weight_unit',
                'main_image',
                'browse_classification',
                'open_date',
                'style',
                'size',
                'item_length',
                'item_width',
                'item_height',
                'item_dimensions_unit',
                'item_weight',
                'item_weight_unit',
                'number_of_items',
                'department',
                'item_type_name',
                'item_type_keyword',
                'packageQuantity',
                'listing_id',
                'item_description',
                'material',
                'website_group_name',
                'catalog_data_last_sync',
                'was_catalog_data_sync_attempted',
                'removed_from_amazon',
                'mapped_sku',
                'created_at',
                'updated_at',
                'mapped_at',
            ],
        ]);

        /*
        |--------------------------------------------------------------------------
        | Create sku product
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('amazon.products.create-sku-products', $this->amazonIntegrationInstance->id), CreateSkuProductsFromSalesChannelData::from([
            'create_all_products' => true,
        ])->toArray())->assertOk();

        $product1 = Product::whereSku($amazonProducts[0]['seller_sku'])->first();
        $product2 = Product::whereSku($amazonProducts[1]['seller_sku'])->first();

        $this->assertDatabaseHas((new Product())->getTable(), [
            'sku' => $amazonProducts[0]['seller_sku'],
            'id' => $product1->id,
        ]);

        $this->assertDatabaseHas((new Product())->getTable(), [
            'sku' => $amazonProducts[1]['seller_sku'],
            'id' => $product2->id,
        ]);

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'document_type' => AmazonProduct::class,
            'document_id' => $amazonProducts[0]['id'],
            'listing_sku' => $amazonProducts[0]['seller_sku'],
        ]);

        Queue::assertPushed(CreateAmazonFnSkusToSkuProductsJob::class);
        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->generateFnskuProducts();

        $this->assertDatabaseHas((new AmazonFnskuProduct())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => 'FNSKU1',
            'product_id' => $product1->id,
        ]);

        $this->assertDatabaseHas((new AmazonFnskuProduct())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => 'FNSKU2',
            'product_id' => $product2->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Remap product
        |--------------------------------------------------------------------------
        */

        $product = Product::factory()->create([
            'sku' => 'ABC',
        ]);

        $this->putJson(route('amazon.products.map', $this->amazonIntegrationInstance->id), SalesChannelProductToSkuProductMappingCollectionData::from([
            'mapping' => SalesChannelProductToSkuProductMappingData::collection([
                SalesChannelProductToSkuProductMappingData::from([
                    'sales_channel_listing_id' => $amazonProducts[1]['seller_sku'],
                    'mapped_sku' => $product->sku,
                ]),
            ]),
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'document_type' => AmazonProduct::class,
            'document_id' => $amazonProducts[1]['id'],
            'listing_sku' => $amazonProducts[1]['seller_sku'],
            'product_id' => $product->id,
        ]);

        Queue::assertPushed(CreateAmazonFnSkusToSkuProductsJob::class);

        $this->assertDatabaseHas((new AmazonFnskuProduct())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => 'FNSKU2',
            'product_id' => $product->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Unmap product
        |--------------------------------------------------------------------------
        */

        $this->putJson(route('amazon.products.map', $this->amazonIntegrationInstance->id), SalesChannelProductToSkuProductMappingCollectionData::from([
            'mapping' => SalesChannelProductToSkuProductMappingData::collection([
                SalesChannelProductToSkuProductMappingData::from([
                    'sales_channel_listing_id' => $amazonProducts[0]['seller_sku'],
                    'mapped_sku' => null,
                ]),
                SalesChannelProductToSkuProductMappingData::from([
                    'sales_channel_listing_id' => $amazonProducts[1]['seller_sku'],
                    'mapped_sku' => null,
                ]),
            ]),
        ])->toArray())->assertOk();

        $this->assertDatabaseEmpty((new ProductListing())->getTable());

        $this->assertDatabaseHas((new AmazonFnskuProduct())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => 'FNSKU1',
            'product_id' => null,
        ]);

        $this->assertDatabaseHas((new AmazonFnskuProduct())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => 'FNSKU2',
            'product_id' => null,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Map products
        |--------------------------------------------------------------------------
        */

        //Data to test mapping of sales order lines.
        $salesChannelOrderLine = AmazonOrderItem::factory()->hasOrder()->create()->refresh();

        $salesChannelOrderLine->json_object = array_merge($salesChannelOrderLine->json_object, [
            'SellerSKU' => $amazonProducts[1]['seller_sku'],
        ]);
        $salesChannelOrderLine->save();

        $salesOrder = SalesOrder::factory()->create([
            'sales_channel_order_type' => AmazonOrder::class,
            'sales_channel_order_id' => 1,
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
        ]);

        $salesOrderLine = SalesOrderLine::factory()
            ->for($salesOrder)
            ->create([
                'product_id' => null,
                'product_listing_id' => null,
                'sales_channel_line_id' => $salesChannelOrderLine->OrderItemId,
            ]);

        // $amazonProducts[1] should be the first amazon product created (it appears last in the list from the index method because it has the lowest id)

        $this->putJson(route('amazon.products.map', $this->amazonIntegrationInstance->id), SalesChannelProductToSkuProductMappingCollectionData::from([
            'mapping' => SalesChannelProductToSkuProductMappingData::collection([
                SalesChannelProductToSkuProductMappingData::from([
                    'sales_channel_listing_id' => $amazonProducts[1]['seller_sku'],
                    'mapped_sku' => $product->sku,
                ]),
            ]),
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'document_type' => AmazonProduct::class,
            'document_id' => $amazonProducts[1]['id'],
            'listing_sku' => $amazonProducts[1]['seller_sku'],
        ]);

        Queue::assertPushed(MapSalesOrderLinesToSalesChannelProductsJob::class);
        (end(Queue::pushedJobs()[MapSalesOrderLinesToSalesChannelProductsJob::class])['job'])->handle();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine->id,
            'product_id' => $product->id,
            'product_listing_id' => ProductListing::whereListingSku($amazonProducts[1]['seller_sku'])->first()->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Export products
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('amazon.products.export', $this->amazonIntegrationInstance->id))->assertOk()->assertDownload();

        /*
        |--------------------------------------------------------------------------
        | Download CSV products
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('amazon.products.export-download', $this->amazonIntegrationInstance->id), [
            'included' => [
                'seller_sku',
                'item_name',
                'modelNumber',
                'asin1',
                'mapped_sku',
            ],
        ])->assertOk()->assertDownload();

        /*
        |--------------------------------------------------------------------------
        | Upload CSV products
        |--------------------------------------------------------------------------
        */

        $product = Product::factory()->create([
            'sku' => 'NewProduct',
        ]);

        $csvData = [
            [AmazonProduct::getUniqueField(), 'item_name', 'modelNumber', 'asin1', 'mapped_sku'],
            [$amazonProducts[1]['seller_sku'], $amazonProducts[1]['item_name'], '', $amazonProducts[1]['asin1'], 'NewProduct'],
        ];

        $path = Storage::disk('sales-channel-product-mappings')->path('test listings.csv');
        $fh = fopen($path, 'w');

        foreach ($csvData as $row) {
            fputcsv($fh, $row);
        }

        fclose($fh);

        $mappingData = SalesChannelProductImportMappingData::from([
            'original_name' => 'test.csv',
            'stored_name' => 'test listings.csv',
        ]);

        $this->postJson(route('amazon.products.import-mappings', $this->amazonIntegrationInstance->id), $mappingData->toArray())->assertOk();

        Queue::assertPushed(ImportSalesChannelProductMappingsJob::class);

        $manager = new AmazonProductManager($this->amazonIntegrationInstance);

        (new ImportSalesChannelProductMappingsJob($mappingData, $manager))->handle();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'document_type' => AmazonProduct::class,
            'document_id' => $amazonProducts[1]['id'],
            'sales_channel_listing_id' => $amazonProducts[1]['seller_sku'],
            'listing_sku' => $amazonProducts[1]['seller_sku'],
            'product_id' => $product->id,
        ]);

        Queue::assertPushed(MapSalesOrderLinesToSalesChannelProductsJob::class);
    }

    /**
     * @throws Throwable
     */
    private function amazonAssertionsForCatalogProducts(): void
    {
        $this->mockSearchCatalogItems();

        Queue::assertPushed(CreateAmazonSearchCatalogItemsJobsJob::class);
        (new CreateAmazonSearchCatalogItemsJobsJob($this->amazonIntegrationInstance))->handle();

        Queue::assertPushed(SearchAmazonCatalogItemsJob::class);
        $productsMissingCatalogItem = app(AmazonProductRepository::class)->getProductsMissingCatalogItem($this->amazonIntegrationInstance);
        (new AmazonProductManager($this->amazonIntegrationInstance))->searchCatalogItems($productsMissingCatalogItem);

        $this->assertDatabaseHas((new AmazonProduct())->getTable(), [
            'was_catalog_data_sync_attempted' => true,
            'catalog_data' => json_encode(AmazonTestingData::getCatalogData()['items'][0]),
        ]);
    }
}
