<?php

namespace Feature\Controllers;

use App\Models\InventoryAdjustment;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\User;
use App\Models\Warehouse;
use Carbon\Carbon;
use Database\Factories\WarehouseFactory;
use Illuminate\Support\Arr;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportCustomerReturn;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Queue;
use Tests\TestCase;

/**
 * Ledgers are created from reports.  Set up should set up initial ledgers and initial inventory.
 */
class AmazonLedgerControllerTest extends TestCase
{
    use FastRefreshDatabase;

    private AmazonIntegrationInstance $integrationInstance;

    protected function setUp(): void
    {
        parent::setUp();
        Queue::fake();
        Sanctum::actingAs(User::first());

        $this->integrationInstance = AmazonIntegrationInstance::factory()
            ->hasSalesChannel()
            ->create();

        (new InitializeFbaWarehouse($this->integrationInstance))->handle();

        $this->seedInitialInventory();
        $this->seedLedgers();
    }

    private function seedInitialInventory(): void
    {
        Product::factory(3)
            ->create()
            ->each(function ($product) {
                // First, create an AmazonProduct with default factory settings
                $amazonProduct = AmazonProduct::factory()->create([
                    'integration_instance_id' => $this->integrationInstance->id,
                ]);

                // Now, update the seller_sku within the json_object
                $jsonObject = $amazonProduct->json_object;
                $jsonObject['seller_sku'] = $product->sku;
                $amazonProduct->json_object = $jsonObject;
                $amazonProduct->save(); // Save the updated AmazonProduct

                $amazonProduct->refresh(); // To initialize stored generated fields

                ProductListing::factory()->create([
                    'sales_channel_id' => $this->integrationInstance->salesChannel->id,
                    'product_id' => $product->id,
                    'sales_channel_listing_id' => $amazonProduct->seller_sku,
                    'document_id' => $amazonProduct->id,
                    'document_type' => AmazonProduct::class,
                ]);

                $fnskuProduct = AmazonFnskuProduct::factory()->create([
                    'integration_instance_id' => $this->integrationInstance->id,
                    'product_id' => $product->id,
                    'disposition' => 'SELLABLE',
                ]);

                $amazonInitialInventory = AmazonFbaInitialInventory::factory()->create([
                    'integration_instance_id' => $this->integrationInstance->id,
                ]);

                $jsonObject = $amazonInitialInventory->json_object;
                $jsonObject['fnsku'] = $fnskuProduct->fnsku;
                $jsonObject['disposition'] = 'SELLABLE';
                $amazonInitialInventory->json_object = $jsonObject;
                $amazonInitialInventory->save();

                $amazonInitialInventory->refresh();
            });
    }

    private function seedLedgers(): void
    {
        $fnskuProduct = AmazonFnskuProduct::query()->first();

        $ledger = AmazonFbaReportInventoryLedger::factory()->create([
            'integration_instance_id' => $this->integrationInstance->id,
        ]);

        $jsonObject = $ledger->json_object;
        $jsonObject['date'] = Carbon::now()->toDateString();
        $jsonObject['fnsku'] = $fnskuProduct->fnsku;
        $jsonObject['disposition'] = 'SELLABLE';
        $jsonObject['quantity'] = 2;
        $jsonObject['msku'] = 'abc';
        $jsonObject['event_type'] = FbaInventoryLedgerReportEventTypeEnum::CustomerReturns;
        $jsonObject['fulfillment_center'] = 'DEF';

        $ledger->json_object = $jsonObject;
        $ledger->save();
        $ledger->refresh();

        $customerReturn = AmazonFbaReportCustomerReturn::factory()->create([
            'integration_instance_id' => $this->integrationInstance->id,
        ]);

        $jsonObject = $customerReturn->json_object;
        $jsonObject['fnsku'] = $fnskuProduct->fnsku;
        $jsonObject['detailed_disposition'] = 'SELLABLE';
        $jsonObject['quantity'] = $ledger->quantity;
        $jsonObject['sku'] = 'abc';
        $jsonObject['fulfillment_center_id'] = 'DEF';
        $customerReturn->json_object = $jsonObject;
        $customerReturn->save();
        $customerReturn->refresh();
    }

    public function test_amazon_ledger_controller(): void
    {
        $customerReturn = AmazonFbaReportCustomerReturn::query()->first();

        /*
        |--------------------------------------------------------------------------
        | Get Ledgers
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('amazon.ledgers.index', [
            'integrationInstance' => $this->integrationInstance->id,
        ]))->assertOk();

        $response->assertJsonCount(1, 'data');

        /*
        |--------------------------------------------------------------------------
        | Show Ledger
        |--------------------------------------------------------------------------
        */

        $ledger = AmazonFbaReportInventoryLedger::query()->first();

        $this->getJson(route('amazon.ledgers.show', [
            'integrationInstance' => $this->integrationInstance->id,
            'ledger' => $ledger->id,
        ]))->assertOk();

        /*
        |--------------------------------------------------------------------------
        | Get Unmatched Detail Reports
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('amazon.ledgers.unmatched-detail', [
            'integrationInstance' => $this->integrationInstance->id,
            'amazonFbaReportInventoryLedger' => $ledger->id,
        ]))->assertOk();

        $response->assertJsonCount(1, 'data');

        /*
        |--------------------------------------------------------------------------
        | Get Unmatched Ledgers
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('amazon.ledgers.unmatched-ledgers', [
            'integrationInstance' => $this->integrationInstance->id,
        ]) . '?' . http_build_query([
            'detailReportId' => $customerReturn->id,
            'eventType' => $ledger->event_type->value,
        ]))->assertOk();

        $response->assertJsonCount(1, 'data');

        /*
        |--------------------------------------------------------------------------
        | Link to Detail Report
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('amazon.ledgers.link-to-detail-reports', [
            'integrationInstance' => $this->integrationInstance->id,
            'ledger' => $ledger->id,
        ]), [
            'detailReportIds' => [$customerReturn->id],
        ])->assertOk();

        /*
        |--------------------------------------------------------------------------
        | Reconcile Ledger
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('amazon.ledgers.reconcile', [
            'integrationInstance' => $this->integrationInstance->id,
            'ledger' => $ledger->id,
        ]))->assertOk();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $ledger->id,
            'sku_link_id' => InventoryAdjustment::first()->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Unreconcile Ledger
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('amazon.ledgers.unreconcile', [
            'integrationInstance' => $this->integrationInstance->id,
            'ledger' => $ledger->id,
        ]))->assertOk();

        $this->assertDatabaseEmpty((new InventoryAdjustment())->getTable());

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $ledger->id,
            'sku_link_id' => null,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Reconcile Ledgers
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('amazon.ledgers.reconcile-ledgers', [
            'integrationInstance' => $this->integrationInstance->id,
        ]), [
            'ids' => Arr::wrap($ledger->id),
        ])->assertOk();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $ledger->id,
            'sku_link_id' => InventoryAdjustment::first()->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Unreconcile Ledgers
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('amazon.ledgers.unreconcile-ledgers', [
            'integrationInstance' => $this->integrationInstance->id,
        ]), [
            'ids' => Arr::wrap($ledger->id),
        ])->assertOk();

        /*
        |--------------------------------------------------------------------------
        | Reconcile All
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('amazon.ledgers.reconcile-all', [
            'integrationInstance' => $this->integrationInstance->id,
        ]))->assertOk();

        $this->assertDatabaseHas((new AmazonFbaReportInventoryLedger())->getTable(), [
            'id' => $ledger->id,
            'sku_link_id' => InventoryAdjustment::first()->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Get Last Reconciled Date
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('amazon.ledgers.reconcile-all', [
            'integrationInstance' => $this->integrationInstance->id,
        ]))->assertOk();

        $response = $this->getJson(route('amazon.ledgers.last-reconciled-date', [
            'integrationInstance' => $this->integrationInstance->id,
        ]))->assertOk();

        $this->assertEquals($ledger->refresh()->reconciled_at->setTimezone('utc')->toDateString(), $response->content());

        /*
        |--------------------------------------------------------------------------
        | Get Reconciliation Report
        |--------------------------------------------------------------------------
        */

        // Covered in AmazonReportManagerTest::test_get_reconciliation_report
    }
}
