<?php

namespace Modules\Amazon\Tests\Feature\Controllers;

use App\Data\SalesChannelInventoryLocationsData;
use App\Data\SalesChannelInventorySettingsData;
use App\Data\SalesChannelMasterOfPriceSettingsData;
use App\Data\SalesChannelPricingSettingsData;
use App\Jobs\CacheProductListingPriceJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\IntegrationShippingMethod;
use App\Models\PaymentType;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Models\ShippingCarrier;
use App\Models\ShippingMethod;
use App\Models\Store;
use App\Models\User;
use App\Models\Warehouse;
use Carbon\Carbon;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Data\AmazonAuthorizationResponseData;
use Modules\Amazon\Data\AmazonConnectionSettingsData;
use Modules\Amazon\Data\AmazonIntegrationSettingsData;
use Modules\Amazon\Data\StoreAmazonIntegrationData;
use Modules\Amazon\Data\UpdateAmazonIntegrationData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonOrderItem;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReportRequest;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Jobs\CreateAmazonReportJob;
use Modules\Amazon\Jobs\RefreshAmazonFbaInboundShipmentsJob;
use Modules\Amazon\Jobs\RefreshAmazonFinancialEventGroupsJob;
use Modules\Amazon\Jobs\RefreshAmazonOrdersJob;
use Modules\Amazon\Jobs\RefreshAmazonSettlementReportsJob;
use Modules\Amazon\Services\AmazonClient;
use Modules\Amazon\Tests\AmazonMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Spatie\LaravelData\DataCollection;
use Tests\TestCase;

class AmazonIntegrationInstanceControllerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;
    use WithFaker;

    public function setUp(): void
    {
        parent::setUp();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Exception
     */
    public function test_amazon_integration_instance_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Create integration instance
        |--------------------------------------------------------------------------
        */

        $integration = Integration::where('name', Integration::NAME_AMAZON_US)->first();

        $response = $this->postJson(route('amazon.store'), StoreAmazonIntegrationData::from([
            'name' => 'Amazon',
            'integration_id' => $integration->id,
            'country' => 'NA',
            'connection_settings' => AmazonConnectionSettingsData::from([]),
            'integration_settings' => AmazonIntegrationSettingsData::from([
                'start_date' => Carbon::now()->subYear(1)->toDateTimeString(), // This should be already in UTC from frontend
                'store_id' => Store::first()->id,
                'auto_link_products' => false,
                'auto_create_products' => false,
                'sales_nominal_code_id' => null,
                'cogs_nominal_code_id' => null,
                'pricing' => SalesChannelPricingSettingsData::from([
                    'masterOfPrice' => SalesChannelMasterOfPriceSettingsData::empty(),
                ]),
                'inventory' => SalesChannelInventorySettingsData::from([]),
                'is_fba_enabled' => true,
                'fba_inventory_tracking_start_date' => Carbon::now()->subYear()->toDateTimeString(),
                'batch_sales_order_invoices' => false,
                'emailCustomers' => false,
                'proforma_marketplace_cost_percentage' => null,
                'proforma_payment_cost_percentage' => null,
                'automatically_create_warehouse_transfers_from_inbounds' => false,
                'automatically_create_purchase_orders_from_inbounds' => false,
                'automatically_create_warehouse_transfers_from_removal_orders' => false,
                'removal_order_warehouse_id' => null,
                'sync_sales_order_invoices_to_accounting' => true
            ]),
        ])->toArray())->assertOk();

        $integrationInstance = AmazonIntegrationInstance::find($response->json()['data']['id']);

        $this->assertEquals((new AmazonClient($integrationInstance))
            ->getRedirectUrl(), $response->json('redirect_url'));

        $this->assertDatabaseHas((new IntegrationInstance())->getTable(), [
            'name' => 'Amazon',
        ]);

        $this->assertDatabaseHas((new SalesChannel())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'store_id' => Store::first()->id,
        ]);

        $this->assertDatabaseHas((new PaymentType())->getTable(), [
            'name' => 'Amazon Payment',
        ]);

        /*
        |--------------------------------------------------------------------------
        | Update integration instance
        |--------------------------------------------------------------------------
        */

        $newStore = Store::factory()->create();

        $newIntegrationSettings = [
            'start_date' => Carbon::now('UTC')->subYear()->toDateTimeString(),
            'store_id' => $newStore->id,
            'auto_link_products' => false,
            'auto_create_products' => false,
            'sales_nominal_code_id' => null,
            'cogs_nominal_code_id' => null,
            'pricing' => SalesChannelPricingSettingsData::from([
                'masterOfPrice' => SalesChannelMasterOfPriceSettingsData::from([
                    'name' => 'sku.io',
                ]),
            ]),
            'inventory' => SalesChannelInventorySettingsData::from([
                'masterOfStock' => 'sku.io',
            ]),
            'inventoryLocations' => new DataCollection(SalesChannelInventoryLocationsData::class,
                [
                    SalesChannelInventoryLocationsData::from([
                        'masterOfStock' => 'Neither',
                        'maxRuleTypeValue' => null,
                        'minRuleTypeValue' => null,
                        'subtractBufferStock' => null,
                        'selectedWarehouses' => ['xyz'],
                        'fulfillmentLatency' => '1',
                        'maxRuleType' => 'None',
                        'minRuleType' => 'None',
                    ]),
                ]),
            'is_fba_enabled' => true,
            'fba_inventory_tracking_start_date' => Carbon::now('UTC')->subYear()->toDateTimeString(),
            'batch_sales_order_invoices' => false,
            'emailCustomers' => false,
            'proforma_marketplace_cost_percentage' => null,
            'proforma_payment_cost_percentage' => null,
            'automatically_create_warehouse_transfers_from_inbounds' => false,
            'automatically_create_purchase_orders_from_inbounds' => false,
            'automatically_create_warehouse_transfers_from_removal_orders' => false,
            'removal_order_warehouse_id' => null,
            'sync_sales_order_invoices_to_accounting' => true
        ];

        $this->putJson(route('amazon.update', $integrationInstance->id), UpdateAmazonIntegrationData::from([
            'integration_settings' => AmazonIntegrationSettingsData::from($newIntegrationSettings),
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new IntegrationInstance())->getTable(), [
            'name' => 'Amazon',
            'integration_settings' => json_encode($newIntegrationSettings),
        ]);

        $this->assertDatabaseHas((new SalesChannel())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'store_id' => $newStore->id,
        ]);

        Queue::assertPushed(CacheProductListingPriceJob::class);
        Queue::assertPushed(GenerateCacheProductListingQuantityJob::class);

        /*
        |--------------------------------------------------------------------------
        | Get redirect Url
        |--------------------------------------------------------------------------
        */

        $this->get(route('amazon.get-redirect-url', $integrationInstance->id))
            ->assertOk()
            ->assertJson([
                'data' => (new AmazonClient($integrationInstance))->getRedirectUrl(),
            ]);

        /*
        |--------------------------------------------------------------------------
        | Handle authorization response
        |--------------------------------------------------------------------------
        */

        $this->mockRefreshTokenFromAuthCode();
        $this->mockCreateReport();
        $this->mockGetMarketplaceParticipations();

        $this->postJson(route('amazon.callback'), AmazonAuthorizationResponseData::from([
            'state' => 'subdomain_' . $integrationInstance->id,
            'spapi_oauth_code' => 'authorization code',
            'selling_partner_id' => 'selling partner id',
        ])->toArray())->assertRedirect('/integrations/amazon-us/' . $integrationInstance->id . '/dashboard');

        $this->assertDatabaseHas((new IntegrationInstance())->getTable(), [
            'id' => $integrationInstance->id,
            'connection_settings' => AmazonConnectionSettingsData::from([
                'selling_partner_id' => 'selling partner id',
                'refresh_token' => 'refresh token',
                'access_token' => 'access token',
            ])->toJson(),
        ]);

        $this->amazonAssertionsForNewInstance($integrationInstance->refresh());

        /*
        |--------------------------------------------------------------------------
        | Show integration instance
        |--------------------------------------------------------------------------
        */

        $this->get(route('amazon.show', $integrationInstance->id))
            ->assertOk()
            ->assertJsonStructure([
                'data' => [
                    'id',
                    'name',
                    'connection_settings',
                    'integration_settings',
                    'connection_status',
                    'integration_name',
                    'created_at',
                    'updated_at',
                ],
            ]);

        /*
        |--------------------------------------------------------------------------
        | Delete integration instance
        |--------------------------------------------------------------------------
        */

        $amazonProduct = AmazonProduct::factory()->create();
        ProductListing::factory()->create([
            'document_id' => $amazonProduct->id,
            'document_type' => AmazonProduct::class,
        ]);
        $amazonOrder = AmazonOrder::factory(2)->has(AmazonOrderItem::factory(), 'orderItems')->create();
        SalesOrder::factory()->create([
            'sales_channel_order_id' => $amazonOrder->first()->id,
            'sales_channel_order_type' => AmazonOrder::class,
        ]);

        $this->assertEquals(1, AmazonIntegrationInstance::count());
        $this->assertEquals(1, AmazonProduct::count());
        $this->assertEquals(1, ProductListing::count());
        $this->assertEquals(2, AmazonOrder::count());
        $this->assertEquals(1, SalesOrder::count());

        $this->delete(route('amazon.destroy', $integrationInstance->id));

        $this->assertEquals(0, AmazonIntegrationInstance::query()->count());
        $this->assertEquals(0, AmazonProduct::query()->count());
        $this->assertEquals(0, ProductListing::query()->count());
        $this->assertEquals(0, AmazonOrder::query()->count());
        $this->assertEquals(0, SalesOrder::query()->count());

        $this->assertDatabaseMissing((new PaymentType())->getTable(), [
            'name' => 'Amazon Payment',
        ]);

        $this->assertDatabaseMissing((new ShippingMethod())->getTable(), [
            'full_name' => 'Amazon MCF Standard',
        ]);
        $this->assertDatabaseMissing((new ShippingCarrier())->getTable(), [
            'name' => 'Amazon MCF',
        ]);
    }

    private function amazonAssertionsForNewInstance(AmazonIntegrationInstance $integrationInstance): void
    {
        $this->assertDatabaseHas((new AmazonReportRequest())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'reportType' => AmazonReportTypeEnum::PRODUCTS,
        ]);

        $this->amazonAssertionsForNewFba($integrationInstance);

        Queue::assertPushed(CreateAmazonReportJob::class);
        Queue::assertPushed(RefreshAmazonSettlementReportsJob::class);
        Queue::assertPushed(RefreshAmazonFinancialEventGroupsJob::class);
        Queue::assertPushed(RefreshAmazonOrdersJob::class);
    }

    private function amazonAssertionsForNewFba(AmazonIntegrationInstance $integrationInstance): void
    {
        $this->assertCount(11, $integrationInstance->integration_settings['marketplaceParticipations']);

        $this->assertDatabaseHas((new Warehouse())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'name' => 'FBA (' . $integrationInstance->name . ')',
            'type' => Warehouse::TYPE_AMAZON_FBA
        ]);

        $this->assertDatabaseHas((new Setting())->getTable(), [
            'key' => Setting::KEY_WAREHOUSE_PRIORITY,
            'value' => json_encode(json_encode([
                1, // Default warehouse
                Warehouse::where('type', Warehouse::TYPE_AMAZON_FBA)->first()->id,
            ])),
        ]);

        $this->amazonAssertionsForFbaShippingMethods($integrationInstance);

        Queue::assertPushed(RefreshAmazonFbaInboundShipmentsJob::class);

    }

    private function amazonAssertionsForFbaShippingMethods(AmazonIntegrationInstance $integrationInstance): void
    {
        $this->assertDatabaseHas((new ShippingCarrier())->getTable(), [
            'name' => 'Amazon MCF',
        ]);
        $this->assertDatabaseHas((new ShippingMethod())->getTable(), [
            'name' => 'Standard',
        ]);
        $this->assertDatabaseHas((new IntegrationShippingMethod())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'shipping_method_id' => ShippingMethod::where('name', 'Standard')->first()->id,
            'code' => 'Standard',
        ]);
        $this->assertDatabaseHas((new ShippingMethod())->getTable(), [
            'name' => 'Expedited',
        ]);
        $this->assertDatabaseHas((new IntegrationShippingMethod())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'shipping_method_id' => ShippingMethod::where('name', 'Expedited')->first()->id,
            'code' => 'Expedited',
        ]);
        $this->assertDatabaseHas((new ShippingMethod())->getTable(), [
            'name' => 'Priority',
        ]);
        $this->assertDatabaseHas((new IntegrationShippingMethod())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'shipping_method_id' => ShippingMethod::where('name', 'Priority')->first()->id,
            'code' => 'Priority',
        ]);
        $this->assertDatabaseHas((new ShippingMethod())->getTable(), [
            'name' => 'ScheduledDelivery',
        ]);
        $this->assertDatabaseHas((new IntegrationShippingMethod())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'shipping_method_id' => ShippingMethod::where('name', 'ScheduledDelivery')->first()->id,
            'code' => 'ScheduledDelivery',
        ]);
    }
}
