<?php

namespace Modules\Amazon\Services;

use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Client\ConnectionException;
use Modules\Amazon\Data\AmazonFulfillmentOrderData;
use Modules\Amazon\Data\AmazonOutboundFulfillmentOrderData;
use Modules\Amazon\Data\AmazonOutboundFulfillmentStatusData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Exceptions\AmazonFulfillmentOrderException;
use Modules\Amazon\Exceptions\AmazonTimeoutException;

class AmazonFulfillmentClient extends AmazonAuthenticationClient
{
    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     * @throws Exception
     */
    public function createFulfillmentOrder(AmazonFulfillmentOrderData $dto): void
    {
        $response = $this->request(HttpMethodEnum::POST, '/fba/outbound/2020-07-01/fulfillmentOrders', $dto->toArray());

        if (!$response->successful()) {

            if ($response->status() === 400) {
                $errors = collect($response->json()['errors']);
                $codes = $errors->pluck('code')->unique();
                if ($codes->count() === 1 && $codes->first() === 'InvalidInput') {
                    $errorMessages = $errors->pluck('message')->implode(', ');
                    throw new AmazonFulfillmentOrderException($errorMessages);
                }
            }

            throw new Exception('Failed to create amazon fulfillment order: '. $response->body());
        }

        if (isset($response->json()['errors'])) {
            throw new Exception('Failed to create amazon fulfillment order (errors): '. $response->body());
        }
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function cancelFulfillmentOrder(string $sellerFulfillmentOrderId): void
    {
        $this->request(HttpMethodEnum::PUT, '/fba/outbound/2020-07-01/fulfillmentOrders/'.$sellerFulfillmentOrderId.'/cancel', []);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function listAllFulfillmentOrders(Carbon $queryStartDate, ?string $nextToken = null): AmazonResponseData
    {
        $requestParams = [
            'queryStartDate' => $queryStartDate->toIso8601ZuluString(),
        ];

        if (! is_null($nextToken)) {
            $requestParams['nextToken'] = $nextToken;
        }

        $response = $this->request(HttpMethodEnum::GET, '/fba/outbound/2020-07-01/fulfillmentOrders', $requestParams);

        $collection = !isset($response->json()['payload']['fulfillmentOrders']) ?
            collect() :
            AmazonOutboundFulfillmentStatusData::collection($response->json()['payload']['fulfillmentOrders']);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['payload']['nextToken'],
        ]);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getFulfillmentOrder(string $sellerFulfillmentOrderId): AmazonResponseData
    {
        $response = $this->request(HttpMethodEnum::GET, '/fba/outbound/2020-07-01/fulfillmentOrders/'.$sellerFulfillmentOrderId);

        $collection = AmazonOutboundFulfillmentOrderData::collection([$response->json()['payload']]);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => null,
        ]);
    }
}
