<?php

namespace Modules\Amazon\Services;

use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use Carbon\Carbon;
use Illuminate\Http\Client\ConnectionException;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetFinancialEventsAdt;
use Modules\Amazon\Data\AmazonFinancialEventData;
use Modules\Amazon\Data\AmazonFinancialEventGroupData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Entities\AmazonFinancialEventGroup;
use Modules\Amazon\Exceptions\AmazonTimeoutException;

class AmazonFinancialClient extends AmazonAuthenticationClient
{
    /**
     * Returns financial event groups for a given date range.
     *
     * @throws ApiException
     */
    public function getFinancialEventGroups(
        ?string $startedAfter = null,
        ?string $startedBefore = null,
        ?string $nextToken = null
    ): AmazonResponseData {
        $requestParams = [
            'FinancialEventGroupStartedAfter' => $startedAfter ?
                (Carbon::parse($startedAfter)->toIso8601ZuluString()) : null,
        ];

        if (! is_null($startedBefore)) {
            $requestParams['FinancialEventGroupStartedBefore'] = Carbon::parse($startedBefore)->toIso8601ZuluString();
        }

        if (! is_null($nextToken)) {
            $requestParams['NextToken'] = $nextToken;
        }

        $response = $this->request(HttpMethodEnum::GET, '/finances/v0/financialEventGroups', $requestParams);

        if (! isset($response->json()['payload'])) {
            dd($response->json(), $requestParams);
        }

        $collection = AmazonFinancialEventGroupData::collection($response->json()['payload']['FinancialEventGroupList']);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['payload']['NextToken'],
        ]);
    }

    /**
     * Returns financial events by group id for a given date range.
     *
     * @throws ApiException
     * @throws ConnectionException
     * @throws AmazonTimeoutException
     */
    public function getFinancialEventsByGroupId(
        AmazonFinancialEventGroup $amazonFinancialEventGroup,
        ?string $postedAfter = null,
        ?string $nextToken = null
    ): AmazonResponseData {
        $requestParams = [
            'PostedAfter' => $postedAfter ?
                (Carbon::parse($postedAfter)->toIso8601ZuluString()) : null,
        ];

        if (! is_null($nextToken)) {
            $requestParams['NextToken'] = $nextToken;
        }

        $response = $this->request(
            HttpMethodEnum::GET,
            "/finances/v0/financialEventGroups/$amazonFinancialEventGroup->FinancialEventGroupId/financialEvents",
            $requestParams
        );

        $collection = collect();

        /*
         * TODO: Decide if I want financial events table to store all events in one json or separate record for each
         *  event.
         * TODO: We have to be careful about processing financial events that are not completed
         */

        if (! isset($response->json()['payload'])) {
            dd($response->json());
        }

        foreach ($response->json()['payload']['FinancialEvents'] as $event_type => $items) {
            if (count($items) == 0) {
                continue;
            }
            $group = AmazonFinancialEventData::from([
                'amazon_financial_event_group_id' => $amazonFinancialEventGroup->id,
                'event_type' => $event_type,
                'events' => $items,
            ]);
            $collection->add(
                item: $group,
            );
        }

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['payload']['NextToken'],
        ]);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getFinancialEvents(AmazonGetFinancialEventsAdt $parameters): AmazonResponseData
    {
        $response = $this->request(HttpMethodEnum::GET, '/finances/v0/financialEvents', $parameters->transform());

        return AmazonResponseData::from([
            'collection' => AmazonFinancialEventData::collection($response->json()['payload']['FinancialEvents']),
            'nextToken' => @$response->json()['payload']['NextToken'],
        ]);
    }
}
