<?php

namespace Modules\Amazon\Repositories;

use App\Abstractions\AbstractRepository;
use App\Helpers;
use App\Models\Setting;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Storage;
use Modules\Amazon\Data\AbstractAmazonFbaDetailReportData;
use Modules\Amazon\Data\AbstractAmazonReportData;
use Modules\Amazon\Data\AmazonReportData;
use Modules\Amazon\Data\AmazonReportSettlementDataData;
use Modules\Amazon\Entities\AbstractAmazonFbaDetailReport;
use Modules\Amazon\Entities\AmazonFbaReportCustomerReturn;
use Modules\Amazon\Entities\AmazonFbaReportInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedgerSummary;
use Modules\Amazon\Entities\AmazonFbaReportRemovalOrder;
use Modules\Amazon\Entities\AmazonFbaReportRemovalShipment;
use Modules\Amazon\Entities\AmazonFbaReportRestock;
use Modules\Amazon\Entities\AmazonFbaReportShipment;
use Modules\Amazon\Entities\AmazonFinancialEventGroup;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Entities\AmazonReportRequest;
use Modules\Amazon\Entities\AmazonReportSettlementData;
use Modules\Amazon\Entities\AmazonReportSettlementTypeMapping;
use Modules\Amazon\Enums\Entities\ReportProcessingStatusEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Throwable;

class AmazonReportRepository extends AbstractRepository
{
    private AmazonFinancialEventGroupRepository $amazonFinancialEventGroupRepository;

    public function __construct()
    {
        $this->amazonFinancialEventGroupRepository = app(AmazonFinancialEventGroupRepository::class);
    }

    private function getAmazonReportQuery()
    {
        return clone AmazonReport::amazonReports();
    }

    /**
     * Create the Amazon Report from Amazon Report Request and API result
     */
    public function create(AmazonReportRequest $amazonReportRequest, int $amazonReportId): AmazonReport|Model
    {
        return AmazonReport::query()->create([
            'integration_instance_id' => $amazonReportRequest->integration_instance_id,
            'reportId' => $amazonReportId,
            'reportType' => $amazonReportRequest->reportType,
            'dataStartTime' => $amazonReportRequest->dataStartTime,
            'dataEndTime' => $amazonReportRequest->dataEndTime,
            'options' => $amazonReportRequest->options,
            'marketplaceIds' => $amazonReportRequest->marketplaceIds,
            'requested_by_user_id' => $amazonReportRequest->requested_by_user_id,
        ]);
    }

    /**
     * Get Amazon report requests
     */
    public function getRequests(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return AmazonReportRequest::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->get();
    }

    /**
     * Get initialized reports that have been created on Amazon
     */
    public function getInitializedReports(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return $this->getAmazonReportQuery()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereNotNull('reportId')
            ->where(function ($query) {
                $query->whereNull('processingStatus');
                $query->orWhere(function (Builder $query) {
                    $query->where('processingStatus', ReportProcessingStatusEnum::IN_PROGRESS());
                    $query->orWhere('processingStatus', ReportProcessingStatusEnum::IN_QUEUE());
                    $query->orWhere(function (Builder $query) {
                        $query->where('processingStatus', ReportProcessingStatusEnum::DONE());
                        $query->where(function (Builder $query) {
                            $query->whereNull('reportDocumentId')
                                ->orWhereNull('filename');
                        });
                    });
                });
            })
            ->get();
    }

    public function getFromReportIdForIntegrationInstance(AmazonIntegrationInstance $amazonIntegrationInstance, string $report_id): ?AmazonReport
    {
        return $this->getAmazonReportQuery()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('reportId', $report_id)->first();
    }

    public function getFromReportId(string $report_id): ?AmazonReport
    {
        return $this->getAmazonReportQuery()
            ->where('reportId', $report_id)->first();
    }

    public function getRequestFromId(int $id): AmazonReportRequest|Model
    {
        return AmazonReportRequest::query()->findOrFail($id);
    }

    /**
     * @throws Throwable
     */
    public function saveOne(AmazonReport $amazonReport, AmazonReportData $report): Model
    {
        return AmazonReport::query()->updateOrCreate([
            'integration_instance_id' => $amazonReport->integration_instance_id,
            'reportId' => $report->reportId,
            'reportType' => $report->reportType,
        ], $report->toArray());
    }

    /**
     * @throws Throwable
     */
    public function saveRequest(
        int $integration_instance_id,
        AmazonReportTypeEnum $report_type,
        ?Carbon $data_start_time,
        ?Carbon $data_end_time,
        ?array $options,
        ?array $marketplace_ids
    ): Model|AmazonReportRequest {
        return AmazonReportRequest::query()->updateOrCreate([
            'integration_instance_id' => $integration_instance_id,
            'reportType' => $report_type,
            'dataStartTime' => $data_start_time,
            'dataEndTime' => $data_end_time,
            'marketplaceIds' => $marketplace_ids,
        ], [
            'options' => $options,
            'marketplaceIds' => $marketplace_ids,
        ]);
    }

    public function find(int $id): ?AmazonReport
    {
        return $this->getAmazonReportQuery()->with('integrationInstance')->find($id);
    }

    /**
     * @param  null|string  $marketplaceId
     *
     * @throws \InvalidArgumentException
     */
    public function requestExists(int $integrationInstanceId, AmazonReportTypeEnum $report_type, ?Carbon $startTime, ?Carbon $endTime, ?array $marketplaceIds = null): bool
    {
        $query = AmazonReportRequest::query()
            ->where('integration_instance_id', $integrationInstanceId)
            ->where('dataStartTime', $startTime)
            // Requests don't usually have an end time, so we can't check it or this method will always return false
            //->where('dataEndTime', $endTime)
            ->where('reportType', $report_type());

        if ($marketplaceIds) {
            $query->where('marketplaceIds', json_encode($marketplaceIds));
        }

        return $query->count() > 0;
    }

    /**
     * Get existing amazon report request.
     */
    public function getExistingRequestForType(int $integrationInstanceId, AmazonReportTypeEnum $report_type, ?array $marketplaceIds = null): ?AmazonReportRequest
    {
        $query = AmazonReportRequest::query()
            ->where('integration_instance_id', $integrationInstanceId)
            ->where('reportType', $report_type());

        if ($marketplaceIds) {
            $query->where('marketplaceIds', json_encode($marketplaceIds));
        }

        return $query->first();
    }

    /**
     * @throws Exception
     */
    public function getModelClassFromReportType(AmazonReportTypeEnum $reportType): string
    {
        return match ($reportType) {
            AmazonReportTypeEnum::FBA_INVENTORY => AmazonFbaReportInventory::class,
            AmazonReportTypeEnum::FBA_REPORT_CUSTOMER_RETURNS => AmazonFbaReportCustomerReturn::class,
            AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER => AmazonFbaReportInventoryLedger::class,
            AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY => AmazonFbaReportInventoryLedgerSummary::class,
            AmazonReportTypeEnum::FBA_REPORT_SHIPMENTS => AmazonFbaReportShipment::class,
            AmazonReportTypeEnum::FBA_REPORT_REMOVAL_SHIPMENTS => AmazonFbaReportRemovalShipment::class,
            AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS => AmazonFbaReportRemovalOrder::class,
            AmazonReportTypeEnum::FBA_REPORT_RESTOCK => AmazonFbaReportRestock::class,
            AmazonReportTypeEnum::SETTLEMENT_REPORT => AmazonReportSettlementData::class,
            AmazonReportTypeEnum::PRODUCTS => AmazonProduct::class,
            default => throw new Exception('Unhandled report type: '.$reportType->value),
        };
    }

    public function upsertSettlementReportTypes(array $data): void
    {
        AmazonReportSettlementTypeMapping::query()
            ->upsert($data, ['integration_instance_id', 'transaction_type', 'amount_type', 'amount_description']);
    }

    /**
     * @throws Exception
     */
    public function saveReport(AmazonReport $amazonReport): void
    {
        customlog('amazon', "Saving amazon report {$amazonReport->reportId}");

        /*
        * The same report type is used for AmazonFbaInitialInventory and AmazonFbaReportInventoryLedgerSummary.  The difference is the date used
        * The initial inventory is for a single date that is prior to the fba inventory tracking start date.  All other cases should be for the inventory
        * ledger summary
        */
        if ($amazonReport->reportType == AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY && $amazonReport->dataEndTime < $amazonReport->integrationInstance->fbaInventoryTrackingStartDate()) {
            $modelClass = AmazonFbaInitialInventory::class;
        } else
        {
            $modelClass = $this->getModelClassFromReportType($amazonReport->reportType);
        }
        $model = app($modelClass);

        $filesize_mb = round(filesize(Storage::disk('amazon_reports')->path($amazonReport->filename)) / 1024 / 1024);

        if ($filesize_mb > 1) {
            ini_set('memory_limit', ($filesize_mb * 100).'M');
        }
        $records = tsvToArray(file_get_contents(Storage::disk('amazon_reports')->path($amazonReport->filename)), [
            'ignoreMissingColumns' => false,
            'trimColumns' => in_array($modelClass, [AmazonFbaReportInventoryLedger::class, AmazonFbaInitialInventory::class, AmazonFbaReportInventoryLedgerSummary::class]),
        ]);

        customlog('amazon', "Saving amazon report {$amazonReport->reportId} - ".count($records).' records');

        if (
            $modelClass == AmazonReportSettlementData::class
        ) {
            $header = array_shift($records);
            if (! $financialEventGroup = $this->amazonFinancialEventGroupRepository
                ->getFinancialEventGroupForSettlementReport($amazonReport->integrationInstance, $header)) {
                return;
            }
            $records = $this->sanitizeSettlementReportRecords($records, $financialEventGroup);

            $type_mappings = Arr::map($records, function ($record) use ($amazonReport) {
                return array_merge(
                    ['integration_instance_id' => $amazonReport->integration_instance_id],
                    Arr::only($record, ['transaction_type', 'amount_type', 'amount_description'])
                );
            });
            $this->upsertSettlementReportTypes(array_unique_multidimensional($type_mappings));
        }

        $reportCollection = collect($records)->map(function ($record) use ($modelClass, $model, $amazonReport) {
            $reportData = array_merge($record, [
                'json_object' => json_encode($record),
                'checksum' => md5(json_encode($record)),
            ]);

            // If the report was user requested, enforce checksum uniqueness to avoid duplicate records
            if ($amazonReport->requestedByUser || $amazonReport->reportType == AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY || $amazonReport->reportType == AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS) {
                if ($model::query()->where('checksum', $reportData['checksum'])->exists()) {
                    customlog('amazon', "Skipping duplicate record for report type " . $amazonReport->reportType->value . " for checksum {$reportData['checksum']}");
                    return null;
                }
            }

            $reportData['integration_instance_id'] = $amazonReport->integration_instance_id;
            $reportData['amazon_report_id'] = $amazonReport->id;


            if ($model instanceof AbstractAmazonFbaDetailReport) {
                $reportData['event_datetime'] = Carbon::parse($reportData[$model->getDateField()])->setTimezone('UTC')->toDateTimeString();
                $amazonReportDto = AbstractAmazonFbaDetailReportData::from($reportData);
            }
            else if ($model instanceof AmazonFbaReportInventoryLedger || $model instanceof AmazonFbaReportInventoryLedgerSummary) {
                $reportData['event_datetime'] = Carbon::parse($reportData['date'], $amazonReport->integrationInstance->getTimezone())->setTimezone('UTC')->toDateTimeString();
                $amazonReportDto = AbstractAmazonFbaDetailReportData::from($reportData);
            }
            else if ($model instanceof AmazonReportSettlementData) {
                $amazonReportDto = AmazonReportSettlementDataData::from($reportData);
            }
            else {
                $amazonReportDto = AbstractAmazonReportData::from($reportData);
            }

            return $amazonReportDto;
        });

        // Sanitize the initial inventory report to not include reports where total_inventory_quantity = 0
        if ($model instanceof AmazonFbaInitialInventory) {
            $reportCollection = $reportCollection->reject(function ($report) {
                $jsonObject = json_decode($report->json_object, 1);
                return ($jsonObject['ending_warehouse_balance'] + $jsonObject['in_transit_between_warehouses']) == 0;
            });
        }

        // Sanitize the report collection to not include event_datetime before the fba_inventory_tracking_start_date
        $reportCollection = $reportCollection->reject(function ($report) use ($amazonReport) {
            if ($report instanceof AbstractAmazonFbaDetailReportData) {
                return Carbon::parse($report->event_datetime)->lt($amazonReport->integrationInstance->integration_settings['fba_inventory_tracking_start_date']);
            }
            return false;
        });

        customlog('amazon', "Saving amazon report {$amazonReport->reportId} - sanitization complete");

        if (in_array($amazonReport->reportType, AmazonReportTypeEnum::REPLACEABLE_DATA_REPORTS)) {
            $model->query()
                ->where('integration_instance_id', $amazonReport->integration_instance_id)
                ->delete();
        }

        /**
         * Sometimes we get duplicates for inventory ledger report and since checksum is unique
         * for handling the reports in bulk import so we are appending the index for checksum record
         *
         * Ex.
         * "08/22/2022"	"B0022ZZULK"	"B0022ZZULK"	"1722575-FBA"	"Tecmate Optimate Cable O-01, Weatherproof Battery Lead, powersport"	"Shipments"	""	"-1"	"BWI2"	"SELLABLE"		"US"	""	""	"2022-08-22T00:00:00-0700"
         * "08/22/2022"	"B0022ZZULK"	"B0022ZZULK"	"1722575-FBA"	"Tecmate Optimate Cable O-01, Weatherproof Battery Lead, powersport"	"Shipments"	""	"-1"	"BWI2"	"SELLABLE"		"US"	""	""	"2022-08-22T00:00:00-0700"
         *
         * Here these 2 records are duplicates so checksum will be:
         * 1-<CHECKSUM>
         * 2.<CHECKSUM>
         */

        // TODO: Jatin, fix this... it is causing record duplications
//        if ($modelClass === AmazonFbaReportInventoryLedger::class || $modelClass === AmazonReportSettlementData::class) {
//            $tempRecords = clone $reportCollection;
//
//            $tempRecords->groupBy('checksum')->map(function ($tempRecord) use (&$reportCollection) {
//                if ($tempRecord->count() > 1) {
//                    $tempRecord->each(function ($_tempRecord, $index) use (&$reportCollection) {
//                        $_tempRecord->checksum = ($index + 1).'-'.$_tempRecord->checksum;
//                        $reportCollection->push($_tempRecord);
//                    });
//                } else {
//                    $reportCollection->push($tempRecord->first());
//                }
//            });
//        }

        customlog('amazon', "Saving amazon report {$amazonReport->reportId} - checksums generated");

        $reportCollection = $this->save($reportCollection, $modelClass);

        customlog('amazon', "Finished saving amazon report {$amazonReport->reportId}");
    }

    public function getStartDateForReportType(AmazonIntegrationInstance $amazonIntegrationInstance, AmazonReportTypeEnum $reportType): ?Carbon
    {
        if ($dataEndTime = AmazonReport::query()
            ->where('reportType', $reportType)
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereNotNull('dataEndTime')
            ->where(function ($query) {
                $query->whereNull('processingStatus');
                $query->orWhere('processingStatus', ReportProcessingStatusEnum::IN_PROGRESS());
                $query->orWhere('processingStatus', ReportProcessingStatusEnum::IN_QUEUE());
                $query->orWhere('processingStatus', ReportProcessingStatusEnum::DONE());
            })->max('dataEndTime')) {
            return Carbon::parse($dataEndTime)->addSecond();
        } else {
            return null;
        }
    }

    /**
     * @throws Exception
     */
    public function getStartDateForNew(?string $modelClass = null): ?Carbon
    {
        switch ($modelClass) {
            case AmazonFbaReportInventoryLedger::class:
                $startDate = AmazonFbaReportInventoryLedger::query()
                    ->select('date')
                    ->latest('date')
                    ->pluck('date')
                    ->first();
                break;
            case AmazonFbaReportCustomerReturn::class:
                $startDate = AmazonFbaReportCustomerReturn::query()
                    ->select('return_date')
                    ->latest('return_date')
                    ->pluck('return_date')
                    ->first();
                break;
            case AmazonFbaReportShipment::class:
                $startDate = AmazonFbaReportShipment::query()
                    ->select('reporting_date')
                    ->latest('reporting_date')
                    ->pluck('reporting_date')
                    ->first();
                break;
            case AmazonFbaReportRemovalOrder::class:
                $startDate = AmazonFbaReportRemovalOrder::query()
                    ->select('last_updated_date')
                    ->latest('last_updated_date')
                    ->pluck('last_updated_date')
                    ->first();
                break;
            case AmazonFbaReportRemovalShipment::class:
                $startDate = AmazonFbaReportRemovalShipment::query()
                    ->select('shipment_date')
                    ->latest('shipment_date')
                    ->pluck('shipment_date')
                    ->first();
                break;
            case AmazonReportSettlementData::class:
                $startDate = AmazonReportSettlementData::query()
                    ->select('posted_date_time')
                    ->latest('posted_date_time')
                    ->pluck('posted_date_time')
                    ->first();
                $startDate = max($startDate, Carbon::now()->subDays(90));
                break;
            case AmazonProduct::class:
            case AmazonFbaReportInventory::class:
                $startDate = Carbon::now()->subDay();
                break;
            default:
                return throw new Exception('Unhandled modelClass: '.$modelClass);
        }

        return $startDate ?
            Carbon::parse($startDate)->addSecond() :
            Helpers::setting(Setting::KEY_INVENTORY_START_DATE);
    }

    private function sanitizeSettlementReportRecords(array $records, AmazonFinancialEventGroup $financialEventGroup): array
    {
        return array_map(function ($record) use ($financialEventGroup) {
            unset(
                $record['settlement_start_date'],
                $record['settlement_end_date'],
                $record['deposit_date'],
                $record['total_amount'],
                $record['currency'],
            );
            $record['amazon_financial_event_group_id'] = $financialEventGroup->id;

            return $record;
        }, array_filter($records, function ($record) {
            return ! empty($record['settlement_id']);
        }));
    }

    public function getUnprocessedReports(): EloquentCollection
    {
        return AmazonReport::query()
            ->where('processingStatus', ReportProcessingStatusEnum::DONE)
            ->whereNotNull('filename')
            ->whereNull('processed_at')
            ->get();
    }

    public function wasReportTypeProcessedForDate(AmazonReportTypeEnum $reportType, Carbon $date): bool
    {
        $query = AmazonReport::query()
            ->where('reportType', $reportType)
            ->where('processingStatus', ReportProcessingStatusEnum::DONE)
            ->whereNotNull('processed_at');

        $queryDataStartBeforeOrOnDate = $query
            ->clone()
            ->where('dataStartTime', '<=', $date);

        $queryDataEndAfterOrOnDate = $query
            ->clone()
            ->where('dataEndTime', '>=', $date->copy()->endOfDay());

        return $queryDataStartBeforeOrOnDate->exists() && $queryDataEndAfterOrOnDate->exists();
    }

    public function getLastRequestedForType(AmazonReportTypeEnum $reportType, AmazonIntegrationInstance $amazonIntegrationInstance): ?AmazonReportRequest
    {
        return AmazonReportRequest::query()
            ->where('reportType', $reportType)
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->latest()
            ->first();
    }

    public function getLastReportInProcessForType(AmazonReportTypeEnum $reportType, AmazonIntegrationInstance $amazonIntegrationInstance): ?AmazonReport
    {
        return AmazonReport::query()
            ->where('reportType', $reportType)
            ->where(function ($query) {
                $query->whereNull('processingStatus');
                $query->orWhereNotIn('processingStatus', ReportProcessingStatusEnum::COMPLETE_STATUSES);
            })
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->latest()
            ->first();
    }

    /**
     * @throws Exception
     */
    public function getLastReportForType(AmazonReportTypeEnum $reportType, AmazonIntegrationInstance $amazonIntegrationInstance): ?AmazonReport
    {
        $recordClass = $this->getModelClassFromReportType($reportType);

        $recordsTable = app($recordClass)->getTable();

        return AmazonReport::query()
            ->selectRaw('amazon_reports.*, COUNT('.$recordsTable.'.id) as recordsCount')
            ->leftJoin($recordsTable, $recordsTable.'.amazon_report_id', '=', 'amazon_reports.id')
            ->where('amazon_reports.reportType', $reportType)
            ->whereIn('amazon_reports.processingStatus', ReportProcessingStatusEnum::COMPLETE_STATUSES)
            ->where('amazon_reports.integration_instance_id', $amazonIntegrationInstance->id)
            ->groupBy('amazon_reports.id')
            ->latest()
            ->first();
    }

    /**
     * @throws Exception
     */
    public function getLastDataDateForType(AmazonReportTypeEnum $reportType, AmazonIntegrationInstance $amazonIntegrationInstance): ?Carbon
    {
        $lastProcessedDate = AmazonReport::query()
            ->where('reportType', $reportType)
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereNotNull('processed_at')
            ->max('dataEndTime');

        return $lastProcessedDate ? Carbon::parse($lastProcessedDate) : null;
    }

    public function getOverlappingReport(AmazonReportRequest $amazonReportRequest): ?AmazonReport
    {
        return AmazonReport::query()
            ->where('integration_instance_id', $amazonReportRequest->integration_instance_id)
            ->where('reportType', $amazonReportRequest->reportType)
            ->where('dataStartTime', '<=', $amazonReportRequest->dataEndTime)
            ->where('dataEndTime', '>=', $amazonReportRequest->dataStartTime)
            ->where(function (Builder $query) {
                $query->whereNull('processingStatus');
                $query->orWhereIn('processingStatus', ReportProcessingStatusEnum::NON_FATAL_STATUSES);

            })
            ->where('marketplaceIds', json_encode($amazonReportRequest->marketplaceIds))
            ->first();
    }

    public function getReportsForTypeAndStatus(
        AmazonIntegrationInstance $amazonIntegrationInstance,
        AmazonReportTypeEnum $reportType,
        ReportProcessingStatusEnum $status
    ): EloquentCollection
    {
        return AmazonReport::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('reportType', $reportType)
            ->where('processingStatus', $status)
            ->get();
    }

    public function getCustomerReturnFromLpn(AmazonIntegrationInstance $amazonIntegrationInstance, string $lpn): ?AmazonFbaReportCustomerReturn
    {
        return AmazonFbaReportCustomerReturn::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('license_plate_number', $lpn)
            ->first();
    }
}
