<?php

namespace Modules\Amazon\Repositories;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProductRepository;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\Warehouse;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonFnskuGenerationData;
use Modules\Amazon\Entities\AmazonFbaReportInventory;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;

class AmazonProductRepository extends AbstractSalesChannelProductRepository
{
    public function getModelClassName(): string
    {
        return AmazonProduct::class;
    }

    public function getProductsMissingCatalogItem(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return AmazonProduct::distinct('asin1')
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('was_catalog_data_sync_attempted', false)->get();
    }

    public function getProductListingsInventory(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return ProductListing::where('sales_channel_id', $amazonIntegrationInstance->salesChannel->id)
            ->where(function ($query) {
                $query->orWhereNotNull('quantity');
            })
            ->where('is_fba', false)
            ->get();
    }

    public function getProductListingsPrices(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return ProductListing::where('sales_channel_id', $amazonIntegrationInstance->salesChannel->id)
            ->where(function ($query) {
                $query->orWhereNotNull('price');
            })
            ->get();
    }

    public function getProductListingsMappingsInfo(Request $request, AmazonIntegrationInstance $amazonIntegrationInstance): array
    {
        $query = AmazonProduct::with(['productListing'])
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where(function ($builder) use ($request) {
                $builder->filter($request);
            })
            ->addRelations($request)
            ->addCustomColumns($request);

        $total = $query->count();
        $mapped = $query->whereHas('productListing')->count();

        return [
            'mapped' => $mapped,
            'total' => $total,
        ];
    }

    public function extractRealTimeInventory(Collection $data, string $type = 'total', AmazonIntegrationInstance|IntegrationInstance|null $amazonIntegrationInstance = null)
    {
        if ($amazonIntegrationInstance) {
            $data
                ->where('integration_instance_id', $amazonIntegrationInstance->id);
        }

        switch ($type) {
            case 'total':
                return
                    $data->sum('afn_fulfillable_quantity') +
                    $data->sum('afn_reserved_quantity') +
                    $data->sum('afn_inbound_shipped_quantity') +
                    $data->sum('afn_inbound_receiving_quantity');
            case 'available':
                return $data->sum('afn_fulfillable_quantity');
            case 'reserved':
                return $data->sum('afn_reserved_quantity');
            case 'inbound':
                return
                    $data->sum('afn_inbound_shipped_quantity') +
                    $data->sum('afn_inbound_receiving_quantity');
        }
    }

    public function getFBAReportInventoryForIntegrationInstances(int $productId, array|Collection $integrationInstanceIds = []): EloquentCollection|array
    {
        if(empty($integrationInstanceIds)) {
            $integrationInstanceIds = Warehouse::query()->where('type', Warehouse::TYPE_AMAZON_FBA)->pluck('integration_instance_id');
        }

        return AmazonFbaReportInventory::with(['integrationInstance.warehouse'])
            ->whereHas('amazonProduct', function (Builder $builder) use ($productId) {
                $builder->whereHas('productListing', function (Builder $builder) use ($productId) {
                    $builder->where('product_id', $productId);
                });
            })
            ->whereIn('integration_instance_id', $integrationInstanceIds)
            ->get();
    }

    public function getUnmappedProductsCount(IntegrationInstanceInterface $integrationInstance): int
    {
        return AmazonProduct::whereDoesntHave('productListing')
            ->where('integration_instance_id', $integrationInstance->id)
            ->count();
    }

    public function findAmazonProductThroughInitialInventory(AmazonIntegrationInstance $integrationInstance, AmazonFnskuGenerationData $fnskuGenerationData): EloquentCollection
    {
        return AmazonProduct::query()
            ->join('amazon_fba_initial_inventory', function (JoinClause $join) {
                $join->on('amazon_fba_initial_inventory.integration_instance_id', 'amazon_products.integration_instance_id');
                $join->on('amazon_fba_initial_inventory.msku', 'amazon_products.seller_sku');
            })
            ->join('product_listings', function (JoinClause $join) {
                $join->on('product_listings.document_id', 'amazon_products.id');
                $join->where('product_listings.document_type', AmazonProduct::class);
            })
            ->join('products', 'products.id', 'product_listings.product_id')
            ->where('amazon_products.integration_instance_id', $integrationInstance->id)
            ->where('amazon_fba_initial_inventory.integration_instance_id', $fnskuGenerationData->integration_instance_id)
            ->where('amazon_fba_initial_inventory.fnsku', $fnskuGenerationData->fnsku)
            ->where('amazon_fba_initial_inventory.location', $fnskuGenerationData->location)
            ->where('amazon_fba_initial_inventory.disposition', $fnskuGenerationData->disposition)
            ->groupBy('products.id')
            ->get();
    }

    public function findAmazonProductsThroughAsin(AmazonIntegrationInstance $integrationInstance, AmazonFnskuGenerationData $fnskuGenerationData): EloquentCollection
    {
        return AmazonProduct::query()
            ->where('asin1', $fnskuGenerationData->asin)
            ->where('integration_instance_id', $integrationInstance->id)
            ->join('product_listings', function (JoinClause $join) {
                $join->on('product_listings.document_id', 'amazon_products.id');
                $join->where('product_listings.document_type', AmazonProduct::class);
            })
            ->join('products', 'products.id', 'product_listings.product_id')
            ->groupBy('products.id')
            ->get();
    }
}
