<?php

namespace Modules\Amazon\Repositories;

use App\Abstractions\AbstractRepository;
use App\Models\SalesOrderFulfillment;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonFulfillmentOrderData;
use Modules\Amazon\Data\AmazonOutboundFulfillmentOrderData;
use Modules\Amazon\Entities\AmazonFulfillmentOrder;
use Modules\Amazon\Entities\AmazonFulfillmentOrderItem;
use Modules\Amazon\Entities\AmazonFulfillmentShipment;
use Modules\Amazon\Entities\AmazonFulfillmentShipmentItem;
use Modules\Amazon\Entities\AmazonFulfillmentShipmentPackage;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Spatie\LaravelData\DataCollection;

class AmazonOutboundFulfillmentRepository extends AbstractRepository
{
    public function saveOutbound(AmazonIntegrationInstance $integrationInstance, SalesOrderFulfillment $salesOrderFulfillment, AmazonFulfillmentOrderData $order): void
    {
        $insertData = [
            'integration_instance_id' => $integrationInstance->id,
            'sales_order_fulfillment_id' => $salesOrderFulfillment->id,
            'json_object' => $order->except('items')->toArray(),
        ];

        $amazonFulfillmentOrder = AmazonFulfillmentOrder::create($insertData);

        $this->saveLines($amazonFulfillmentOrder, $order->items);
    }

    private function saveLines(AmazonFulfillmentOrder $amazonFulfillmentOrder, DataCollection $items): void
    {
        $insertData = $items->toCollection()->map(function ($item) use ($amazonFulfillmentOrder) {
            return [
                'amazon_fulfillment_order_id' => $amazonFulfillmentOrder->id,
                'json_object' => $item->toJson(),
            ];
        })->toArray();

        AmazonFulfillmentOrderItem::insert($insertData);
    }

    public function getStartDateForNew(AmazonIntegrationInstance $integrationInstance)
    {
        $startDate = AmazonFulfillmentOrder::where('integration_instance_id', $integrationInstance->id)
            ->where(function (Builder $query) {
                $query->whereNull('fulfillmentOrderStatus')
                    ->orWhereNotIn('fulfillmentOrderStatus', [
                        'Cancelled',
                        'Complete',
                        'Unfulfillable',
                        'Invalid'
                    ]);
            })
            // TODO: Should be statusUpdatedDate but need to change the response on the original request to save the data from the api response instead of the request
            //  However, this https://developer-docs.amazon.com/sp-api/docs/fulfillment-outbound-api-v2020-07-01-reference#createfulfillmentorderresponse suggests that
            //  the response does not contain any pertinent information... needs testing
            ->oldest('displayableOrderDate')
            ->pluck('displayableOrderDate')
            ->first();

        if ($startDate) {
            $date = Carbon::parse($startDate)->addSecond();
        } else {
            $date = Carbon::parse($integrationInstance->fbaInventoryTrackingStartDate());
        }

        return $date;
    }

//    public function saveForIntegration(AmazonIntegrationInstance $integrationInstance, Collection $data): void
//    {
//        $data = $data->map(function ($item) use ($integrationInstance) {
//            return [
//                'integration_instance_id' => $integrationInstance->id,
//                'json_object' => json_encode($item->json_object),
//            ];
//        });
//
//        AmazonFulfillmentOrder::upsert($data->toArray(), ['integration_instance_id', AmazonFulfillmentOrder::getUniqueField()]);
//    }

    public function saveStatusUpdates(AmazonIntegrationInstance $integrationInstance, DataCollection $data): void
    {
        $data = $data->toCollection();

        AmazonFulfillmentOrder::query()
            ->whereIn('sellerFulfillmentOrderId', $data->pluck('sellerFulfillmentOrderId'))
            ->where('integration_instance_id', $integrationInstance->id)
            ->each(function (AmazonFulfillmentOrder $amazonFulfillmentOrder) use ($data) {
                $dataForOrder = $data->firstWhere('sellerFulfillmentOrderId', $amazonFulfillmentOrder->sellerFulfillmentOrderId);
                $json_object = $amazonFulfillmentOrder->json_object;
                $json_object['receivedDate'] = $dataForOrder->receivedDate;
                $json_object['fulfillmentOrderStatus'] = $dataForOrder->fulfillmentOrderStatus;
                $json_object['statusUpdatedDate'] = $dataForOrder->statusUpdatedDate;
                $amazonFulfillmentOrder->json_object = $json_object;
                $amazonFulfillmentOrder->update();
            });
    }

    public function getOrdersNeedingTracking(AmazonIntegrationInstance $integrationInstance): Collection
    {
        return AmazonFulfillmentOrder::where('integration_instance_id', $integrationInstance->id)
            ->whereIn('fulfillmentOrderStatus', ['Complete', 'Processing'])
            ->whereDoesntHave('amazonFulfillmentShipments.amazonFulfillmentShipmentPackages')
            ->get();
    }

    public function saveFulfillmentOrder(AmazonFulfillmentOrder $amazonFulfillmentOrder, AmazonOutboundFulfillmentOrderData $data): void
    {
        $fulfillmentOrder = $data->json_object['fulfillmentOrder'];
        $fulfillmentOrderItems = $data->json_object['fulfillmentOrderItems'];
        $fulfillmentShipments = $data->json_object['fulfillmentShipments'];
        $amazonFulfillmentOrder->json_object = $fulfillmentOrder;
        $amazonFulfillmentOrder->update();

        foreach ($fulfillmentOrderItems as $fulfillmentOrderItem) {
            AmazonFulfillmentOrderItem::where('amazon_fulfillment_order_id', $amazonFulfillmentOrder->id)
                ->where('sellerFulfillmentOrderItemId', $fulfillmentOrderItem['sellerFulfillmentOrderItemId'])
                ->updateOrCreate([
                    'amazon_fulfillment_order_id' => $amazonFulfillmentOrder->id,
                ], [
                    'json_object' => $fulfillmentOrderItem,
                ]);
        }

        foreach ($fulfillmentShipments as $fulfillmentShipment) {
            $amazonFulfillmentShipment = AmazonFulfillmentShipment::where('amazon_fulfillment_order_id', $amazonFulfillmentOrder->id)
                ->where('amazonShipmentId', $fulfillmentShipment['amazonShipmentId'])
                ->updateOrCreate([
                    'amazon_fulfillment_order_id' => $amazonFulfillmentOrder->id,
                ], [
                    'json_object' => collect($fulfillmentShipment)->except(['fulfillmentShipmentItems', 'fulfillmentShipmentPackage'])->toArray(),
                ]);

            foreach ($fulfillmentShipment['fulfillmentShipmentItem'] as $fulfillmentShipmentItem) {
                AmazonFulfillmentShipmentItem::where('amazon_fulfillment_shipment_id', $amazonFulfillmentShipment->id)
                    ->where('sellerFulfillmentOrderItemId', $fulfillmentShipmentItem['sellerFulfillmentOrderItemId'])
                    ->updateOrCreate([
                        'amazon_fulfillment_shipment_id' => $amazonFulfillmentShipment->id,
                    ], [
                        'json_object' => $fulfillmentShipmentItem,
                    ]);
            }

            foreach ($fulfillmentShipment['fulfillmentShipmentPackage'] as $fulfillmentShipmentPackage) {
                AmazonFulfillmentShipmentPackage::where('amazon_fulfillment_shipment_id', $amazonFulfillmentShipment->id)
                    ->where('packageNumber', $fulfillmentShipmentPackage['packageNumber'])
                    ->updateOrCreate([
                        'amazon_fulfillment_shipment_id' => $amazonFulfillmentShipment->id,
                    ], [
                        'json_object' => $fulfillmentShipmentPackage
                    ]);
            }
        }
    }

    public function getOrphanFulfillment(AmazonIntegrationInstance $integrationInstance, SalesOrderFulfillment $salesOrderFulfillment): ?AmazonFulfillmentOrder
    {
        return AmazonFulfillmentOrder::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->where('sellerFulfillmentOrderId', $salesOrderFulfillment->fulfillment_number)
            ->whereNull('sales_order_fulfillment_id')
            ->where('fulfillmentOrderStatus', '!=', 'Cancelled')
            ->first();
    }
}