<?php

namespace Modules\Amazon\Repositories;

use Modules\Amazon\Data\AmazonOrderItemData;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonOrderItem;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Throwable;

class AmazonOrderItemRepository
{
    protected AmazonOrderRepository $amazonOrderRepository;

    public function __construct()
    {
        $this->amazonOrderRepository = app(AmazonOrderRepository::class);
    }

    public function save(
        int $orderId,
        #[DataCollectionOf(AmazonOrderItemData::class)] DataCollection $data
    ): void
    {
        $data = $data->toCollection();
        $upsertData = $data->map(function (AmazonOrderItemData $item) use ($orderId) {
            return [
                AmazonOrderItem::getParentRelationId() => $orderId,
                'json_object' => json_encode($item->json_object),
            ];
        })->toArray();

        AmazonOrderItem::upsert($upsertData, [AmazonOrderItem::getParentRelationId(), AmazonOrderItem::getTableUniqueId()]);

        // Clear error log
        AmazonOrder::where('id', $orderId)->update(['error_log' => null]);
    }

    public function getFromOrderItemId(int $amazonOrderId, string $orderItemId): ?AmazonOrderItem
    {
        return AmazonOrderItem::where('amazon_order_id', $amazonOrderId)
            ->where('orderItemId', $orderItemId)
            ->first();
    }

    /**
     * @throws Throwable
     */
    public function getFromAsin(AmazonOrder $amazonOrder, string $asin): ?AmazonOrderItem
    {
        $amazonOrderItemQuery = AmazonOrderItem::query()->where('amazon_order_id', $amazonOrder->id)
            ->where('ASIN', $asin);

        throw_if($amazonOrderItemQuery->count() > 1, 'More than one order item found for order '.$amazonOrder->AmazonOrderId.' and ASIN '.$asin);

        /** @var AmazonOrderItem $amazonOrderItem */
        $amazonOrderItem = $amazonOrderItemQuery->first();

        return $amazonOrderItem;
    }

//    public function getFromSkuSalesOrderId(int $skuSalesOrderId): AmazonOrder
//    {
//        return AmazonOrder::with([])
//            ->whereHas('salesOrder', function ($query) use ($skuSalesOrderId) {
//                $query->where('id', $skuSalesOrderId);
//            })
//            ->firstOrFail();
//    }

//    /**
//     * @throws Throwable
//     */
//    public function getFromAmazonProduct(AmazonOrder $amazonOrder, AmazonProduct $amazonProduct): ?AmazonOrderItem
//    {
//        $amazonOrderItemQuery = AmazonOrderItem::query()->where('amazon_order_id', $amazonOrder->id)
//            ->where('SellerSKU', $amazonProduct->seller_sku);
//
//        throw_if($amazonOrderItemQuery->count() > 1, 'More than one order item found for order '.$amazonOrder->AmazonOrderId.' and product '.$amazonProduct->seller_sku);
//
//        /** @var AmazonOrderItem $amazonOrderItem */
//        $amazonOrderItem = $amazonOrderItemQuery->first();
//
//        return $amazonOrderItem;
//    }
}
