<?php

namespace Modules\Amazon\Repositories;

use App\Abstractions\Integrations\AbstractIntegrationRepository;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonNewFbaInboundShipmentData;
use Modules\Amazon\Entities\AmazonFbaInboundShipFromMapping;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonNewFbaInboundPlan;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipment;
use Modules\Amazon\Enums\Entities\AmazonNewFbaInboundShipmentStatusEnum;

class AmazonNewFbaInboundShipmentRepository extends AbstractIntegrationRepository
{
    public function getModelClassName(): string
    {
        return AmazonNewFbaInboundShipment::class;
    }

    public function getActiveItemlessShipmentIds(int $limit = 300): Collection
    {
        return AmazonNewFbaInboundShipment::query()
            ->whereIn('status', AmazonNewFbaInboundShipmentStatusEnum::STATUSES_ACTIVE)
            ->whereDoesntHave('amazonFbaInboundShipmentItems')
            ->limit($limit)
            ->orderBy('created_at')
            ->pluck('shipmentId');
    }

    public function getFromShipmentId(AmazonIntegrationInstance $amazonIntegrationInstance, string $shipmentId): ?AmazonNewFbaInboundShipment
    {
        /** @var AmazonNewFbaInboundShipment $shipment */
        $shipment = AmazonNewFbaInboundShipment::with('amazonFbaInboundPlan')
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('shipmentId', $shipmentId)
            ->first();

        return $shipment;
    }

    public function getFromShipmentConfirmationId(AmazonIntegrationInstance $amazonIntegrationInstance, string $shipmentConfirmationId): ?AmazonNewFbaInboundShipment
    {
        /** @var AmazonNewFbaInboundShipment $shipment */
        $shipment = AmazonNewFbaInboundShipment::with('amazonFbaInboundPlan')
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('shipmentConfirmationId', $shipmentConfirmationId)
            ->first();

        return $shipment;
    }

    public function saveShipment(AmazonNewFbaInboundPlan $plan, AmazonNewFbaInboundShipmentData $data): AmazonNewFbaInboundShipment
    {
        if ($existingShipment = $this->getFromShipmentId($plan->integrationInstance, $data->json_object['shipmentId'])) {
            return $existingShipment;
        }

        $shipment = new AmazonNewFbaInboundShipment();
        $shipment->fill($data->toArray());
        $shipment->amazon_new_fba_inbound_plan_id = $plan->id;
        $shipment->integration_instance_id = $plan->integration_instance_id;
        $shipment->save();

        return $shipment;
    }

    public function checkLegacyShipment(IntegrationInstanceInterface|AmazonIntegrationInstance $amazonIntegrationInstance, string $shipmentConfirmationId): ?AmazonFbaInboundShipment
    {
        return AmazonFbaInboundShipment::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('ShipmentId', $shipmentConfirmationId)
            ->first();
    }

    public function getLegacyShipmentDuplicates(IntegrationInstanceInterface|AmazonIntegrationInstance $amazonIntegrationInstance): Collection
    {
        return AmazonNewFbaInboundShipment::query()
            ->join('amazon_fba_inbound_shipments', function (JoinClause $join) {
                $join->on('amazon_fba_inbound_shipments.ShipmentId', '=', 'amazon_new_fba_inbound_shipments.shipmentConfirmationId')
                    ->whereColumn('amazon_fba_inbound_shipments.integration_instance_id', '=', 'amazon_new_fba_inbound_shipments.integration_instance_id');
            })
            ->where('amazon_new_fba_inbound_shipments.integration_instance_id', $amazonIntegrationInstance->id)
            ->get();
    }

    public function getUnlinkedShipments(AmazonIntegrationInstance $amazonIntegrationInstance): Collection
    {
        return AmazonNewFbaInboundShipment::query()
            ->select(['id', 'shipmentConfirmationId'])
            ->selectRaw('CONCAT(shipmentConfirmationId, " (", sourceName, ") - ", name) as description')
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereHas('amazonFbaInboundShipmentItems')
            ->whereNull('sku_link_id')
            ->whereIn('status', AmazonNewFbaInboundShipmentStatusEnum::STATUSES_ACTIVE)
            ->get();
    }

    public function getUnprocessedCancellations(AmazonIntegrationInstance $amazonIntegrationInstance): Collection
    {
        return AmazonNewFbaInboundShipment::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereNotNull('sku_link_id')
            ->whereIn('status', [
                AmazonNewFbaInboundShipmentStatusEnum::CANCELLED,
                AmazonNewFbaInboundShipmentStatusEnum::DELETED,
            ])
            ->get();
    }

    public function getUnprocessedWithMappingOfType(AmazonIntegrationInstance $amazonIntegrationInstance, string $type): Collection
    {
        $mappings = AmazonFbaInboundShipFromMapping::where('link_type', $type)
            ->get();

        return AmazonNewFbaInboundShipment::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereHas('amazonFbaInboundShipmentItems')
            ->whereNull('sku_link_id')
            ->where('is_before_initial_count', false)
            ->whereIn('sourceName', $mappings->pluck('name'))
            ->get();
    }
}
