<?php

namespace Modules\Amazon\Repositories;

use App\Abstractions\Integrations\AbstractIntegrationRepository;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonNewFbaInboundPlan;
use Modules\Amazon\Enums\Entities\FbaInboundPlanStatusEnum;

class AmazonNewFbaInboundPlanRepository extends AbstractIntegrationRepository
{
    public function getModelClassName(): string
    {
        return AmazonNewFbaInboundPlan::class;
    }

    public function getActiveItemlessPlanIds(int $limit = 30): Collection
    {
        return AmazonNewFbaInboundPlan::query()
            ->whereIn('status', FbaInboundPlanStatusEnum::STATUSES_ACTIVE)
            ->whereDoesntHave('amazonNewFbaInboundPlanItems')
            ->whereNot('name', '')
            ->limit($limit)
            ->orderBy('created_at')
            ->pluck('inboundPlanId');
    }

    public function getLastUpdateDate(): ?string
    {
        return AmazonNewFbaInboundPlan::query()->latest('lastUpdatedAt')->pluck('lastUpdatedAt')->first();
    }

    public function getFromPlanId(AmazonIntegrationInstance $amazonIntegrationInstance, string $inboundPlanId): ?AmazonNewFbaInboundPlan
    {
        /** @var AmazonNewFbaInboundPlan $amazonFbaInboundPlan */
        $amazonFbaInboundPlan = AmazonNewFbaInboundPlan::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('inboundPlanId', $inboundPlanId)
            ->first();

        return $amazonFbaInboundPlan;
    }

    public function getShipmentsNeedingDetails(AmazonIntegrationInstance $integrationInstance): array
    {
        return AmazonNewFbaInboundPlan::query()
            ->select(['inboundPlanId', 'shipments'])
            ->whereNotNull('shipments')
            ->where('integration_instance_id', $integrationInstance->id)
            ->where(function (Builder $query) {
                $query->whereDoesntHave('amazonNewFbaInboundShipments')
                    ->orWhereHas('amazonNewFbaInboundShipments', function ($query) {
                        $query->havingRaw('COUNT(*) < JSON_LENGTH(shipments)');
                    });
            })
            ->get()
            ->reduce(function ($carry, AmazonNewFbaInboundPlan $plan) {
                foreach ($plan->shipments as $shipment) {
                    $carry[$plan->inboundPlanId][] = $shipment['shipmentId'];
                }
                return $carry;
            }, []);
    }

    public function getPlansNeedingDetail(): array
    {
        return AmazonNewFbaInboundPlan::whereDoesntHave('amazonNewFbaInboundShipments')
            ->select(['inboundPlanId'])
            ->whereIn('status', FbaInboundPlanStatusEnum::STATUSES_ACTIVE)
            ->get()
            ->pluck('inboundPlanId')
            ->toArray();
    }

    public function getExistingPlanIds(AmazonIntegrationInstance $integrationInstance): Collection
    {
        return AmazonNewFbaInboundPlan::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->pluck('inboundPlanId');
    }

    public function insertSummaries(AmazonIntegrationInstance $integrationInstance, Collection $data): void
    {
        $existingPlanIds = $this->getExistingPlanIds($integrationInstance);

        $data = $data->filter(function ($plan) use ($existingPlanIds) {
            return !$existingPlanIds->contains($plan->json_object['inboundPlanId']);
        })->values();

        $data = $data->map(function ($plan) use ($integrationInstance) {
            return [
                'integration_instance_id' => $integrationInstance->id,
                'json_object' => json_encode($plan->json_object),
            ];
        });

        AmazonNewFbaInboundPlan::insert($data->toArray());
    }
}
