<?php

namespace Modules\Amazon\Repositories;

use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferShipmentLine;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Data\AmazonLedgerSummaryAdjustmentData;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedgerSummary;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonLedgerSummaryAdjustment;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Spatie\LaravelData\DataCollection;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;
use Throwable;

class AmazonLedgerSummaryRepository
{
    private AmazonFnskuRepository $fnskus;
    private AmazonReportRepository $reports;

    public function __construct()
    {
        $this->fnskus = app(AmazonFnskuRepository::class);
        $this->reports = app(AmazonReportRepository::class);
    }

    public static function getFieldByEventType(FbaInventoryLedgerReportEventTypeEnum $eventType): string
    {
        return match ($eventType) {
            FbaInventoryLedgerReportEventTypeEnum::Receipts => 'receipts',
            FbaInventoryLedgerReportEventTypeEnum::Shipments => 'customer_shipments',
            FbaInventoryLedgerReportEventTypeEnum::CustomerReturns => 'customer_returns',
            FbaInventoryLedgerReportEventTypeEnum::VendorReturns => 'vendor_returns',
            FbaInventoryLedgerReportEventTypeEnum::WarehouseTransfers => 'warehouse_transfer_in_out',
            FbaInventoryLedgerReportEventTypeEnum::Adjustments => 'adjustments',
        };
    }

    /**
     * @throws Exception
     */
    public function getUnreconciled(AmazonIntegrationInstance $integrationInstance): EloquentCollection
    {
        $lastLedgerProcessedDate = $this->reports->getLastDataDateForType(AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER, $integrationInstance);

        return AmazonFbaReportInventoryLedgerSummary::with(['inventoryAdjustments'])
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNull('reconciled_at')
            /*
             * Ledger summaries can be processed if:
             * - They don't have any unreconciled ledgers prior to the ledger summary event_datetime and...
             * - They are prior to the last ledger processed date
             */
            ->whereDoesntHave('ledgers', function ($query) {
                $query->whereNull('reconciled_at');
                $query->whereColumn('amazon_fba_report_inventory_ledger_summaries.event_datetime', '<=', 'amazon_fba_report_inventory_ledger.event_datetime');
            })
            ->where('event_datetime', '<=', $lastLedgerProcessedDate)
            ->orderBy('event_datetime')
            ->get();
    }

    /**
     * @throws Exception
     */
    public function getUnreconciledForFnskuProduct(AmazonIntegrationInstance $integrationInstance, AmazonFbaReportInventoryLedgerSummary $ledgerSummary): EloquentCollection
    {
        $lastLedgerProcessedDate = $this->reports->getLastDataDateForType(AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER, $integrationInstance);

        if (!$lastLedgerProcessedDate) {
            return new EloquentCollection();
        }

        return AmazonFbaReportInventoryLedgerSummary::with(['inventoryAdjustments'])
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereHas('amazonFnskuProduct', function ($query) use ($ledgerSummary) {
                $query->where('id', $ledgerSummary->amazonFnskuProduct->id)
                /*
                 * If unreconciled ledgers exist before or on the summary date, we can not process the summary.
                 * Edge case: If the summary date is after the last ledger processed date, we have to wait for the
                 * ledgers to catch up before processing the summary.  So don't include summaries after the last
                 * reconciled date
                 */
                ->whereDoesntHave('amazonFbaReportInventoryLedgers', function ($query) {
                    $query->whereNull('reconciled_at');
                    $query->whereColumn('amazon_fba_report_inventory_ledger.event_datetime', '<=', 'amazon_fba_report_inventory_ledger_summaries.event_datetime');
                });
            })
            ->where('event_datetime', '<', $lastLedgerProcessedDate)
            ->whereNull('reconciled_at')
            ->orderBy('event_datetime')
            ->get();
    }

    public function getReconciled(AmazonIntegrationInstance $integrationInstance): EloquentCollection
    {
        return AmazonFbaReportInventoryLedgerSummary::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNotNull('reconciled_at')
            ->orderBy('event_datetime', 'desc')
            ->get();
    }

    public function getReconciledForFnskuProduct(AmazonIntegrationInstance $integrationInstance, AmazonFbaReportInventoryLedgerSummary $ledgerSummary): EloquentCollection
    {
        return AmazonFbaReportInventoryLedgerSummary::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereHas('amazonFnskuProduct', function ($query) use ($ledgerSummary) {
                $query->where('id', $ledgerSummary->amazonFnskuProduct->id)
                /*
                 * If reconciled ledgers exist after the summary date, we can not process the summary.
                 */
                ->whereDoesntHave('amazonFbaReportInventoryLedgers', function ($query) {
                    $query->whereNotNull('reconciled_at');
                    $query->whereColumn('amazon_fba_report_inventory_ledger.event_datetime', '>', 'amazon_fba_report_inventory_ledger_summaries.event_datetime');
                });
            })
            ->whereNotNull('reconciled_at')
            ->orderBy('event_datetime', 'desc')
            ->get();
    }

    public function getSummariesWithAdjustmentsForValuation(AmazonIntegrationInstance $integrationInstance): EloquentCollection
    {
        return QueryBuilder::for(AmazonFbaReportInventoryLedgerSummary::class)
            ->with(['amazonFnskuProduct', 'inventoryAdjustments.inventoryMovements'])
            ->where('integration_instance_id', $integrationInstance->id)
            ->allowedFilters([
                AllowedFilter::scope('before_event_date')
            ])
            ->whereHas('amazonFnskuProduct')
            ->whereHas('inventoryAdjustments')
            ->fnskuSort()
            ->get();
    }

    public function getSummariesForValuationDate(AmazonIntegrationInstance $integrationInstance, Carbon $date): EloquentCollection
    {
        return AmazonFbaReportInventoryLedgerSummary::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->where('event_datetime', $date)
            ->get();
    }

    public function getOldestUnreconciledDate(AmazonIntegrationInstance $integrationInstance): ?string
    {
        return AmazonFbaReportInventoryLedgerSummary::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNull('reconciled_at')
            ->min('date');
    }

    public function getLastReconciledDate(AmazonIntegrationInstance $integrationInstance): ?string
    {
        return AmazonFbaReportInventoryLedgerSummary::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNotNull('reconciled_at')
            ->max('reconciled_at');
    }

    public function getPreviousSummary(AmazonFbaReportInventoryLedgerSummary $summary): AmazonFbaInitialInventory|AmazonFbaReportInventoryLedgerSummary|null
    {
        /** @var AmazonFbaReportInventoryLedgerSummary $previousSummary */
        $previousSummary = AmazonFbaReportInventoryLedgerSummary::query()
            ->where('integration_instance_id', $summary->integration_instance_id)
            ->where('fnsku', $summary->fnsku)
            ->where('disposition', $summary->disposition)
            ->where('location', $summary->location)
            ->where('event_datetime', $summary->event_datetime->subDay())
            ->first();

        if ($previousSummary || $summary->event_datetime != Carbon::parse($summary->integrationInstance->integration_settings['fba_inventory_tracking_start_date'])) {
            return $previousSummary;
        }

        /** @var AmazonFbaInitialInventory $initialInventory */
        $initialInventory = AmazonFbaInitialInventory::query()
            ->where('integration_instance_id', $summary->integration_instance_id)
            ->where('fnsku', $summary->fnsku)
            ->where('disposition', $summary->disposition)
            ->where('location', $summary->location)
            ->first();

        if ($initialInventory) {
            return $initialInventory;
        }

        return null;
    }

    public function getNextSummary(AmazonFbaReportInventoryLedgerSummary $summary): ?AmazonFbaReportInventoryLedgerSummary
    {
        /** @var AmazonFbaReportInventoryLedgerSummary $ledgerSummary */
        $ledgerSummary = AmazonFbaReportInventoryLedgerSummary::query()
            ->where('integration_instance_id', $summary->integration_instance_id)
            ->where('fnsku', $summary->fnsku)
            ->where('disposition', $summary->disposition)
            ->where('location', $summary->location)
            ->where('event_datetime', '=', $summary->event_datetime->addDay())
            ->first();

        return $ledgerSummary;
    }

    public function linkAdjustmentsToLedgerSummary(DataCollection $data): void
    {
        /** @var AmazonLedgerSummaryAdjustmentData $amazonLedgerSummaryAdjustmentData */
        foreach ($data as $amazonLedgerSummaryAdjustmentData)
        {
            $ledgerSummaryAdjustmentData = new AmazonLedgerSummaryAdjustment([
                'amazon_fba_report_inventory_ledger_summary_id' => $amazonLedgerSummaryAdjustmentData->amazon_fba_report_inventory_ledger_summary_id,
                'type' => $amazonLedgerSummaryAdjustmentData->type,
                'inventory_adjustment_id' => $amazonLedgerSummaryAdjustmentData->inventory_adjustment_id,
            ]);
            $ledgerSummaryAdjustmentData->save();
        }
    }

    public function findOpenTransfersToCoverLedgerSummary(AmazonFbaReportInventoryLedgerSummary $ledgerSummary, ?WarehouseTransferShipmentLine $initialOutboundShipmentLine, int $quantityToReceive): ?Collection
    {
        $unallocatedToReceive = abs($quantityToReceive);
        $shipmentLines = $this->fnskus->getWarehouseTransferShipmentLines($ledgerSummary->amazonFnskuProduct);

        if ($initialOutboundShipmentLine && $quantityToReceive > 0) {
            $shipmentLines->push($initialOutboundShipmentLine);
        }

        $warehouseTransferCollection = new Collection();
        foreach ($shipmentLines as $shipmentLine)
        {
            if ($shipmentLine->unreceived == 0) {
                // We don't want to do fully received, so we skip
                continue;
            }
            $allocatableToReceive = min($unallocatedToReceive, $shipmentLine->unreceived);
            if ($allocatableToReceive > 0) {
                $unallocatedToReceive -= $allocatableToReceive;
                $warehouseTransferCollection->push($shipmentLine->warehouseTransferShipment->warehouseTransfer);
            }
            if ($unallocatedToReceive == 0) {
                return $warehouseTransferCollection;
            }
        }
        return null;
    }

    public function markReconciled(AmazonFbaReportInventoryLedgerSummary $ledgerSummary): AmazonFbaReportInventoryLedgerSummary
    {
        $ledgerSummary->reconciled_at = now();
        $ledgerSummary->errorLog = null;
        $ledgerSummary->save();
        return $ledgerSummary;
    }

    /**
     * @throws Throwable
     */
    public function markUnreconciled(AmazonFbaReportInventoryLedgerSummary $ledgerSummary): AmazonFbaReportInventoryLedgerSummary
    {
        return DB::transaction(function () use ($ledgerSummary) {
            $ledgerSummary->reconciled_at = null;
            $ledgerSummary->errorLog = null;
            $ledgerSummary->save();
            $this->deleteAdjustments($ledgerSummary);
            return $ledgerSummary;
        });
    }

    public function deleteAdjustments(AmazonFbaReportInventoryLedgerSummary $ledgerSummary): void
    {
        $ledgerSummary->ledgerSummaryAdjustments->sortBy(fn ($ledgerSummaryAdjustment) => $ledgerSummaryAdjustment->inventoryAdjustment->quantity)->each(/**
         * @throws Throwable
         */ function (AmazonLedgerSummaryAdjustment $ledgerSummaryAdjustment) use ($ledgerSummary)
        {
            customlog('amazon', 'Deleting adjustment for ledger summary ' . $ledgerSummary->id . ' and adjustment ' . $ledgerSummaryAdjustment->id . ' for qty ' . $ledgerSummaryAdjustment->inventoryAdjustment->quantity);
            $ledgerSummaryAdjustment->delete();
        });
    }

    public function markError(AmazonFbaReportInventoryLedgerSummary $ledgerSummary, string $errorLog): AmazonFbaReportInventoryLedgerSummary
    {
        $ledgerSummary->errorLog = $errorLog;
        $ledgerSummary->save();
        return $ledgerSummary;
    }

}