<?php

namespace Modules\Amazon\Repositories;

use App\Helpers;
use App\Models\InventoryAdjustment;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\Setting;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceiptLine;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\UniqueConstraintViolationException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Data\AmazonLedgerForValuationData;
use Modules\Amazon\Entities\AbstractAmazonFbaDetailReport;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportCustomerReturn;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFbaReportRemovalOrder;
use Modules\Amazon\Entities\AmazonFbaReportRemovalShipment;
use Modules\Amazon\Entities\AmazonFbaReportShipment;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonLedgerDetail;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\AmazonLedgerDispositionEnum;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Exceptions\LedgerEventTypeWithNoDetailReportException;
use Modules\Amazon\Exceptions\LedgerEventTypeWithNoSkuLinkException;
use Spatie\LaravelData\DataCollection;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;

class AmazonLedgerRepository
{
    public function __construct(
        private readonly AmazonReportRepository $reports,
        private readonly AmazonFnskuRepository $fnskus,
    )
    {
    }

    public function markReconciliationErrorsForMissingInitialInventoryReport(AmazonIntegrationInstance $amazonIntegrationInstance, ?AmazonReport $amazonReport = null): void
    {
        // If the summary report has already been processed, then we don't need the initial inventory.  We can assume any ledgers
        // without initial inventory just didn't have starting inventory.
        if (
            AmazonReport::query()
                ->where('reportType', AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY)
                ->whereNotNull('processed_at')
                ->exists()
        ) {
            return;
        }

        $query = AmazonFbaReportInventoryLedger::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereHas('amazonFnskuProduct', function (Builder $query) {
                $query->whereDoesntHave('amazonFbaInitialInventory');
            });

        if (! is_null($amazonReport)) {
            $query->where('amazon_report_id', $amazonReport->id);
        }

        if ($query->count() > 0) {
            customlog('amazon', 'The following FNSKU\'s do not have initial inventory records', $query->pluck('fnsku')->unique()->toArray());
        }

        $query->update([
            'last_reconciliation_attempted_at' => Carbon::now(),
            'errorLog' => 'FNSKU does not have initial inventory record',
        ]);
    }

    public function markReconciliationErrorsForMissingAmazonFnskus(AmazonIntegrationInstance $amazonIntegrationInstance, ?AmazonReport $amazonReport = null): void
    {
        // Get ledgers for report that are missing fnskus
        $query = AmazonFbaReportInventoryLedger::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereDoesntHave('amazonFnskuProduct');

        if (! is_null($amazonReport)) {
            $query->where('amazon_report_id', $amazonReport->id);
        }

        if ($query->count() > 0) {
            customlog('amazon', 'The following FNSKU\'s are not linked to a SKUio product.  Check mappings', $query->pluck('fnsku')->unique()->toArray());
        }

        /*
         * We use the ledger report asin to find the amazon product associated with the asin
         * We also use the product listing relationship to find the SKUio product associated to the amazon product
         */
        $query->update([
            'last_reconciliation_attempted_at' => Carbon::now(),
            'errorLog' => 'FNSKU is not yet linked to a SKUio product.  Check mappings.',
        ]);
    }

    public function markReconciliationErrorsForUninitializedAmazonFnskus(AmazonIntegrationInstance $amazonIntegrationInstance, ?AmazonReport $amazonReport = null): void
    {
        // It should be initialized if the initial inventory exists and amazon fnsku exists.  Could be just a timing thing
        $query = AmazonFbaReportInventoryLedger::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereHas('amazonFnskuProduct', function (Builder $query) {
                $query->whereHas('amazonFbaInitialInventory', function (Builder $query) {
                    $query->whereDoesntHave('fifoLayer');
                });
            });

        if (! is_null($amazonReport)) {
            $query->where('amazon_report_id', $amazonReport->id);
        }

        if ($query->count() > 0) {
            customlog('amazon', 'The following FNSKU\'s have unreconciled starting inventory.', $query->pluck('fnsku')->unique()->toArray());
        }

        $query->update([
            'last_reconciliation_attempted_at' => Carbon::now(),
            'errorLog' => 'Initial inventory is not yet reconciled',
        ]);
    }

    public function markReconciliationErrorsForHaltedBy(AmazonFbaReportInventoryLedger $ledger): Collection
    {
        $ledger->amazonFnskuProduct->amazonFbaReportInventoryLedgers()
            ->whereNull('reconciled_at')
            ->whereNot('id', $ledger->id)
            ->where('event_datetime', '>=', $ledger->event_datetime)
            ->update([
                'errorLog' => 'Reconciliation blocked by earlier ledger',
                'blocked_by_ledger_id' => $ledger->id,
                'last_reconciliation_attempted_at' => Carbon::now(),
            ]);

        return $ledger->amazonFnskuProduct->amazonFbaReportInventoryLedgers()->get();
    }

    /**
     * @throws Exception
     * @throws LedgerEventTypeWithNoDetailReportException
     */
    public function getDetailModelFromLedgerEventType(FbaInventoryLedgerReportEventTypeEnum $eventType): string
    {
        return match ($eventType) {
            FbaInventoryLedgerReportEventTypeEnum::CustomerReturns => AmazonFbaReportCustomerReturn::class,
            FbaInventoryLedgerReportEventTypeEnum::Shipments => AmazonFbaReportShipment::class,
            FbaInventoryLedgerReportEventTypeEnum::VendorReturns => AmazonFbaReportRemovalShipment::class,
            default => throw new LedgerEventTypeWithNoDetailReportException('Unhandled event type: '.$eventType->value),
        };
    }

    /**
     * @throws Exception
     */
    public function getSkuModelFromLedgerEventType(FbaInventoryLedgerReportEventTypeEnum $eventType): string|array
    {
        return match ($eventType) {
            FbaInventoryLedgerReportEventTypeEnum::Adjustments => InventoryAdjustment::class,
            FbaInventoryLedgerReportEventTypeEnum::CustomerReturns => SalesCreditReturnLine::class,
            FbaInventoryLedgerReportEventTypeEnum::Receipts => [WarehouseTransferShipmentReceiptLine::class, PurchaseOrderShipmentReceiptLine::class],
            FbaInventoryLedgerReportEventTypeEnum::Shipments => SalesOrderFulfillmentLine::class,
            FbaInventoryLedgerReportEventTypeEnum::VendorReturns => WarehouseTransferShipmentLine::class,
            default => throw new LedgerEventTypeWithNoSkuLinkException('Unhandled event type: '.$eventType->value),
        };
    }

    /**
     * @throws Exception
     */
    public function getLedgerEventTypeFromDetailModel(Model $model): FbaInventoryLedgerReportEventTypeEnum
    {
        return match ($model::class) {
            AmazonFbaReportCustomerReturn::class => FbaInventoryLedgerReportEventTypeEnum::CustomerReturns,
            AmazonFbaReportShipment::class => FbaInventoryLedgerReportEventTypeEnum::Shipments,
            AmazonFbaReportRemovalShipment::class => FbaInventoryLedgerReportEventTypeEnum::VendorReturns,
            default => throw new Exception('Unhandled model: '.$model::class),
        };
    }

    public function getLedgerReportLastUpdatedAfter(AmazonIntegrationInstance $amazonIntegrationInstance): Carbon
    {
        $ledgerMaxDate = AmazonFbaReportInventoryLedger::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->max('date');

        return $ledgerMaxDate ? Carbon::parse($ledgerMaxDate) :
            Helpers::setting(Setting::KEY_INVENTORY_START_DATE);
    }

    public function getUnprocessedLedgerReportsOrderedByDateAndQuantity(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection|array
    {
        // We can skip ledger entries that are 0
        return AmazonFbaReportInventoryLedger::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('quantity', '!=', 0)
            //->where('processed', false)
            ->orderBy('date')
            ->orderBy('quantity', 'desc')
            ->get();
    }

    public function getLedgerReportsNotLinkedToDetail(): EloquentCollection
    {
        $query = AmazonFbaReportInventoryLedger::query()
            ->whereDoesntHave('ledgerDetails')
            ->whereNull('sku_link_id')
            ->whereIn('event_type', FbaInventoryLedgerReportEventTypeEnum::HasDetailReports);

        return $query->get();
    }

    /**
     * @throws Exception
     */
    public function getDetailReportsNotLinkedToLedger(): Collection
    {
        $detailReportCollection = new Collection();

        collect(AmazonReportTypeEnum::FBA_DETAIL_REPORTS)->each(function (AmazonReportTypeEnum $reportType) use ($detailReportCollection) {
            $model = $this->reports->getModelClassFromReportType($reportType);

            $detailReportCollection->push(app($model)::query()
                ->whereDoesntHave('ledger')
                ->get());
        });

        return $detailReportCollection->flatten();
    }



    /**
     * @throws Exception
     */
    public function getDetailReportLedgerMatches(AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger, int $dateRangeDays = 0, bool $looseQuantity = false, $useMappings = false): EloquentCollection
    {
        $modelClass = $this->getDetailModelFromLedgerEventType($amazonFbaReportInventoryLedger->event_type);

        /** @var AbstractAmazonFbaDetailReport $detailReport */
        $detailReport = app($modelClass);

        $matches = $detailReport::query()
            ->where('integration_instance_id', $amazonFbaReportInventoryLedger->integration_instance_id)
            ->whereDoesntHave('ledger')
            ->where(function (Builder $query) use ($amazonFbaReportInventoryLedger, $modelClass, $useMappings) {
                $query->where('sku', $amazonFbaReportInventoryLedger->msku);

                if ($modelClass !== AmazonFbaReportShipment::class) {
                    $query->orWhere('fnsku', $amazonFbaReportInventoryLedger->fnsku);
                } else {
                    if ($initialInventory = $amazonFbaReportInventoryLedger->amazonFnskuProduct?->amazonFbaInitialInventory) {
                        $query->orWhere('sku', $initialInventory->msku);
                    }
                }

                if ($useMappings && $amazonFbaReportInventoryLedger->event_type == FbaInventoryLedgerReportEventTypeEnum::Shipments) {
                    $skus = AmazonProduct::whereAsin1($amazonFbaReportInventoryLedger->asin)->pluck('seller_sku')->flatten()->unique();
                    $query->orWhereIn('sku', $skus);
                }
            });

        $quantity = $amazonFbaReportInventoryLedger->quantity;
        if ($detailReport::hasNegativeLedgerImpact())
        {
            $quantity = -$quantity;
        }
        if (!$looseQuantity)
        {
            $matches->where($detailReport::getQuantityField(), $quantity);
        }

        switch ($amazonFbaReportInventoryLedger->event_type) {
            case FbaInventoryLedgerReportEventTypeEnum::CustomerReturns:
            case FbaInventoryLedgerReportEventTypeEnum::Shipments:
                $matches
                    ->where('fulfillment_center_id', $amazonFbaReportInventoryLedger->fulfillment_center);
                break;
            case FbaInventoryLedgerReportEventTypeEnum::VendorReturns:
                $matches
                    ->where(
                        'disposition',
                        AmazonLedgerDispositionEnum::VENDOR_RETURN_DISPOSITIONS[$amazonFbaReportInventoryLedger->disposition]
                    );
                break;
            default:
                throw new Exception('Unhandled eventType: '.$amazonFbaReportInventoryLedger->event_type->value);
        }
        if ($dateRangeDays) {
            $matches->whereRaw('CONVERT_TZ(event_datetime, "UTC", "' . $amazonFbaReportInventoryLedger->integrationInstance->getTimezone() . '") BETWEEN "' .
                Carbon::parse($amazonFbaReportInventoryLedger->date)->subDays($dateRangeDays)->startOfDay() . '" AND "' .
                Carbon::parse($amazonFbaReportInventoryLedger->date)->addDays($dateRangeDays)->endOfDay() . '"'
            );
        } else {
            $matches->whereRaw('DATE(CONVERT_TZ(event_datetime, "UTC", "' . $amazonFbaReportInventoryLedger->integrationInstance->getTimezone() . '")) = DATE("' . Carbon::parse($amazonFbaReportInventoryLedger->date) . '")');
        }

        $matches->selectRaw('*, ' . $detailReport::getQuantityField().' as detailQuantity');

        $matches->orderByRaw('ABS(TIMESTAMPDIFF(SECOND, event_datetime, ?))', [$amazonFbaReportInventoryLedger->event_datetime]);

        return $matches->get();
    }

    /**
     * @throws Exception
     */
    public function getShipmentReportLooseMatchesToLedger(AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger, int $dateRangeDays = 0): EloquentCollection
    {
        /**
         * For shipments, because the detail report has a sku but no fnsku, there are going to be a lot of missing details…
         * we need a backup method to find a match between the ledger and the detail report based on asin->amazon_product->sku
         */

        // First get all potential sku's for a given asin
        $potentialSkus = AmazonProduct::query()
            ->where('integration_instance_id', $amazonFbaReportInventoryLedger->integration_instance_id)
            ->where('asin1', $amazonFbaReportInventoryLedger->asin)
            ->pluck('seller_sku')
            ->toArray();

        $matches = AmazonFbaReportShipment::query()
            ->where('integration_instance_id', $amazonFbaReportInventoryLedger->integration_instance_id)
            ->whereDoesntHave('ledger')
            ->where('fulfillment_center_id', $amazonFbaReportInventoryLedger->fulfillment_center)
            ->where('quantity_shipped', -$amazonFbaReportInventoryLedger->quantity)
            ->whereIn('sku', $potentialSkus);

        if ($dateRangeDays) {
            $matches->whereBetween('event_datetime', [
                Carbon::parse($amazonFbaReportInventoryLedger->date)->subDays($dateRangeDays)->startOfDay(),
                Carbon::parse($amazonFbaReportInventoryLedger->date)->addDays($dateRangeDays)->endOfDay(),
            ]);
        } else {
            $matches->whereDate('event_datetime', Carbon::parse($amazonFbaReportInventoryLedger->date));
        }

        return $matches->get();
    }

    /**
     * @throws Exception
     */
    public function getLedgerReportDetailMatches(
        AbstractAmazonFbaDetailReport $detailReportModel,
        int $dateRangeDays = 0
    ): EloquentCollection {
        $eventType = $this->getLedgerEventTypeFromDetailModel($detailReportModel);

        $matches = AmazonFbaReportInventoryLedger::query()
            ->where('integration_instance_id', $detailReportModel->integration_instance_id)
            ->whereDoesntHave('ledgerDetails')
            ->where('event_type', $eventType->value)
            ->where(function (Builder $query) use ($detailReportModel) {
                $query->where('msku', $detailReportModel->sku)
                    ->orWhere('fnsku', $detailReportModel->fnsku)
                    ->orWhereHas('amazonFnskuProduct.amazonFbaInitialInventory', function (Builder $query) use ($detailReportModel) {
                        $query->where('amazon_fba_initial_inventory.msku', $detailReportModel->sku);
                    });
            });

        switch ($eventType) {
            case FbaInventoryLedgerReportEventTypeEnum::CustomerReturns:
                $matches
                    ->where('fulfillment_center', $detailReportModel->fulfillment_center_id)
                    ->where('quantity', abs($detailReportModel->quantity));
                break;
            case FbaInventoryLedgerReportEventTypeEnum::Shipments:
                $matches
                    ->where('fulfillment_center', $detailReportModel->fulfillment_center_id)
                    ->where('quantity', -$detailReportModel->quantity_shipped);
                break;
            case FbaInventoryLedgerReportEventTypeEnum::VendorReturns:
                $matches
                    ->where('quantity', -$detailReportModel->shipped_quantity);
                break;
            default:
                throw new Exception('Unhandled eventType: '.$eventType->value);
        }

        if ($dateRangeDays) {
            $matches->whereBetween('event_datetime', [
                Carbon::parse($detailReportModel->event_datetime)->subDays($dateRangeDays)->startOfDay(),
                Carbon::parse($detailReportModel->event_datetime)->addDays($dateRangeDays)->endOfDay(),
            ]);
        } else {
            $matches->whereDate('event_datetime', Carbon::parse($detailReportModel->event_datetime));
        }

        // Sort by how close the match is to the ledger date (event_datetime)
        $matches->orderByRaw('ABS(TIMESTAMPDIFF(SECOND, event_datetime, ?))', [$detailReportModel->event_datetime]);

        return $matches->get();
    }

    public function getReconciledFromInitialInventoryReport(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return AmazonFbaInitialInventory::with(['amazonFnskuProduct', 'fifoLayer'])
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereNotNull('fifo_layer_id')
            ->whereHas('amazonFnskuProduct', function (Builder $query) {
                $query->whereHas('product');
            })
            ->fnskuSort()
            ->get();
    }

    public function getUnreconciledFromInitialInventoryReport(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return AmazonFbaInitialInventory::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereNull('fifo_layer_id')
            ->whereHas('amazonFnskuProduct', function (Builder $query) {
                $query->whereHas('product');
            })
            ->get();
    }

    public function getUnlinkedRemovalOrders(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return AmazonFbaReportRemovalOrder::query()
            ->select('order_id')
            ->selectRaw('min(date(request_date)) as date')
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('order_type', 'Return')
            ->whereNull('sku_link_id')
            ->groupBy('order_id')
            ->orderBy('request_date', 'desc')
            ->get();
    }

    public function clearSkuLinkForSkuModel(Model $amazonReportModel, Model $skuModel): void
    {
        $data = [
            'sku_link_type' => null,
            'sku_link_id' => null,
        ];

        if ($amazonReportModel instanceof AmazonFbaReportInventoryLedger) {
            $data['errorLog'] = null;
            $data['last_reconciliation_attempted_at'] = null;
            $data['reconciled_at'] = null;
            $data['blocked_by_ledger_id'] = null;
        }
        $amazonReportModel::query()
            ->where('sku_link_type', $skuModel::class)
            ->where('sku_link_id', $skuModel->id)
            ->update($data);
    }

    public function clearLedgerReconciliation(AmazonFbaReportInventoryLedger $ledger): void
    {
        $ledger->sku_link_type = null;
        $ledger->sku_link_id = null;
        $ledger->errorLog = null;
        $ledger->blocked_by_ledger_id = null;
        $ledger->last_reconciliation_attempted_at = null;
        $ledger->reconciled_at = null;
        $ledger->save();
    }

    public function linkDetailsToLedger(AmazonFbaReportInventoryLedger $ledger, array $detailReports): void
    {
        foreach ($detailReports as $detailReport)
        {
            $ledgerDetail = new AmazonLedgerDetail([
                'amazon_fba_report_inventory_ledger_id' => $ledger->id,
                'detail_id' => $detailReport->id,
                'detail_type' => get_class($detailReport),
            ]);
            try {
                $ledgerDetail->save();
            } catch (UniqueConstraintViolationException) {
                return;
            }
        }
    }

    public function clearLedgerDetails(AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger): void
    {
        $amazonFbaReportInventoryLedger->ledgerDetails()->delete();
    }

    public function deleteSkuModelFromLedger(AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger): void
    {
        if ($amazonFbaReportInventoryLedger->skuLink) {
            $amazonFbaReportInventoryLedger->skuLink->delete();
            $amazonFbaReportInventoryLedger->sku_link_id = null;
            $amazonFbaReportInventoryLedger->sku_link_type = null;
        }
        $amazonFbaReportInventoryLedger->errorLog = null;

        $amazonFbaReportInventoryLedger->save();
    }

    /**
     * @throws Exception
     */
    public function getUnmatchedLedgersForDetailReport(int $detailReportId, FbaInventoryLedgerReportEventTypeEnum $eventType): EloquentCollection
    {
        $modelClass = $this->getDetailModelFromLedgerEventType($eventType);

        /** @var AbstractAmazonFbaDetailReport $detailReport */
        $detailReport = app($modelClass)->findOrFail($detailReportId);

        $quantity = $detailReport->{$detailReport::getQuantityField()};
        if ($detailReport::hasNegativeLedgerImpact())
        {
            $quantity = -$quantity;
        }

        $query = AmazonFbaReportInventoryLedger::query()
            ->select('id', 'sku_link_id', 'sku_link_type', 'event_datetime', 'fnsku', 'asin', 'msku', 'quantity', 'fulfillment_center', 'disposition', 'country')
            ->where('integration_instance_id', $detailReport->integration_instance_id)
            ->whereDoesntHave('ledgerDetails')
            ->where('event_type', $eventType);

        if ($eventType === FbaInventoryLedgerReportEventTypeEnum::Shipments) {
            /*
             * This may be unreliable since Amazon doesn't always provide an accurate msku in the ledger report or sku in the shipments report
             * To mitigate this we use the getPotentialFnskusForMerchantSku method to get all potential fnskus for a given sku
             */
            $query->where(function (Builder $builder) use ($detailReport, $quantity) {
                $builder->whereIn('fnsku', $this->fnskus->getPotentialFnskusForMerchantSku($detailReport->integrationInstance, $detailReport->sku))
                    ->orWhere('msku', $detailReport->sku);
                $builder->where('quantity', '>=', $quantity);
            });
        } else {
            $query->where('fnsku', $detailReport->fnsku);
            $query->where('quantity', $quantity);
        }

        if ($eventType !== FbaInventoryLedgerReportEventTypeEnum::VendorReturns) {
            $query->where('fulfillment_center', $detailReport->fulfillment_center_id);
        }

        // I'm not sure what comes first, the ledger or the detail report?
        //$query->where('event_datetime', '<=', Carbon::parse($detailReport->event_datetime, $detailReport->integrationInstance->getTimezone()));

        $query->orderByRaw('ABS(TIMESTAMPDIFF(SECOND, event_datetime, ?))', [$detailReport->event_datetime]);

        // TODO: consider disposition to narrow results (for removal shipments)?

        return $query->get();
    }

    /**
     * @throws Exception
     */
    public function getUnmatchedDetailReportsForLedger(AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger): EloquentCollection
    {
        $model = $this->getDetailModelFromLedgerEventType($amazonFbaReportInventoryLedger->event_type);

        $matches = app($model)::query()
            ->with([
                'ledger',
            ])
            ->where('integration_instance_id', $amazonFbaReportInventoryLedger->integration_instance_id)
            ->whereDoesntHave('ledger');

        switch ($amazonFbaReportInventoryLedger->event_type) {
            case FbaInventoryLedgerReportEventTypeEnum::CustomerReturns:
                $matches
                    ->select('id', 'return_date', 'event_datetime', 'sku', 'asin', 'fnsku', 'quantity', 'fulfillment_center_id', 'detailed_disposition')
                    ->where('sku', $amazonFbaReportInventoryLedger->msku)
                    ->where('fulfillment_center_id', $amazonFbaReportInventoryLedger->fulfillment_center)
                    ->where('quantity', abs($amazonFbaReportInventoryLedger->quantity));
                break;
            case FbaInventoryLedgerReportEventTypeEnum::Shipments:
                $potentialSkus = AmazonProduct::query()
                    ->where('integration_instance_id', $amazonFbaReportInventoryLedger->integration_instance_id)
                    ->where('asin1', $amazonFbaReportInventoryLedger->asin)
                    ->pluck('seller_sku')
                    ->toArray();

                $matches
                    ->select('id', 'purchase_date', 'shipment_date', 'reporting_date', 'event_datetime', 'sku', DB::raw('quantity_shipped as quantity'), 'fulfillment_center_id', 'sales_channel')
                    ->where('fulfillment_center_id', $amazonFbaReportInventoryLedger->fulfillment_center)
                    ->where('quantity_shipped', '<=', -$amazonFbaReportInventoryLedger->quantity)
                    ->whereIn('sku', $potentialSkus);

                break;
            case FbaInventoryLedgerReportEventTypeEnum::VendorReturns:
                $matches
                    ->select('id', 'order_id', 'request_date', 'shipment_date', 'event_datetime', 'sku', 'fnsku', DB::raw('shipped_quantity as quantity'), 'carrier', 'tracking_number', 'removal_order_type', 'disposition')
                    ->where('sku', $amazonFbaReportInventoryLedger->msku)
                    ->where('shipped_quantity', -$amazonFbaReportInventoryLedger->quantity)
                    ->where(
                        'disposition',
                        AmazonLedgerDispositionEnum::VENDOR_RETURN_DISPOSITIONS[$amazonFbaReportInventoryLedger->disposition]
                    );
                break;
            default:
                throw new Exception('Unhandled eventType: '.$amazonFbaReportInventoryLedger->event_type->value);
        }

        // I'm not sure what comes first, the ledger or the detail report?
        #$matches->where('event_datetime', '>=', $amazonFbaReportInventoryLedger->event_datetime);

        // Sort by how close the match is to the ledger date
        $matches->orderByRaw('ABS(TIMESTAMPDIFF(SECOND, event_datetime, ?))', [$amazonFbaReportInventoryLedger->event_datetime]);

        return $matches->get();
    }

    public function getOldestUnreconciledLedgerDate(AmazonIntegrationInstance $integrationInstance): ?string
    {
        return AmazonFbaReportInventoryLedger::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNull('reconciled_at')
            ->min('date');
    }

    public function getLastReconciledLedgerDate(AmazonIntegrationInstance $integrationInstance): ?string
    {
        return AmazonFbaReportInventoryLedger::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNotNull('reconciled_at')
            ->max('reconciled_at');
    }

    public function getReconciliationDatesByFnsku(AmazonIntegrationInstance $integrationInstance): EloquentCollection
    {
        return AmazonFbaReportInventoryLedger::select(
            'amazon_fba_report_inventory_ledger.fnsku',
            DB::raw('amazon_fnsku_products.id AS fnskuProductId'),
            'amazon_fba_report_inventory_ledger.msku',
            'products.sku',
            DB::raw('MAX(amazon_fba_report_inventory_ledger.date) as max_date'),
            DB::raw('MAX(CASE WHEN amazon_fba_report_inventory_ledger.reconciled_at IS NOT NULL THEN amazon_fba_report_inventory_ledger.date ELSE NULL END) as max_reconciled_date')
        )
            ->leftJoin('amazon_fnsku_products', 'amazon_fnsku_products.fnsku', '=', 'amazon_fba_report_inventory_ledger.fnsku')
            ->leftJoin('products', 'products.id', '=', 'amazon_fnsku_products.product_id')
            ->where('amazon_fba_report_inventory_ledger.integration_instance_id', $integrationInstance->id)
            ->groupBy('amazon_fba_report_inventory_ledger.fnsku')
            ->orderBy('max_reconciled_date', 'desc')
            ->orderBy('fnsku')
            ->get();
    }

    public function getLedgersReconciledCount(AmazonIntegrationInstance $integrationInstance, $isReconciled = true): int
    {
        $query = AmazonFbaReportInventoryLedger::query()
            ->where('integration_instance_id', $integrationInstance->id);

        if ($isReconciled) {
            $query->whereNotNull('reconciled_at');
        } else {
            $query->whereNull('reconciled_at');
        }

        return $query->count();
    }

    public function getTopLedgerErrors(AmazonIntegrationInstance $integrationInstance): EloquentCollection
    {
        return AmazonFbaReportInventoryLedger::query()
            ->select('errorLog', DB::raw('COUNT(*) as count'))
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNotNull('errorLog')
            ->groupBy('errorLog')
            ->orderByDesc('count')
            ->get();
    }

    public function getLedgerFnskuProductGroupedByDate(AmazonFnskuProduct $fnskuProduct, ?array $ids = null, bool $reconcileSort = true): Collection
    {
        $query = $fnskuProduct->amazonFbaReportInventoryLedgers();

        if ($reconcileSort) {
            $query->whereNull('sku_link_id');
        } else {
            $query->whereNotNull('sku_link_id');
        }

        if ($ids)
        {
            $dateQuery = $query->clone()->whereIn('id', $ids);
            if ($reconcileSort) {
                if ($maxDate = $dateQuery->max('event_datetime')) {
                    $query->where('event_datetime', '<=', $maxDate);
                }
            } else {
                if ($minDate = $dateQuery->min('event_datetime')) {
                    $query->where('event_datetime', '>=', $minDate);
                }
            }
        }

        if ($reconcileSort) {
            $query->reconcileSort();
        } else {
            $query->unreconcileSort();
        }

        return $query->get()->groupBy('event_datetime');
    }

    public function sortLedgersForProcessing(AmazonFnskuProduct $fnskuProduct, Collection $ledgers, bool $reconcileSort = true): Collection
    {
        $currentReconciledQuantity = $fnskuProduct->reconciled_quantity;

        customlog('amazon', "Reconciled tally $currentReconciledQuantity");

        $processedLedgers = collect();
        $deferredLedgers = collect(); // Collection to hold ledgers that can"t be processed initially

        foreach ($ledgers as $ledger) {
            if ($this->isStockDependencyMet($ledger, $currentReconciledQuantity, $reconcileSort)) {
                $this->adjustStockLevels($ledger, $currentReconciledQuantity, $reconcileSort);
                $processedLedgers->push($ledger);
            } else {
                // Add to the deferred queue for later processing
                $deferredLedgers->push($ledger);
            }
        }

        // Attempt to process deferred ledgers after initial processing
        $retryCount = 0; // To prevent infinite loops
        while (!$deferredLedgers->isEmpty() && $retryCount < $deferredLedgers->count()) {
            $ledger = $deferredLedgers->shift(); // Take the first ledger from the queue
            customlog('amazon', 'Processing deferred ledger ' . $ledger->id);
            if ($this->isStockDependencyMet($ledger, $currentReconciledQuantity, $reconcileSort)) {
                $this->adjustStockLevels($ledger, $currentReconciledQuantity, $reconcileSort);
                $processedLedgers->push($ledger);
                $retryCount = 0; // Reset retry count since we were able to process a ledger
            } else {
                // If still can't be processed, put it back at the end of the queue
                $deferredLedgers->push($ledger);
                $retryCount++; // Increment retry count to avoid infinite looping
            }
        }

        return $processedLedgers->merge($deferredLedgers);
    }

    private function isStockDependencyMet(
        $ledger,
        &$currentReconciledQuantity,
        $reconcileSort
    ): bool
    {
        if ($ledger->quantity < 0 || (!$reconcileSort && $ledger->quantity > 0)) {
            // Check if there's enough stock for outgoing
            return -$ledger->quantity <= $currentReconciledQuantity;
        } else {
            // Incoming stock, assume it can always be accommodated
            return true;
        }
    }

    private function adjustStockLevels(
        AmazonFbaReportInventoryLedger $ledger,
        int &$currentReconciledQuantity,
        bool $reconcileSort
    ): void
    {
        $quantity = $reconcileSort ? $ledger->quantity : -$ledger->quantity;

        // Adjust reconciled_quantity directly for all event types
        $currentReconciledQuantity += $quantity;
    }

    public function getLedgersForValuation(AmazonIntegrationInstance $integrationInstance): DataCollection
    {
        $standardLedgers = QueryBuilder::for(AmazonFbaReportInventoryLedger::class)
            ->with(['amazonFnskuProduct', 'skuLink.inventoryMovements'])
            ->where('integration_instance_id', $integrationInstance->id)
            ->allowedFilters([
                AllowedFilter::scope('before_event_date')
            ])
            ->whereHas('amazonFnskuProduct')
            ->whereHas('skuLink')
            ->whereHas('ledgerSummary')
            ->fnskuSort()
            ->get();

        return AmazonLedgerForValuationData::collection($standardLedgers->map(function (AmazonFbaReportInventoryLedger $ledger) {
            return AmazonLedgerForValuationData::from([
                'quantity' => $ledger->quantity,
                'inventoryMovements' => $ledger->skuLink->inventoryMovements,
                'fnskuProduct' => $ledger->amazonFnskuProduct,
            ]);
        }));
    }
}