<?php

namespace Modules\Amazon\Repositories;

use Exception;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonFinancialEventGroupData;
use Modules\Amazon\Entities\AmazonFinancialEvent;
use Modules\Amazon\Entities\AmazonFinancialEventGroup;
use Modules\Amazon\Entities\AmazonFinancialShipmentEvent;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\AmazonFinancialEventTypeEnum;

class AmazonFinancialEventRepository
{
    public function save(AmazonIntegrationInstance $amazonIntegrationInstance, Collection $groups): void
    {
        /** @var AmazonFinancialEventGroupData $group */
        foreach ($groups as $group) {
            if ($amazonFinancialEvent = $this->getMatchingEventForGroupAndEventType(
                $amazonIntegrationInstance,
                $group->amazon_financial_event_group_id,
                $group->event_type
            )) {
                $amazonFinancialEvent->update(['json_object' => $group->events]);
            } else {
                AmazonFinancialEvent::query()->create(array_merge(
                    ['integration_instance_id' => $amazonIntegrationInstance->id],
                    ['amazon_financial_event_group_id' => $group->amazon_financial_event_group_id],
                    ['event_type' => $group->event_type],
                    ['json_object' => $group->events]
                ));
            }
        }
    }

    private function getFromFinancialEventGroupId(int $amazonIntegrationInstanceId, string $FinancialEventGroupId): ?AmazonFinancialEventGroup
    {
        /** @var AmazonFinancialEventGroup $amazonFinancialEventGroup */
        $amazonFinancialEventGroup = AmazonFinancialEventGroup::query()
            ->where('integration_instance_id', $amazonIntegrationInstanceId)
            ->where('FinancialEventGroupId', $FinancialEventGroupId)
            ->first();

        return $amazonFinancialEventGroup;
    }

    private function getMatchingEventForGroupAndEventType(
        AmazonIntegrationInstance $amazonIntegrationInstance,
        int $amazonFinancialEventGroupId,
        string $eventType
    ): ?AmazonFinancialEvent {
        /** @var AmazonFinancialEvent $amazonFinancialEvent */
        $amazonFinancialEvent = AmazonFinancialEvent::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('amazon_financial_event_group_id', $amazonFinancialEventGroupId)
            ->where('event_type', $eventType)
            ->first();

        return $amazonFinancialEvent;
    }

    public function getLastPostedDate(AmazonIntegrationInstance $amazonIntegrationInstance): ?string
    {
        return null;

        return AmazonFinancialEventGroup::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->latest('FinancialEventGroupStart')
            ->pluck('FinancialEventGroupStart')
            ->first();
    }

    /**
     * @throws Exception
     */
    public function getModelFromEventType(AmazonFinancialEventTypeEnum $eventType): string
    {
        return match ($eventType) {
            AmazonFinancialEventTypeEnum::ShipmentEventList => AmazonFinancialShipmentEvent::class,
            default => throw new Exception("Unknown event type $eventType->value"),
        };
    }

    /**
     * @throws Exception
     */
    public function createEventsFromFinancialEvent(AmazonFinancialEvent $amazonFinancialEvent): void
    {
        foreach ($amazonFinancialEvent->json_object as $event) {
            app($this->getModelFromEventType($amazonFinancialEvent->event_type))::query()->create([
                'integration_instance_id' => $amazonFinancialEvent->integration_instance_id,
                'amazon_financial_event_group_id' => $amazonFinancialEvent->amazon_financial_event_group_id,
                'json_object' => $event,
            ]);
        }
    }

    public function getFinancialEventsForFinancialLines(AmazonFinancialEvent $amazonFinancialEvent): void
    {
        foreach ($amazonFinancialEvent->json_object as $event) {
            app($this->getModelFromEventType($amazonFinancialEvent->event_type))::query()->create([
                'integration_instance_id' => $amazonFinancialEvent->integration_instance_id,
                'amazon_financial_event_group_id' => $amazonFinancialEvent->amazon_financial_event_group_id,
                'json_object' => $event,
            ]);
        }
    }
}
