<?php

namespace Modules\Amazon\Managers;

use App\Data\NoteData;
use App\DTO\WarehouseTransferDto;
use App\DTO\WarehouseTransferLineDto;
use App\Models\Note;
use App\Models\Product;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use App\Repositories\NoteRepository;
use App\Repositories\WarehouseTransferRepository;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\RemovalOrderGroupData;
use Modules\Amazon\Data\RemovalOrderLineData;
use Modules\Amazon\Data\UpdateAmazonRemovalOrderData;
use Modules\Amazon\Entities\AmazonFbaReportRemovalOrder;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\FbaRemovalOrderTypeEnum;
use Modules\Amazon\Exceptions\UnmappedAmazonSellerSkuException;
use Modules\Amazon\Repositories\AmazonFnskuRepository;
use Modules\Amazon\Repositories\AmazonRemovalOrderRepository;
use Throwable;

class AmazonRemovalOrderManager
{
    private WarehouseTransferRepository $warehouseTransfers;
    private NoteRepository $notes;
    private AmazonRemovalOrderRepository $removalOrders;
    private AmazonFnskuRepository $fnskus;

    public function __construct(private readonly AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        $this->warehouseTransfers = app(WarehouseTransferRepository::class);
        $this->notes = app(NoteRepository::class);
        $this->removalOrders = app(AmazonRemovalOrderRepository::class);
        $this->fnskus = app(AmazonFnskuRepository::class);
    }

    /**
     * @throws Throwable
     */
    public function processUnprocessed(): void
    {
        $idsToProcess = $this->removalOrders->getUnprocessed($this->amazonIntegrationInstance)->pluck('id');

        if (!empty($idsToProcess)) {
            $this->process($idsToProcess->toArray());
        }
    }

    /**
     * @throws Throwable
     */
    public function process(array|int $ids, ?int $warehouse_id = null): Collection
    {
        if (!is_array($ids)) {
            $ids = [$ids];
        }

        $warehouse_id = $warehouse_id ?? $this->amazonIntegrationInstance->integration_settings['removal_order_warehouse_id'];
        if (!$warehouse_id) {
            return collect();
        }

        $removalOrders = $this->removalOrders->getForValues($ids, 'id', AmazonFbaReportRemovalOrder::class);

        // TODO: Trigger audit trail processing here

        return $this->processWarehouseTransfers($removalOrders, $warehouse_id);
    }

    public function unprocess(array $ids): void
    {
        $removalOrders = $this->removalOrders->getForValues($ids, 'id', AmazonFbaReportRemovalOrder::class);

        $removalOrderIds = $removalOrders->pluck('order_id')->unique();

        foreach ($removalOrderIds as $removalOrderId)
        {
            $warehouseTransfer = $this->warehouseTransfers->getFromWarehouseTransferNumber($removalOrderId);
            $warehouseTransfer?->delete();
        }
    }

    /**
     * @throws Throwable
     */
    public function processWarehouseTransfers(Collection $removalOrders, $warehouse_id): Collection
    {
        // Warehouse transfers can only be created for removal orders of type Return
        $removalOrders = $removalOrders->reject(function (AmazonFbaReportRemovalOrder $removalOrder) {
            return $removalOrder->order_type !== FbaRemovalOrderTypeEnum::Return;
        });

        $removalOrderIds = $removalOrders->pluck('order_id')->unique();

        $removalOrdersCollection = collect();
        foreach ($removalOrderIds as $order_id) {
            try {
                $removalOrderGroup = $this->getRemovalOrderGroupFromOrderId($order_id);
            } catch (UnmappedAmazonSellerSkuException $e) {
                $this->removalOrders->saveErrorForRemovalOrderId($order_id, $e->getMessage());
                continue;
            }
            $removalOrdersCollection->add($this->createOrLinkWarehouseTransfer($removalOrderGroup, $warehouse_id));
        }
        return $removalOrdersCollection->flatten();
    }

    /**
     * @throws Throwable
     */
    public function createOrLinkWarehouseTransfer(RemovalOrderGroupData $removalOrderGroup, $warehouse_id): Collection
    {
        /** @var WarehouseTransfer $warehouseTransfer */
        if ($warehouseTransfer = $this->warehouseTransfers->getFromWarehouseTransferNumber($removalOrderGroup->order_id))
        {
            $warehouseTransferLineData = $removalOrderGroup->removalOrderLines->reject(fn(RemovalOrderLineData $line) => $line->is_unknown_item)->map(function (RemovalOrderLineData $line) use ($warehouseTransfer) {
                return WarehouseTransferLineDto::from([
                    'warehouse_transfer_id' => $warehouseTransfer->id,
                    'product_id' => $line->product_id,
                    'quantity' => $line->requested_quantity,
                    'description' => 'SellerSKU: '.$line->sku.', FNSKU: '.$line->fnsku.', Condition: '.$line->disposition,
                ]);
            })->toCollection();

            $warehouseTransferLineCollection = collect($this->warehouseTransfers->save($warehouseTransferLineData, WarehouseTransferLine::class));
        } else
        {
            $warehouseTransferLines = $removalOrderGroup->removalOrderLines->reject(fn(RemovalOrderLineData $line) => $line->is_unknown_item)->map(function (RemovalOrderLineData $line) {
                return WarehouseTransferLineDto::from([
                    'product_id' => $line->product_id,
                    'quantity' => $line->requested_quantity,
                    'description' => 'SellerSKU: '.$line->sku.', FNSKU: '.$line->fnsku.', Condition: '.$line->disposition,
                ]);
            });

            $warehouseTransferDto = WarehouseTransferDto::from([
                'warehouse_transfer_number' => $removalOrderGroup->order_id,
                'transfer_date' => Carbon::parse($removalOrderGroup->request_date)->toDateTimeString(),
                'from_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
                'to_warehouse_id' => $warehouse_id,
                // Inventory movements will be performed later when Warehouse Transfer Shipments are processed
                'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_OPEN,
                'warehouseTransferLines' => $warehouseTransferLines,
            ]);

            $warehouseTransfer = $this->warehouseTransfers->saveWithRelations(WarehouseTransferDto::collection([$warehouseTransferDto]));
            $warehouseTransferLineCollection = $warehouseTransfer->warehouseTransferLines;

            $noteData = $removalOrderGroup->removalOrderLines->filter(fn(RemovalOrderLineData $line) => $line->is_unknown_item)->map(function (RemovalOrderLineData $line) use ($warehouseTransfer) {
                return NoteData::from([
                    'note' => 'Unknown item: SellerSKU: '.$line->sku.', FNSKU: '.$line->fnsku.', Quantity: ' . $line->requested_quantity . ', Condition: '.$line->disposition . '.  You must create a manual adjustment for this item once received and inspected.',
                    'link_id' => $warehouseTransfer->id,
                    'link_type' => WarehouseTransfer::class,
                ]);
            })->toCollection();

            $this->notes->save($noteData, Note::class);
        }

        $updateRemovalOrderData = $warehouseTransferLineCollection->map(function ($line) use ($removalOrderGroup) {
            $removalOrderLineData = $removalOrderGroup->removalOrderLines
                ->where('product_id', $line['product_id'])
                ->first();
            return UpdateAmazonRemovalOrderData::from([
                'id' => $removalOrderLineData->removalOrder->id,
                'sku_link_id' => $line['id'],
                'sku_link_type' => WarehouseTransferLine::class,
                'errorLog' => null,
            ]);
        });

        batch()->update(new AmazonFbaReportRemovalOrder(), $updateRemovalOrderData->toArray(), 'id');

        return $this->removalOrders->getRemovalOrdersFromOrderId($removalOrderGroup->order_id);
    }

    /**
     * @throws UnmappedAmazonSellerSkuException
     */
    private function getProductFromRemovalOrderLine(RemovalOrderLineData $line): Product
    {
        if (!$product = $this->fnskus->getProductFromFnsku($this->amazonIntegrationInstance,
            $line->fnsku)) {
            $product = $line->removalOrder->amazonProduct?->productListing?->product;
        }
        if (!$product) {
            throw new UnmappedAmazonSellerSkuException("$line->sku needs to be mapped before creating this from removal order.  FNSKU: ".$line->fnsku.' not initialized either.');
        }
        return $product;
    }

    public function getRemovalOrderGroupFromOrderId(string $orderId): ?RemovalOrderGroupData
    {
        $removalOrders = $this->removalOrders->getRemovalOrdersFromOrderId($orderId);
        if ($removalOrders->isEmpty()) {
            return null;
        }

        $firstRemovalOrder = $removalOrders->first();

        $removalOrderLines = $removalOrders->map(function (AmazonFbaReportRemovalOrder $removalOrder) {
            $removalOrderLineData = RemovalOrderLineData::from([
                'removalOrder' => $removalOrder,
                'sku' => $removalOrder->sku,
                'fnsku' => $removalOrder->fnsku,
                'disposition' => $removalOrder->disposition,
                'requested_quantity' => $removalOrder->requested_quantity,
                'cancelled_quantity' => $removalOrder->cancelled_quantity ?? 0,
                'disposed_quantity' => $removalOrder->disposed_quantity ?? 0,
                'shipped_quantity' => $removalOrder->shipped_quantity ?? 0,
                'in_process_quantity' => $removalOrder->in_process_quantity ?? 0,
                'removal_fee' => $removalOrder->removal_fee ?? 0,
                'currency' => $removalOrder->currency ?? 'USD',
                'is_unknown_item' => $removalOrder->is_unknown_item,
            ]);
            $removalOrderLineData->product_id = $removalOrderLineData->is_unknown_item ? null : $this->getProductFromRemovalOrderLine($removalOrderLineData)->id;
            return $removalOrderLineData;
        });

        return RemovalOrderGroupData::from([
            'order_id' => $firstRemovalOrder->order_id,
            'request_date' => $firstRemovalOrder->request_date,
            'order_type' => $firstRemovalOrder->order_type,
            'order_status' => $firstRemovalOrder->order_status,
            'last_updated_date' => $firstRemovalOrder->last_updated_date,
            'removalOrderLines' => $removalOrderLines,
        ]);
    }

    public function markItemAsUnknown(AmazonFbaReportRemovalOrder $removalOrder): void
    {
        $removalOrder->is_unknown_item = true;
        $removalOrder->save();
    }

    public function unmarkItemAsUnknown(AmazonFbaReportRemovalOrder $removalOrder): void
    {
        $removalOrder->is_unknown_item = false;
        $removalOrder->save();
    }
}
