<?php

namespace Modules\Amazon\Managers;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProduct;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProductManager;
use App\Data\CreateInventoryAdjustmentData;
use App\Data\SalesChannelProductToSkuProductMappingCollectionData;
use App\Data\SalesChannelProductToSkuProductMappingData;
use App\Exceptions\SalesChannelProductMappingException;
use App\Helpers;
use App\Managers\InventoryAdjustmentManager;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Models\TaskStatus\TaskStatus;
use App\Repositories\FifoLayerRepository;
use App\Repositories\InventoryMovementRepository;
use App\Repositories\ProductRepository;
use App\Repositories\StockTakeRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Storage;
use Modules\Amazon\Data\AmazonCreateInventoryFeedData;
use Modules\Amazon\Data\AmazonCreatePriceFeedData;
use Modules\Amazon\Data\AmazonProductData;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;
use Modules\Amazon\Jobs\AmazonInitializeSkuProductsJob;
use Modules\Amazon\Jobs\CreateAmazonFnSkusToSkuProductsJob;
use Modules\Amazon\Jobs\CreateAuditTrailFromAmazonLedgerJob;
use Modules\Amazon\Repositories\AmazonProductRepository;
use Modules\Amazon\Services\AmazonClient;
use SellingPartnerApi\ApiException;
use Throwable;

class AmazonProductManager extends AbstractSalesChannelProductManager
{
    const AMAZON_REPORT_TYPE_GET_MERCHANT_LISTING_DATA = 'GET_MERCHANT_LISTINGS_DATA';

    protected AmazonProductRepository $amazonProductRepository;

    private FifoLayerRepository $fifoLayerRepository;

    private InventoryMovementRepository $inventoryMovementRepository;

    private StockTakeRepository $stockTakes;

    /**
     * @throws Exception
     */
    public function __construct(protected AmazonIntegrationInstance|IntegrationInstanceInterface $amazonIntegrationInstance)
    {
        $this->amazonProductRepository = app(AmazonProductRepository::class);
        $this->fifoLayerRepository = app(FifoLayerRepository::class);
        $this->inventoryMovementRepository = app(InventoryMovementRepository::class);
        $this->stockTakes = app(StockTakeRepository::class);
        $this->setModel(new AmazonProduct());
        parent::__construct($amazonIntegrationInstance, new AmazonClient($this->amazonIntegrationInstance), $this->amazonProductRepository);

        config()->set('amazon_mappings', $this->mappings);
    }

    public function cacheProductListingQuantity(?array $productIds = []): void
    {
        // TODO: Implement cacheProductListingQuantity() method.
    }

    /**
     * @throws Exception
     */
    public function import(AmazonReport $amazonReport): void
    {
        $records = tsvToArray(file_get_contents(Storage::disk('amazon_reports')->path($amazonReport->filename)));

        $records = AmazonProductData::collection($records);

        $this->amazonProductRepository->saveForIntegration($amazonReport->integrationInstance, $records->toCollection());
    }

    /**
     * @throws ApiException
     * @throws Exception
     */
    public function getCatalogItem(AmazonProduct $amazonProduct)
    {
        $catalogData = $this->client->getCatalogItem($amazonProduct->asin1);

        if ($catalogData) {
            AmazonProduct::query()->where('asin1', $amazonProduct->asin1)->update([
                'was_catalog_data_sync_attempted' => true,
                'catalog_data_last_sync' => now(),
                'catalog_data' => $catalogData->toArray(),
            ]);
        } else {
            // If no catalog data available, we want to mark that we attempted to download data so that we don't try again
            AmazonProduct::query()->where('asin1', $amazonProduct->asin1)->update([
                'was_catalog_data_sync_attempted' => true,
                'removed_from_amazon' => 1,
            ]);
        }
    }

    public function searchCatalogItems(Collection $amazonProducts): void
    {
        $asins = $amazonProducts->pluck('asin1')->values()->all();

        $catalogItems = $this->client->searchCatalogItems($asins);

        if ($catalogItems->count() > 0) {
            $catalogItems->each(function ($catalogData) {
                if ($asin = @$catalogData->json_object['asin']) {
                    AmazonProduct::query()->where('asin1', $asin)->update([
                        'was_catalog_data_sync_attempted' => true,
                        'catalog_data_last_sync' => now(),
                        'catalog_data' => $catalogData->json_object,
                    ]);
                }
            });
        } else {
            // If no catalog data available, we want to mark that we attempted to download data so that we don't try again
            AmazonProduct::query()->whereIn('asin1', $asins)
                ->update([
                    'was_catalog_data_sync_attempted' => true,
                    'removed_from_amazon' => 1,
                ]);
        }
    }

    public function mapSalesChannelProductsToSkuProducts(
        SalesChannelProductToSkuProductMappingCollectionData $productMappingData,
        AbstractSalesChannelProduct $salesChannelProduct,
        IntegrationInstanceInterface $integrationInstance
    ): void
    {
        parent::mapSalesChannelProductsToSkuProducts($productMappingData, $salesChannelProduct,
            $integrationInstance);

        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->remapFnskuProducts($productMappingData);
    }

    public function postCreateSkuProducts(): void
    {
        Bus::dispatchChain([
            new CreateAmazonFnSkusToSkuProductsJob($this->amazonIntegrationInstance),
            new AmazonInitializeSkuProductsJob($this->amazonIntegrationInstance),
            new CreateAuditTrailFromAmazonLedgerJob($this->amazonIntegrationInstance),
        ]);
    }

    /**
     * Get product listings for quantity.
     */
    public function getProductListingsForInventoryFeed(): Collection
    {
        return $this->amazonProductRepository
            ->getProductListingsInventory($this->amazonIntegrationInstance)
            ->map(function ($productListing) {
                return AmazonCreateInventoryFeedData::from([
                    'SKU' => $productListing->listing_sku,
                    'Quantity' => $productListing->quantity,
                    'FulfillmentLatency' => (! is_null($productListing->active_fulfillment_latency) ? $productListing->active_fulfillment_latency : 1),
                ]);
            });
    }

    /**
     * Get product listings for pricing feed.
     */
    public function getProductListingsForPricingFeed(): Collection
    {
        return $this->amazonProductRepository
            ->getProductListingsPrices($this->amazonIntegrationInstance)
            ->map(function ($productListing) {
                return AmazonCreatePriceFeedData::from([
                    'SKU' => $productListing->listing_sku,
                    'StandardPrice' => $productListing->price,
                    'currency' => 'USD',
                ]);
            });
    }

    public function getProductFilePath(Request $request): string
    {
        $originalFilename = $request->input('original_name');
        $filename = $request->input('stored_name');
        // Prepare TaskStatus record to report background progress to the UI:
        $task = new TaskStatus();
        $task->title = 'Importing mappings';

        $task->addMessage('File uploaded successfully:  '.$originalFilename);

        // Find out real file path
        $uploader_info = config('uploader.listing-mappings', ['target' => 'imports/listing_mappings']);

        return storage_path(rtrim($uploader_info['target'], '/').'/'.$filename);
    }

    /**
     * @throws Throwable
     */
    public function initializeFbaInventoryFifoLayer(AmazonFbaInitialInventory $initialInventory, $initialCount = false): FifoLayer
    {
        $product = $initialInventory->amazonFnskuProduct->product;

        /*
         * $initialInventoryDate will be 1 day prior to the date needed, with no timestamp, but will represent end of day in user timezone
         * So what we need is to turn for example 12/31/23 into 1/1/2024 08:00:00 UTC
         */
        $inventoryDate = Carbon::parse($initialInventory->date, Helpers::getAppTimezone())->addDay()->setTimezone('UTC');

        /*
         * Reference:
         * https://forum.ecommercefuel.com/t/where-to-find-fba-inventory-levels-including-available-reserved-inbound-shipments/73787/16
         *
         */
        $quantity = $initialInventory->total_inventory_quantity;

        if ($initialCount) {
            $product->setInitialInventory($initialInventory->integrationInstance->warehouse->id, $quantity, $product->unit_cost);
            $fifoLayer = $this->stockTakes->getFifoLayerForInitialInventoryForWarehouseProduct($initialInventory->integrationInstance->warehouse->id, $product->id);
        } else {
            /*
             * TODO: Not sure if it ever makes sense to not do an initial count, need to understand how fnskus get initialized... there should be one of 2 options:
             *  1. Inventory is present at fba inventory start date and is included in initial count
             *  2. Inventory is new and is received as an inbound as the first movement
             */
            // Create adjustment if not initial count
            $inventoryAdjustment = app(InventoryAdjustmentManager::class)->createAdjustment(CreateInventoryAdjustmentData::from([
                'adjustment_date' => $inventoryDate->toDateTimeString(),
                'product_id' => $product->id,
                'warehouse_id' => $initialInventory->integrationInstance->warehouse->id,
                'quantity' => $quantity,
                'unit_cost' => $product->unit_cost,
                'adjustment_type' => CreateInventoryAdjustmentData::ADJUSTMENT_TYPE_INCREASE,
                'notes' => 'Amazon FBA Initial Inventory (FNSKU: ' . $initialInventory->fnsku . ')',
            ]));
            $fifoLayer = $inventoryAdjustment->creationFifoLayer;
        }

        $initialInventory->amazonFnskuProduct->reconciled_quantity = $initialInventory->total_inventory_quantity;
        $initialInventory->amazonFnskuProduct->save();

        return $fifoLayer;
    }

    public function getProductRepository()
    {
        return $this->amazonProductRepository;
    }

    /**
     * @throws Throwable
     */
    public function getSalesChannelProducts(array $productIds = [], bool $createForAll = false): Collection
    {
        $amazonProducts = parent::getSalesChannelProducts($productIds, $createForAll);
        return $this->sanitizeAmazonProducts($amazonProducts);
    }

    /**
     * Amazon specific override for hydrating mappings to remove any mappings for amazon products that fail  to
     * pass through sanitizeAmazonProducts
     *
     * @throws SalesChannelProductMappingException
     */
    public function hydrateSalesChannelProductToSkuProductMappings(
        SalesChannelProductToSkuProductMappingCollectionData $productMappingData,
        AbstractSalesChannelProduct $salesChannelProduct,
        IntegrationInstanceInterface $integrationInstance
    ): Collection
    {
        $hydratedMappings = parent::hydrateSalesChannelProductToSkuProductMappings(
            $productMappingData,
            $salesChannelProduct,
            $integrationInstance
        );

        $sanitizedAmazonProducts = $this->sanitizeAmazonProducts($hydratedMappings->pluck('salesChannelProduct'), $hydratedMappings->pluck('product.sku', 'salesChannelProduct.id'));

        return $hydratedMappings->reject(function (SalesChannelProductToSkuProductMappingData $mapping) use ($sanitizedAmazonProducts) {
            return ! $sanitizedAmazonProducts->contains('id', $mapping->salesChannelProduct->id);
        });
    }

    /**
     * User should not be able to match FBA products to Matrix products or Non FBA products to Matrix/Bundle products.
     * (Throw an exception if these situations occur, to be handled by controller)
     * In addition, we should skip any records that don't have catalog data sync attempted.
     *
     * @throws SalesChannelProductMappingException
     */
    private function sanitizeAmazonProducts(Collection $amazonProducts, ?Collection $mappings = null): Collection
    {
        // We only reject here if we are creating, not matching
        if (!$mappings) {
            $amazonProducts = $amazonProducts->reject(fn ($amazonProduct) => !$amazonProduct->was_catalog_data_sync_attempted);
        }

        $skusToMatchTo = [];
        $amazonProducts->each(function (AmazonProduct $amazonProduct) use (&$skusToMatchTo, $mappings) {
            if ($mappings) {
                $skusToMatchTo[$amazonProduct->fulfillment_channel->value][] = $mappings[$amazonProduct->id];
            }
            else {
                $skusToMatchTo[$amazonProduct->fulfillment_channel->value][] = $amazonProduct->{$amazonProduct::getUniqueField()};
            }
        });

        if ($skus = app(ProductRepository::class)->skusForMatchingProductTypes(
            $skusToMatchTo[AmazonProductFulfillmentChannelEnum::AMAZON_NA->value] ?? [],
            [
                Product::TYPE_MATRIX,
                Product::TYPE_BUNDLE,
            ]
        )) {
            throw new SalesChannelProductMappingException('Unable to map FBA Amazon products to Matrix/Bundle products: ' . implode(', ', $skus));
        }

        if ($skus = app(ProductRepository::class)->skusForMatchingProductTypes(
            $skusToMatchTo[AmazonProductFulfillmentChannelEnum::DEFAULT->value] ?? [],
            [
                Product::TYPE_MATRIX,
            ]
        )) {
            throw new SalesChannelProductMappingException('Unable to map Merchant Fulfilled Amazon products to Matrix products: ' . implode(', ', $skus));
        }

        return $amazonProducts;
    }

    protected function syncInventory(?array $ids = []): void
    {
        // TODO: Implement syncInventory() method.
    }


}
