<?php

namespace Modules\Amazon\Managers;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use App\Data\NoteData;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Notifications\SalesOrderFulfilledNotification;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Services\SalesOrderFulfillment\SubmitTrackingInfo;
use Exception;
use Illuminate\Support\Facades\Notification;
use Modules\Amazon\Data\AmazonAddressData;
use Modules\Amazon\Data\AmazonFulfillmentOrderData;
use Modules\Amazon\Data\AmazonFulfillmentOrderItemData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Entities\AmazonFulfillmentOrder;
use Modules\Amazon\Entities\AmazonFulfillmentShipment;
use Modules\Amazon\Entities\AmazonFulfillmentShipmentPackage;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Repositories\AmazonOutboundFulfillmentRepository;
use Modules\Amazon\Services\AmazonClient;

class AmazonOutboundFulfillmentManager extends AbstractSalesChannelManager
{
    private AmazonOutboundFulfillmentRepository $outbounds;
    private SalesOrderRepository $salesOrders;

    /**
     * @throws Exception
     */
    public function __construct(protected AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        parent::__construct(
            $amazonIntegrationInstance,
            new AmazonClient($amazonIntegrationInstance)
        );

        $this->outbounds = app(AmazonOutboundFulfillmentRepository::class);
        $this->salesOrders = app(SalesOrderRepository::class);
    }

    /**
     * Create amazon fulfillment order.
     */
    public function createAmazonFulfillmentOrder(SalesOrderFulfillment $salesOrderFulfillment): void
    {
        $destinationAddress = $salesOrderFulfillment->salesOrder->shippingAddress;

        $createFulfillmentOrderDto = AmazonFulfillmentOrderData::from([
            'sellerFulfillmentOrderId' => $salesOrderFulfillment->fulfillment_number,
            'displayableOrderId' => "{$salesOrderFulfillment->salesOrder->sales_order_number}",
            'displayableOrderDate' => $salesOrderFulfillment->fulfilled_at->toIso8601ZuluString(),
            'displayableOrderComment' => @$salesOrderFulfillment->metadata['comments'],
            // TODO: Restore this once we use shipping methods properly
            //'shippingSpeedCategory' => $salesOrderFulfillment->requestedShippingMethod->name,
            'shippingSpeedCategory' => $salesOrderFulfillment->metadata['shipping_speed'],
            'destinationAddress' => AmazonAddressData::from([
                'name' => $destinationAddress->name,
                'addressLine1' => $destinationAddress->address1,
                'addressLine2' => $destinationAddress->address2,
                'city' => $destinationAddress->city,
                'postalCode' => $destinationAddress->zip,
                'stateOrRegion' => $destinationAddress->province_code,
                'countryCode' => $destinationAddress->country_code,
                'phone' => $destinationAddress->phone ?? '',
            ]),
            'items' => AmazonFulfillmentOrderItemData::collection(
                $salesOrderFulfillment->salesOrderFulfillmentLines
                    ->map(function ($salesOrderFulfillmentLine) use ($salesOrderFulfillment) {
                        return [
                            'sellerSku' => $salesOrderFulfillmentLine->metadata['fba_seller_sku'],
                            'sellerFulfillmentOrderItemId' => $salesOrderFulfillmentLine->id,
                            'quantity' => (int) $salesOrderFulfillmentLine->quantity,
                            'perUnitDeclaredValue' => [
                                'currencyCode' => $salesOrderFulfillment->salesOrder->currency->code,
                                'value' => $salesOrderFulfillmentLine->salesOrderLine->amount,
                            ],
                        ];
                    })
                    ->toArray()
            ),
        ]);

        $this->client->createFulfillmentOrder($createFulfillmentOrderDto);

        $this->outbounds->saveOutbound($this->amazonIntegrationInstance, $salesOrderFulfillment, $createFulfillmentOrderDto);
    }

    /**
     * Cancel amazon fulfillment.
     */
    public function cancelAmazonFulfillmentOrder(SalesOrderFulfillment $salesOrderFulfillment): void
    {
        $this->client->cancelFulfillmentOrder($salesOrderFulfillment->fulfillment_number);
    }

    public function refreshOutboundFulfillments(?string $queryStartDate = null, ?string $nextToken = null): AmazonResponseData
    {
        $queryStartDate = !empty($queryStartDate) ? $queryStartDate : $this->outbounds->getStartDateForNew($this->amazonIntegrationInstance);
        $amazonResponseDto = $this->client->listAllFulfillmentOrders($queryStartDate, $nextToken);

        $amazonOutboundFulfillmentsCollection = $amazonResponseDto->collection;

        if (!$amazonOutboundFulfillmentsCollection->count()) {
            return $amazonResponseDto;
        }

        $this->outbounds->saveStatusUpdates($this->amazonIntegrationInstance, $amazonOutboundFulfillmentsCollection);

        $this->getOutboundTracking();

        return $amazonResponseDto;
    }

    public function getOutboundTracking(): void
    {
        $fulfillmentOrders = $this->outbounds->getOrdersNeedingTracking($this->amazonIntegrationInstance);

        foreach ($fulfillmentOrders as $fulfillmentOrder) {
            $data = $this->client->getFulfillmentOrder($fulfillmentOrder->sellerFulfillmentOrderId)->collection->first();
            $this->outbounds->saveFulfillmentOrder($fulfillmentOrder, $data);
        }

        $fulfillmentOrders = $fulfillmentOrders->filter(
            fn (AmazonFulfillmentOrder $fulfillmentOrder) => $fulfillmentOrder->amazonFulfillmentShipments->count() > 0
        );

        $this->updateSalesOrderFulfillmentsFromFulfillmentOrders($fulfillmentOrders);
    }

    public function updateSalesOrderFulfillmentsFromFulfillmentOrders($fulfillmentOrders): void
    {
        /** @var AmazonFulfillmentOrder $fulfillmentOrder */
        foreach ($fulfillmentOrders as $fulfillmentOrder) {
            $salesOrderFulfillment = $fulfillmentOrder->salesOrderFulfillment;
            $fulfillmentOrder->amazonFulfillmentShipments->each(function (AmazonFulfillmentShipment $amazonFulfillmentShipment) use ($salesOrderFulfillment) {
                /** @var AmazonFulfillmentShipmentPackage $amazonFulfillmentShipmentPackage */
                $amazonFulfillmentShipmentPackage = $amazonFulfillmentShipment->amazonFulfillmentShipmentPackages()->first();

                if (!$amazonFulfillmentShipmentPackage) {
                    return;
                }

                // TODO: Following is repetitive from ShippingProviderOrderManager... refactor in future
                $salesOrderFulfillment->update([
                    'tracking_number' => $amazonFulfillmentShipmentPackage->trackingNumber,
                    'fulfilled_shipping_method' => $amazonFulfillmentShipmentPackage->carrierCode,
                    'fulfilled_at' => $amazonFulfillmentShipment->shippingDate,
                    'status' => SalesOrderFulfillment::STATUS_FULFILLED,
                ]);

                // Update fulfilled quantity
                $salesOrderFulfillment->salesOrderFulfillmentLines->each(/**
                 * @throws Exception
                 */ function (SalesOrderFulfillmentLine $salesOrderFulfillmentLine) {
                    $salesOrderFulfillmentLine->salesOrderLine->incrementFulfilledQuantity($salesOrderFulfillmentLine->quantity);
                });

                // Submit to sales channel if it exists
                SubmitTrackingInfo::factory($salesOrderFulfillment)?->submit();
                $salesOrderFulfillment->salesOrder->updateFulfillmentStatus();
                // Notify the customer that their order has been fulfilled
                if ($salesOrderFulfillment->salesOrder->salesChannel->emailsCustomers() && $salesOrderFulfillment->fulfilled()) {
                    Notification::send(
                        notifiables: [$salesOrderFulfillment->salesOrder->customer],
                        notification: new SalesOrderFulfilledNotification($salesOrderFulfillment->salesOrder)
                    );
                }
            });
        }
    }

    public function linkSalesOrderFulfillmentToExistingOutboundFulfillment(SalesOrderFulfillment $salesOrderFulfillment): bool
    {
        $amazonFulfillmentOrder = $this->outbounds->getOrphanFulfillment($this->amazonIntegrationInstance, $salesOrderFulfillment);

        if (!$amazonFulfillmentOrder) {
            return false;
        }

        $amazonFulfillmentOrder->salesOrderFulfillment()->associate($salesOrderFulfillment);
        $amazonFulfillmentOrder->save();

        $salesOrder = $salesOrderFulfillment->salesOrder;
        $this->salesOrders->addNote($salesOrder, NoteData::from([
            'link_id' => $salesOrder->id,
            'link_type' => SalesOrder::class,
            'note' => "Amazon fulfillment order {$amazonFulfillmentOrder->sellerFulfillmentOrderId} linked to sales order fulfillment {$salesOrderFulfillment->fulfillment_number}",
        ]));

        $this->updateSalesOrderFulfillmentsFromFulfillmentOrders(collect([$amazonFulfillmentOrder]));

        return true;
    }
}
