<?php

namespace Modules\Amazon\Managers;

use App\Abstractions\AbstractRepository;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use App\Data\IdQuantityData;
use App\DTO\PurchaseOrderDto;
use App\DTO\PurchaseOrderLineDto;
use App\DTO\WarehouseTransferDto;
use App\DTO\WarehouseTransferLineDto;
use App\Exceptions\SupplierProductNotFoundException;
use App\Helpers;
use App\Managers\WarehouseTransferManager;
use App\Models\Currency;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\Setting;
use App\Models\Store;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Repositories\PurchaseOrderRepository;
use App\Repositories\WarehouseTransferRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Abstractions\AmazonInboundShipmentSourceInterface;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundPlanAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundPlansAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundShipmentAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundShipmentItemsAdt;
use Modules\Amazon\Data\AmazonFbaInboundPlanData;
use Modules\Amazon\Data\AmazonFbaInboundShipFromMappingData;
use Modules\Amazon\Data\AmazonNewFbaInboundShipmentData;
use Modules\Amazon\Data\AmazonNewFbaInboundShipmentWithMappingData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Entities\AmazonFbaInboundShipFromMapping;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonNewFbaInboundPlan;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipment;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipmentItem;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Enums\Entities\AmazonNewFbaInboundShipmentStatusEnum;
use Modules\Amazon\Enums\Entities\FbaInboundShipmentStatusEnum;
use Modules\Amazon\Exceptions\ShipmentStatusNotEligibleForWarehouseTransferCreationException;
use Modules\Amazon\Exceptions\UnmappedAmazonSellerSkuException;
use Modules\Amazon\Exceptions\WarehouseTransferLinesDontMatchInboundException;
use Modules\Amazon\Jobs\CreateAmazonRefreshFbaInboundPlanItemsJobs;
use Modules\Amazon\Jobs\CreateAmazonRefreshFbaInboundPlanJobs;
use Modules\Amazon\Jobs\CreateAmazonRefreshFbaInboundShipmentsJobs;
use Modules\Amazon\Jobs\RefreshAmazonNewFbaInboundShipmentItemsJob;
use Modules\Amazon\Repositories\AmazonFbaInboundPlanItemRepository;
use Modules\Amazon\Repositories\AmazonFbaInboundShipFromMappingRepository;
use Modules\Amazon\Repositories\AmazonFnskuRepository;
use Modules\Amazon\Repositories\AmazonNewFbaInboundPlanRepository;
use Modules\Amazon\Repositories\AmazonNewFbaInboundShipmentItemRepository;
use Modules\Amazon\Repositories\AmazonNewFbaInboundShipmentRepository;
use Modules\Amazon\Repositories\AmazonProductRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Modules\Amazon\Services\AmazonClient;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Throwable;

class AmazonNewInboundManager extends AbstractSalesChannelManager
{
    protected AmazonNewFbaInboundPlanRepository $plans;

    protected WarehouseTransferRepository $warehouseTransfers;

    protected WarehouseTransferManager $transferProcessor;

    protected AmazonReportRepository $amazonReportRepository;

    protected AmazonFnskuRepository $fnskus;

    protected AmazonFbaInboundPlanItemRepository $planItems;

    protected AmazonNewFbaInboundShipmentRepository $shipments;

    protected AmazonNewFbaInboundShipmentItemRepository $shipmentItems;

    protected PurchaseOrderRepository $purchaseOrders;

    protected AmazonProductRepository $amazonProducts;

    protected AmazonFbaInboundShipFromMappingRepository $shipFromMappings;

    protected AmazonFnskuProductManager $fnskuManager;

    /**
     * @throws Exception
     */
    public function __construct(protected AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        parent::__construct($amazonIntegrationInstance, new AmazonClient($amazonIntegrationInstance));
        $this->plans                  = app(AmazonNewFbaInboundPlanRepository::class);
        $this->planItems              = app(AmazonFbaInboundPlanItemRepository::class);
        $this->shipments              = app(AmazonNewFbaInboundShipmentRepository::class);
        $this->shipmentItems          = app(AmazonNewFbaInboundShipmentItemRepository::class);
        $this->amazonReportRepository = app(AmazonReportRepository::class);
        $this->fnskus  = app(AmazonFnskuRepository::class);
        $this->warehouseTransfers      = app(WarehouseTransferRepository::class);
        $this->transferProcessor       = app(WarehouseTransferManager::class);
        $this->purchaseOrders = app(PurchaseOrderRepository::class);
        $this->amazonProducts          = app(AmazonProductRepository::class);
        $this->shipFromMappings        = app(AmazonFbaInboundShipFromMappingRepository::class);
        $this->fnskuManager            = app(AmazonFnskuProductManager::class);
    }


    protected function fetchShipment(AmazonGetInboundShipmentAdt $parameters): array
    {
        return $this->client->getFbaInboundShipment($parameters);
    }

    protected function fetchShipmentItems(AmazonGetInboundShipmentItemsAdt $parameters): AmazonResponseData
    {
        return $this->client->listShipmentItems($parameters);
    }

    /**
     * @throws Throwable
     */
    public function getShipment(AmazonGetInboundShipmentAdt $parameters): ?AmazonNewFbaInboundShipment
    {
        $shipmentDataArray = $this->fetchShipment($parameters);

        $plan = $this->plans->getFromPlanId($this->amazonIntegrationInstance, $parameters->planId);

        $shipmentData = AmazonNewFbaInboundShipmentData::from($shipmentDataArray);

        RefreshAmazonNewFbaInboundShipmentItemsJob::dispatch($this->amazonIntegrationInstance, new AmazonGetInboundShipmentItemsAdt(
            planId: $parameters->planId,
            shipmentId: $parameters->shipmentId
        ));

        $shipment = $this->shipments->saveShipment($plan, $shipmentData);
        $shipment->refresh();

        $this->transitionLegacyShipment($shipment);
        $this->fnskuManager->generateFnskuProducts();

        return $shipment;
    }

    public function transitionLegacyShipment(AmazonNewFbaInboundShipment $shipment): void
    {
        if ($legacyShipment = $this->shipments->checkLegacyShipment($this->integrationInstance, $shipment->shipmentConfirmationId))
        {
            $shipment->sku_link_id = $legacyShipment->sku_link_id;
            $shipment->sku_link_type = $legacyShipment->sku_link_type;
            $shipment->is_before_initial_count = $legacyShipment->is_before_initial_count;
            $shipment->save();
            $legacyShipment->delete();
        }
    }

    public function transitionLegacyShipments(): void
    {
        $this->shipments->getLegacyShipmentDuplicates($this->integrationInstance)->each(function (AmazonNewFbaInboundShipment $newFbaInboundShipment) {
            $this->transitionLegacyShipment($newFbaInboundShipment);
        });
    }

    public function getShipmentItems(AmazonGetInboundShipmentItemsAdt $parameters): void
    {
        $amazonResponseDto = $this->fetchShipmentItems($parameters);

        $shipmentItemCollection = $amazonResponseDto->collection;

        $shipment = $this->shipments->getFromShipmentId($this->amazonIntegrationInstance, $parameters->shipmentId);

        $this->shipmentItems->save($shipment->id, $shipmentItemCollection);
    }

    /**
     * @throws Exception
     */
    public function refreshPlans(AmazonGetInboundPlansAdt $parameters): AmazonResponseData
    {
        customlog('amazon', 'Refreshing Inbound Plans');
        // TODO: Handle pagination

        $amazonResponseDto = $this->client->listInboundPlans($parameters);

        $amazonFbaInboundPlanCollection = $amazonResponseDto->collection;

        if (!$amazonFbaInboundPlanCollection->count()) {
            return $amazonResponseDto;
        }

        $this->plans
            ->insertSummaries($this->amazonIntegrationInstance, $amazonFbaInboundPlanCollection->toCollection());

        $amazonFbaInboundShipFromMappingCollection = AmazonFbaInboundShipFromMappingData::collection($amazonFbaInboundPlanCollection->map(
            function (AmazonFbaInboundPlanData $inboundPlanData) {
                return AmazonFbaInboundShipFromMappingData::from([
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'name' => $inboundPlanData->json_object['sourceAddress']['name'],
                ]);
            }
        ));

        app(AbstractRepository::class)->save($amazonFbaInboundShipFromMappingCollection->toCollection(),
            AmazonFbaInboundShipFromMapping::class);

        CreateAmazonRefreshFbaInboundPlanItemsJobs::dispatch($this->amazonIntegrationInstance);
        CreateAmazonRefreshFbaInboundPlanJobs::dispatch($this->amazonIntegrationInstance);
        CreateAmazonRefreshFbaInboundShipmentsJobs::dispatch($this->amazonIntegrationInstance);

        // May cause issues... better to handle in job?
//        if ($amazonResponseDto->nextToken) {
//            $parameters->paginationToken = $amazonResponseDto->nextToken;
//            $this->refreshPlans($parameters);
//        }

        return $amazonResponseDto;
    }

    public function refreshPlanItems(array $planIds): void
    {
        foreach ($planIds as $planId) {
            $this->getPlanItems($planId);
        }
    }

    public function getPlan(AmazonGetInboundPlanAdt $parameters): ?AmazonNewFbaInboundPlan
    {
        $planDataArray = $this->client->getInboundPlan($parameters);

        if (empty($planDataArray)) {
            return null;
        }

        $planData = AmazonFbaInboundPlanData::from($planDataArray);

        $this->plans->saveForIntegration($this->integrationInstance, collect([$planData]));

        return $this->plans->getFromPlanId($this->amazonIntegrationInstance, $parameters->planId);
    }

    public function getPlanItems(string $planId): AmazonResponseData
    {
        $planItems = $this->fetchPlanItems($planId);

        $plan = $this->plans->getFromPlanId($this->amazonIntegrationInstance, $planId);

        $this->planItems->save($plan->id, $planItems->collection);

        return $planItems;
    }

    protected function fetchPlanItems(string $planId): AmazonResponseData
    {
        return $this->client->getFbaInboundPlanItems($planId);
    }

    /**
     * @throws Throwable
     */
    public function linkPurchaseOrder(PurchaseOrder $purchaseOrder, string $inboundShipmentId): null|PurchaseOrder|AmazonInboundShipmentSourceInterface
    {
        $shipment = $this->createOrLinkPurchaseOrder(AmazonNewFbaInboundShipmentWithMappingData::from([
            'shipment' => $this->shipments->getFromShipmentId($this->amazonIntegrationInstance, $inboundShipmentId),
            'mapping' => AmazonFbaInboundShipFromMapping::whereName($purchaseOrder->supplier->name)->first(),
        ]));

        return $shipment->skuLink;
    }

    public function processCancellations(): void
    {
        $this->shipments->getUnprocessedCancellations($this->amazonIntegrationInstance)->each(function (AmazonNewFbaInboundShipment $shipment) {
            $shipment->skuLink->delete();
        });
    }

    public function processUnprocessed(): void
    {
        $settings = $this->amazonIntegrationInstance->integration_settings;
        $types = [
            'automatically_create_warehouse_transfers_from_inbounds' => Warehouse::class,
            'automatically_create_purchase_orders_from_inbounds' => Supplier::class,
        ];

        $idsToProcess = collect($types)
            ->filter(function ($type, $setting) use ($settings) {
                return $settings[$setting] ?? false;
            })
            ->flatMap(function ($type) {
                return $this->shipments->getUnprocessedWithMappingOfType($this->amazonIntegrationInstance, $type)->pluck('id');
            })
            ->all();

        if (!empty($idsToProcess)) {
            $this->process($idsToProcess);
        }
    }

    /**
     * @throws Throwable
     */
    public function processWarehouseTransfer(int $id, int $warehouse_id): WarehouseTransfer
    {
        $shipment = $this->createOrLinkWarehouseTransfer(AmazonNewFbaInboundShipmentWithMappingData::from([
            'shipment' => $this->shipments->getForValue($id, 'id', AmazonNewFbaInboundShipment::class),
            'warehouse' => Warehouse::findOrFail($warehouse_id),
        ]));
        if (!$shipment->sku_link_id)
        {
            throw new Exception($shipment->errorLog);
        }
        return WarehouseTransfer::findOrFail($shipment->sku_link_id);
    }

    public function process(array|int $ids, ?int $warehouse_id = null): Collection
    {
        if (!is_array($ids)) {
            $ids = [$ids];
        }

        $shipments = $this->shipments->getForValues($ids, 'id', AmazonNewFbaInboundShipment::class, ['amazonFbaInboundShipmentItems']);

        $warehouse = $warehouse_id ? Warehouse::findOrFail($warehouse_id) : null;

        $mappings = $this->shipFromMappings->getMappings();

        $filterForLinkType = function ($linkType) use ($warehouse, $mappings) {
            return function (AmazonNewFbaInboundShipment $shipment) use ($warehouse, $mappings, $linkType)
            {
                $data['shipment'] = $shipment;

                if ($warehouse && $linkType == Warehouse::class) {
                    $data['warehouse'] = $warehouse;
                } else {
                    // Early return if no warehouse specified and no mappings available
                    if (!$mapping = $mappings->filter(fn ($item) => strtolower($item->name) == strtolower($shipment->sourceName))
                        ->where('link_type', $linkType)
                        ->first()) {
                        return null;
                    }
                    $data['mapping'] = $mapping;
                }

                return AmazonNewFbaInboundShipmentWithMappingData::from($data);
            };
        };

        $warehouseTransferProcessCollection = $shipments->map($filterForLinkType(Warehouse::class))->filter();
        $purchaseOrderProcessCollection = $shipments->map($filterForLinkType(Supplier::class))->filter();

        $processedShipments = collect();
        $processedShipments->push(...$this->processWarehouseTransfers($warehouseTransferProcessCollection));
        $processedShipments->push(...$this->processPurchaseOrders($purchaseOrderProcessCollection));

        // TODO: Trigger audit trail processing here

        return $processedShipments;
    }

    public function unprocess(array $ids): void
    {
        $shipments = $this->shipments->getForValues($ids, 'id', AmazonNewFbaInboundShipment::class);

        // Reject any shipments that don't have a link
        $shipments = $shipments->reject(function (AmazonNewFbaInboundShipment $shipment) {
            return !$shipment->sku_link_id;
        });

        $shipments->each(function (AmazonNewFbaInboundShipment $shipment) {
            $shipment->skuLink->delete();
            $shipment->skuLink()->dissociate();
            $shipment->save();
        });
    }

    public function processWarehouseTransfers(Collection $warehouseTransferProcessCollection): Collection
    {
        $shipments = collect();
        $warehouseTransferProcessCollection->each(/**
         * @throws Throwable
         */ function (AmazonNewFbaInboundShipmentWithMappingData $data) use ($shipments) {
             $shipments->push($this->createOrLinkWarehouseTransfer($data));
        });
        return $shipments;
    }

    public function processPurchaseOrders(Collection $purchaseOrderProcessCollection): Collection
    {
        $shipments = collect();
        $purchaseOrderProcessCollection->each(/**
         * @throws Throwable
         */ function (AmazonNewFbaInboundShipmentWithMappingData $data) use ($shipments) {
            $shipments->push($this->createOrLinkPurchaseOrder($data));
        });
        return $shipments;
    }

    /**
     * @throws Throwable
     */
    public function createOrLinkWarehouseTransfer(AmazonNewFbaInboundShipmentWithMappingData $data): AmazonNewFbaInboundShipment {
        $shipment = $data->shipment;

        try {
            /** @var WarehouseTransfer $warehouseTransfer */
            if ($warehouseTransfer = $this->warehouseTransfers->getFromWarehouseTransferNumber($shipment->shipmentConfirmationId))
            {
                if ($warehouseTransfer->warehouseTransferLines->isEmpty()) {
                    $this->getWarehouseTransferLines($data, $warehouseTransfer)->each(function (WarehouseTransferLineDto $line) use ($warehouseTransfer) {
                        $this->warehouseTransfers->createWarehouseTransferLine($warehouseTransfer, $line->product_id, $line->quantity);
                    });
                    $warehouseTransfer->save();
                } elseif (!$this->checkLinesMatchInbound($warehouseTransfer, $shipment)) {
                    throw new WarehouseTransferLinesDontMatchInboundException("Warehouse Transfer $warehouseTransfer->warehouse_transfer_number already exists but lines do not match inbound shipment");
                }

                $shipment->skuLink()->associate($warehouseTransfer);
                $shipment->save();

                // TODO: Is this creating movements if lines are added?

                return $shipment;
            } else
            {
                $shipmentStatus = FbaInboundShipmentStatusEnum::ALLOWED_SHIPMENT_STATUS_MAPPINGS[$shipment->status->value] ?? null;

                // If Shipment Status wasn't in the enum of allowed values, log the error
                if (!$shipmentStatus) {
                    throw new ShipmentStatusNotEligibleForWarehouseTransferCreationException("Shipment Status $shipment->status is not a Shipment Status eligible for Warehouse Transfer creation");
                }

                $warehouseTransferDto = WarehouseTransferDto::from([
                    'warehouse_transfer_number' => $shipment->shipmentConfirmationId,
                    'transfer_date' => $this->getDateFromShipmentName($shipment->name) ?? Carbon::now(),
                    'from_warehouse_id' => ! ($data->warehouse instanceof Optional) ? $data->warehouse->id : $data->mapping->link_id,
                    'to_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
                    // Always start with draft status so that later actions can perform necessary inventory movements
                    'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_DRAFT,
                    //'shipment_status' => WarehouseTransfer::TRANSFER_SHIPMENT_STATUS_UNSHIPPED,
                    //'receipt_status' => WarehouseTransfer::TRANSFER_RECEIPT_STATUS_UNRECEIVED,
                    'warehouseTransferLines' => $this->getWarehouseTransferLines($data),
                ]);

                DB::transaction(function () use ($warehouseTransferDto, $shipment) {
                    $warehouseTransfer = $this->warehouseTransfers->saveWithRelations(WarehouseTransferDto::collection([$warehouseTransferDto]));

                    $this->transferProcessor->openWarehouseTransfer($warehouseTransfer, []);

                    $shipment->skuLink()->associate($warehouseTransfer);
                    $shipment->save();
                });
            }
        } catch (UnmappedAmazonSellerSkuException|WarehouseTransferLinesDontMatchInboundException|ShipmentStatusNotEligibleForWarehouseTransferCreationException $e) {
            /*
             * TODO: InsufficientStockException is not caught right now, but it should be... need to redo how the exception is passed down and now the shipment error log is updated.
             */
            $shipment->errorLog = $e->getMessage();
            $shipment->save();
            return $shipment;
        }

        return $shipment;
    }

    /**
     * @throws Throwable
     */
    public function createOrLinkPurchaseOrder(AmazonNewFbaInboundShipmentWithMappingData $data
    ): AmazonNewFbaInboundShipment
    {
        $shipment = $data->shipment;

        try {
            /** @var PurchaseOrder $purchaseOrder */
            if ($purchaseOrder = $this->purchaseOrders->getForValue($shipment->shipmentId, 'purchase_order_number', PurchaseOrder::class))
            {
                if ($purchaseOrder->purchaseOrderLines->isEmpty()) {
                    $purchaseOrder->setPurchaseOrderLines($this->getPurchaseOrderLines($data, $purchaseOrder)->toArray());
                    $purchaseOrder->save();
                } elseif (!$this->checkLinesMatchInbound($purchaseOrder, $shipment)) {
                    $shipment->errorLog = "Purchase Order $purchaseOrder->purchase_order_number already exists but lines do not match inbound shipment";
                    $shipment->save();
                    return $shipment;
                }

                $shipment->skuLink()->associate($purchaseOrder);
                $shipment->save();

                return $shipment;
            } else
            {

                $shipmentStatus = AmazonNewFbaInboundShipmentStatusEnum::ALLOWED_SHIPMENT_STATUS_MAPPINGS[$shipment->status->value] ?? null;

                // If Shipment Status wasn't in the enum of allowed values, log the error
                if (!$shipmentStatus) {
                    $shipment->errorLog = "Shipment Status $shipment->status is not a Shipment Status eligible for Purchase Order creation";
                    $shipment->save();
                    return $shipment;
                }

                $purchaseOrderDto = PurchaseOrderDto::from([
                    'supplier_id' => $data->mapping->link_id,
                    'store_id' => Helpers::setting(Setting::KEY_PO_DEFAULT_STORE),
                    'purchase_order_number' => $shipment->shipmentConfirmationId,
                    'sequence' => PurchaseOrder::getNextSequence(),
                    'purchase_order_date' => $this->getDateFromShipmentName($shipment->name) ?? Carbon::now(),
                    'destination_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
                    'order_status' => PurchaseOrder::STATUS_OPEN,
                    'currency_id' => Currency::default()->id,
                    'currency_rate' => Currency::default()->conversion,
                    'purchaseOrderLines' => $this->getPurchaseOrderLines($data),
                ]);

                $purchaseOrder = $this->purchaseOrders->saveWithRelations(PurchaseOrderDto::collection([$purchaseOrderDto]));

                $shipment->skuLink()->associate($purchaseOrder);
                $shipment->save();
            }
        } catch (UnmappedAmazonSellerSkuException|SupplierProductNotFoundException $e) {
            $shipment->errorLog = $e->getMessage();
            $shipment->save();
            return $shipment;
        }

        return $shipment;
    }

    public function getDateFromShipmentName($shipmentName): ?Carbon
    {
        // Regular expression to match different date formats in the shipment name
        $pattern = '/(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})|(\d{2}\/\d{2}\/\d{4} \d{2}:\d{2})/';

        // Try to find a date in the shipment name
        if (preg_match($pattern, $shipmentName, $matches)) {
            // Filter out empty matches
            $dateString = array_filter($matches)[0];

            // Determine the format of the date string
            $format = !str_contains($dateString, '-') ? 'm/d/Y H:i' : 'Y-m-d H:i:s';

            // Return a Carbon instance
            return Carbon::createFromFormat($format, $dateString, $this->integrationInstance->getTimezone())->setTimezone('UTC');
        }

        return null;
    }

    private function checkLinesMatchInbound(
        AmazonInboundShipmentSourceInterface $model,
        AmazonNewFbaInboundShipment $shipment
    ): bool {
        $lineCollection = $model->{$model::getLinesRelationName()}->map(function ($line) {
            return IdQuantityData::from([
                'id' => $line->product_id,
                'quantity' => $line->quantity,
            ])->toArray();
        })->toArray();

        $shipmentLineCollection = $shipment->amazonFbaInboundShipmentItems->map(function (AmazonNewFbaInboundShipmentItem $line)
        {
            $product = $this->getProductFromShipmenItem($line);

            return IdQuantityData::from([
                'id' => $product->id,
                'quantity' => $line->quantity,
            ])->toArray();
        })->toArray();

        return $lineCollection == $shipmentLineCollection;
    }

    /**
     * @throws UnmappedAmazonSellerSkuException
     */
    private function getProductFromShipmenItem(AmazonNewFbaInboundShipmentItem $shipmentItem): Product
    {
        if (!$product = $this->fnskus->getProductFromFnsku($this->amazonIntegrationInstance,
            $shipmentItem->fnsku)) {
            $product = $shipmentItem->amazonProduct?->productListing?->product;
        }
        if (!$product) {
            throw new UnmappedAmazonSellerSkuException("$shipmentItem->msku needs to be mapped before creating this from inbound.  FNSKU: ".$shipmentItem->fnsku.' not initialized either.');
        }
        return $product;
    }

    private function getWarehouseTransferLines(AmazonNewFbaInboundShipmentWithMappingData $data, ?WarehouseTransfer $warehouseTransfer = null): DataCollection
    {
        $shipment = $data->shipment;
        return WarehouseTransferLineDto::collection($shipment->amazonFbaInboundShipmentItems->map(function (AmazonNewFbaInboundShipmentItem $shipmentItem) use ($warehouseTransfer)
        {
            if (!$product = $this->fnskus->getProductFromFnsku($this->amazonIntegrationInstance,
                $shipmentItem->fnsku)) {
                $product = $shipmentItem->amazonProduct?->productListing?->product;
            }

            throw_if(
                !$product,
                new UnmappedAmazonSellerSkuException("$shipmentItem->msku needs to be mapped before creating this warehouse transfer.  FNSKU: ".$shipmentItem->fnsku.' not initialized either.')
            );

            $lineData = [
                'product_id' => $product->id,
                'quantity' => $shipmentItem->quantity,
            ];

            if ($warehouseTransfer) {
                $lineData['warehouse_transfer_id'] = $warehouseTransfer->id;
            }

            return WarehouseTransferLineDto::from($lineData);
        }));
    }

    private function getPurchaseOrderLines(AmazonNewFbaInboundShipmentWithMappingData $data, ?PurchaseOrder $purchaseOrder = null): DataCollection
    {
        $shipment = $data->shipment;
        return PurchaseOrderLineDto::collection($shipment->amazonFbaInboundShipmentItems->map(function (AmazonNewFbaInboundShipmentItem $shipmentItem) use ($data, $purchaseOrder)
        {
            $product = $this->getProductFromShipmenItem($shipmentItem);

            /** @var SupplierProduct $defaultSupplierProduct */
            $defaultSupplierProduct = $product->supplierProducts()->where('supplier_id', $data->mapping->link_id)->first();

            if(!$defaultSupplierProduct) {
                throw new SupplierProductNotFoundException("No default supplier product found for $product->sku for supplier {$data->mapping->link->name}");
            }

            $amount = $defaultSupplierProduct->getDefaultSupplierPricing()?->price ?? $product->unit_cost;

            $lineData = [
                'product_id' => $product->id,
                'quantity' => $shipmentItem->quantity,
                'amount' => $amount,'received_quantity' => 0,
                'description' => $product->name,
                'nominal_code_id' => $product->cogs_nominal_code_id ?: Helpers::setting(Setting::KEY_NC_MAPPING_COGS),
                'created_at' => Carbon::now()->toDateTimeString(),
                'updated_at' => Carbon::now()->toDateTimeString(),
            ];

            if ($purchaseOrder) {
                $lineData['purchase_order_id'] = $purchaseOrder->id;
            }

            return PurchaseOrderLineDto::from($lineData);
        }));
    }
}
