<?php

namespace Modules\Amazon\Managers;

use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Actions\LedgerSummaryActions\AmazonLedgerSummaryErrorHandler;
use Modules\Amazon\Actions\LedgerSummaryActions\AmazonLedgerSummaryMissingWithoutSummaryReportHandler;
use Modules\Amazon\Actions\LedgerSummaryActions\AmazonLedgerSummaryTotalInventoryDiscrepancyHandler;
use Modules\Amazon\Actions\LedgerSummaryActions\AmazonLedgerSummaryUnknownEventsHandler;
use Modules\Amazon\Data\AmazonLedgerSummaryAdjustmentData;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedgerSummary;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Repositories\AmazonLedgerSummaryRepository;
use Throwable;

class AmazonLedgerSummaryManager
{
    private AmazonLedgerSummaryRepository $summaries;

    public function __construct(
        private readonly AmazonIntegrationInstance $integrationInstance,
    )
    {
        $this->summaries = app(AmazonLedgerSummaryRepository::class);
    }

    /**
     * @throws Throwable
     */
    public function reconcileAll(): Collection
    {
        $ledgerSummaryCollection = collect();
        $unreconciled = $this->summaries->getUnreconciled($this->integrationInstance);
        customlog('amazon', 'Reconciling ' . $unreconciled->count() . ' unreconciled summaries');
        $unreconciled->each(/**
         * @throws Throwable
         */ function (AmazonFbaReportInventoryLedgerSummary $ledgerSummary) use ($ledgerSummaryCollection) {
            $ledgerSummaryCollection->push($this->reconcile($ledgerSummary));
        });
        return $ledgerSummaryCollection;
    }

    /**
     * @throws Throwable
     */
    public function reconcileForFnskuProduct(AmazonFbaReportInventoryLedgerSummary $ledgerSummary): Collection
    {
        $ledgerSummaryCollection = collect();
        $this->summaries->getUnreconciledForFnskuProduct($this->integrationInstance, $ledgerSummary)->each(/**
         * @throws Throwable
         */ function (AmazonFbaReportInventoryLedgerSummary $ledgerSummary) use ($ledgerSummaryCollection) {
            $ledgerSummaryCollection->push($this->reconcile($ledgerSummary));
        });
        return $ledgerSummaryCollection;
    }

    /**
     * @throws Throwable
     */
    public function reconcile(AmazonFbaReportInventoryLedgerSummary $ledgerSummary): AmazonFbaReportInventoryLedgerSummary
    {
        return DB::transaction(function () use ($ledgerSummary)
        {
            customlog('amazon', 'Reconciling FNSKU ' . $ledgerSummary->fnsku . ', Summary ID ' . $ledgerSummary->id);
            $amazonLedgerSummaryAdjustmentDataCollection = collect();

            $ledgerSummary = app(AmazonLedgerSummaryErrorHandler::class)->handle($ledgerSummary);
            app(AmazonLedgerSummaryUnknownEventsHandler::class)->handle($ledgerSummary, $amazonLedgerSummaryAdjustmentDataCollection);
            app(AmazonLedgerSummaryTotalInventoryDiscrepancyHandler::class)->handle($ledgerSummary, $amazonLedgerSummaryAdjustmentDataCollection);
            app(AmazonLedgerSummaryMissingWithoutSummaryReportHandler::class)->handle($ledgerSummary, $amazonLedgerSummaryAdjustmentDataCollection);

            $this->summaries->linkAdjustmentsToLedgerSummary(AmazonLedgerSummaryAdjustmentData::collection($amazonLedgerSummaryAdjustmentDataCollection));

            return $this->summaries->markReconciled($ledgerSummary);
        });
    }

    public function unreconcileAll(): Collection
    {
        $ledgerSummaryCollection = collect();
        $this->summaries->getReconciled($this->integrationInstance)->each(/**
         * @throws Throwable
         */ function (AmazonFbaReportInventoryLedgerSummary $ledgerSummary) use ($ledgerSummaryCollection) {
            $ledgerSummaryCollection->push($this->unreconcile($ledgerSummary));
        });
        return $ledgerSummaryCollection;
    }

    public function unreconcileForFnskuProduct(AmazonFbaReportInventoryLedgerSummary $ledgerSummary): Collection
    {
        $ledgerSummaryCollection = collect();
        $this->summaries->getReconciledForFnskuProduct($this->integrationInstance, $ledgerSummary)->each(/**
         * @throws Throwable
         */ function (AmazonFbaReportInventoryLedgerSummary $ledgerSummary) use ($ledgerSummaryCollection) {
            $ledgerSummaryCollection->push($this->unreconcile($ledgerSummary));
        });
        return $ledgerSummaryCollection;
    }

    /**
     * @throws Throwable
     */
    public function unreconcile(AmazonFbaReportInventoryLedgerSummary $ledgerSummary): AmazonFbaReportInventoryLedgerSummary
    {
        customlog('amazon', 'Unreconciling FNSKU ' . $ledgerSummary->fnsku . ', Summary ID ' . $ledgerSummary->id);
        return $this->summaries->markUnreconciled($ledgerSummary);
    }

    /**
     * @throws Exception
     */
    public function getLastReconciledDate(): Carbon
    {
        if ($oldestUnreconciledDate = $this->summaries->getOldestUnreconciledDate($this->integrationInstance))
        {
            return Carbon::parse($oldestUnreconciledDate, $this->integrationInstance->getTimezone());
        }

        if ($lastReconciledDate = $this->summaries->getLastReconciledDate($this->integrationInstance)) {
            return Carbon::parse($lastReconciledDate, $this->integrationInstance->getTimezone());
        }

        return $this->integrationInstance->fbaInventoryTrackingStartDate()->setTimezone($this->integrationInstance->getTimezone());
    }
}