<?php

namespace Modules\Amazon\Managers;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use App\Helpers;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Actions\RequestAmazonReport;
use Modules\Amazon\Data\RequestAmazonReportData;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Exceptions\CantUnreconcileInitialInventoryWithReconciledLedgersException;
use Modules\Amazon\Repositories\AmazonFnskuRepository;
use Modules\Amazon\Repositories\AmazonLedgerRepository;
use Modules\Amazon\Services\AmazonClient;
use Throwable;

/**
 * Class AmazonInitialInventory.
 */
class AmazonInitialInventoryManager extends AbstractSalesChannelManager
{
    private AmazonLedgerRepository $ledgers;

    public function __construct(
        private readonly AmazonIntegrationInstance $amazonIntegrationInstance
    ) {
        parent::__construct($this->amazonIntegrationInstance, new AmazonClient($this->amazonIntegrationInstance));
        $this->ledgers = app(AmazonLedgerRepository::class);
    }

    /**
     *
     * The fba inventory tracking start date should be stored as the user's selected date, stored in the
     * database in UTC.  For example, if the user wants a start date of Jan 1, 2024 and is in PST timezone.  The date will
     * be stored in the database as 2024-01-01 08:00:00 UTC.  For the initial inventory report, we are going to take the
     * start of the day 1 day before the fba inventory start date, in local timezone and convert back to UTC
     * 2023-12-31 08:00:00 UTC.  The end of day will be teh end of the day in local timezone, converted back to UTC, which
     * will be 2023-01-01 07:59:59 UTC.
     *
     * @throws Exception
     * @throws Throwable
     */
    public function requestInitialInventory(): void
    {
        customlog('amazon', 'Requesting Initial Inventory');
        (new RequestAmazonReport(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
            'report_type' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY,
            'data_start_date' => Helpers::utcStartOfLocalDate($this->amazonIntegrationInstance->fbaInventoryTrackingStartDate()->subDay()),
            'data_end_date' => Helpers::utcEndOfLocalDate($this->amazonIntegrationInstance->fbaInventoryTrackingStartDate()->subDay()),
            'options' => [
                'aggregatedByTimePeriod' => 'DAILY',
            ],
        ])))->handle();
    }

    /**
     * This should be run once a product listing is created so that the initial inventory can be properly reconciled.
     *
     * @throws Exception
     * @throws Throwable
     */
    public function reconcileAllInitialInventory(): Collection
    {
        customlog('amazon', 'Reconciling FBA Initial Inventory');

        $initialInventoryCollection = collect();

        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->generateFnskuProducts();

        $unreconciledInitialInventory = $this->ledgers->getUnreconciledFromInitialInventoryReport($this->amazonIntegrationInstance);

        customlog('amazon', 'Found '.$unreconciledInitialInventory->count().' initial inventory items to reconcile');

        $unreconciledInitialInventory->each(function (AmazonFbaInitialInventory $initialInventory) use ($initialInventoryCollection) {
            $initialInventoryCollection->push($this->reconcile($initialInventory, initialCount: true));
        });

        $productIds = $initialInventoryCollection->pluck('amazonFnskuProduct.product_id')->unique()->toArray();

        (new UpdateProductsInventoryAndAvgCost($productIds))->handle();

        return $initialInventoryCollection;
    }

    public function reconcileInitialInventoryForFnskuProducts(array $fnskuProductIds): Collection
    {
        $initialInventories = AmazonFbaInitialInventory::with('amazonFnskuProduct', 'fifoLayer')
            ->whereHas('amazonFnskuProduct', function ($query) use ($fnskuProductIds) {
                $query->whereIn('id', $fnskuProductIds);
            })
            ->whereDoesntHave('fifoLayer')
            ->orderBy('fnsku')
            ->get();

        $initialInventoryCollection = collect();
        $initialInventories->each(/**
         * @throws Throwable
         */ function (AmazonFbaInitialInventory $initialInventory) use ($initialInventoryCollection) {
            $initialInventoryCollection->push($this->reconcile($initialInventory));
        });
        return $initialInventoryCollection;
    }

    public function reconcileInitialInventories(array $initialInventoryIds): Collection
    {
        $initialInventories = AmazonFbaInitialInventory::with('amazonFnskuProduct', 'fifoLayer')
            ->whereIn('id', $initialInventoryIds)
            ->whereDoesntHave('fifoLayer')
            ->orderBy('fnsku')
            ->get();

        $initialInventoryCollection = collect();
        $initialInventories->each(/**
         * @throws Throwable
         */ function (AmazonFbaInitialInventory $initialInventory) use ($initialInventoryCollection) {
            $initialInventoryCollection->push($this->reconcile($initialInventory));
        });
        return $initialInventoryCollection;
    }

    /**
     * @throws Throwable
     */
    public function reconcile(AmazonFbaInitialInventory $initialInventory, $updateInventory = false, $initialCount = false): AmazonFbaInitialInventory
    {
        throw_if(
            ! $initialInventory->integrationInstance->fbaInventoryTrackingStartDate(),
            'Unable to initialize FBA inventory since FBA inventory tracking start date is not set (ID: '.$initialInventory->amazonFnskuProduct->product->id.')'
        );

        return DB::transaction(function () use ($initialInventory, $updateInventory, $initialCount) {
            $amazonProductManager = new AmazonProductManager($this->amazonIntegrationInstance);

            $fifoLayer = $amazonProductManager->initializeFbaInventoryFifoLayer($initialInventory, $initialCount);

            $initializationDate = Carbon::now();

            $initialInventory->fifo_layer_id = $fifoLayer->id;
            $initialInventory->sku_product_initialized_at = $initializationDate;
            $initialInventory->save();

            $fnskuProduct = $initialInventory->amazonFnskuProduct;

            if ($updateInventory)
            {
                (new UpdateProductsInventoryAndAvgCost($fnskuProduct->product_id))->handle();
            }

            return $initialInventory;
        });
    }

    /**
     * @throws Throwable
     */
    public function unreconcileAllInitialInventory(): Collection
    {
        customlog('amazon', 'Unreconciling FBA Initial Inventory');

        $initialInventoryCollection = collect();

        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->generateFnskuProducts();

        $reconciledInitialInventory = $this->ledgers->getReconciledFromInitialInventoryReport($this->amazonIntegrationInstance);

        customlog('amazon', 'Found '.$reconciledInitialInventory->count().' initial inventory items to unreconcile');

        $reconciledInitialInventory->each(function (AmazonFbaInitialInventory $initialInventory) use ($initialInventoryCollection) {
            $initialInventoryCollection->push($this->unreconcile($initialInventory));
        });

        $productIds = $initialInventoryCollection->pluck('amazonFnskuProduct.product_id')->unique()->toArray();

        (new UpdateProductsInventoryAndAvgCost($productIds))->handle();

        return $initialInventoryCollection;
    }

    public function unreconcileInitialInventoryForFnskuProducts(array $fnskuProductIds): Collection
    {
        $initialInventories = AmazonFbaInitialInventory::with('amazonFnskuProduct', 'fifoLayer')
            ->whereHas('amazonFnskuProduct', function ($query) use ($fnskuProductIds) {
                $query->whereIn('id', $fnskuProductIds);
            })
            ->whereHas('fifoLayer')
            ->orderBy('fnsku')
            ->get();

        $initialInventoryCollection = collect();
        $initialInventories->each(function (AmazonFbaInitialInventory $initialInventory) use ($initialInventoryCollection) {
            $initialInventoryCollection->push($this->unreconcile($initialInventory));
        });
        return $initialInventoryCollection;
    }

    public function unreconcileInitialInventories(array $initialInventoryIds): Collection
    {
        $initialInventories = AmazonFbaInitialInventory::with('amazonFnskuProduct', 'fifoLayer')
            ->whereIn('id', $initialInventoryIds)
            ->whereHas('fifoLayer')
            ->orderBy('fnsku')
            ->get();

        $initialInventoryCollection = collect();
        $initialInventories->each(/**
         * @throws Throwable
         */ function (AmazonFbaInitialInventory $initialInventory) use ($initialInventoryCollection) {
            $initialInventoryCollection->push($this->unreconcile($initialInventory));
        });
        return $initialInventoryCollection;
    }

    /**
     * @throws Throwable
     */
    public function unreconcile(AmazonFbaInitialInventory $initialInventory): AmazonFbaInitialInventory
    {
        DB::transaction(function () use ($initialInventory) {
            $fifoLayer = $initialInventory->fifoLayer;
            $fnskuProduct = $initialInventory->amazonFnskuProduct;
            $reconciledLedgersCount = $fnskuProduct->amazonFbaReportInventoryLedgers->whereNotNull('reconciled_at')->count();
            if ($reconciledLedgersCount > 0) {
                throw new CantUnreconcileInitialInventoryWithReconciledLedgersException(
                    "Cannot unreconcile initial inventory with $reconciledLedgersCount reconciled ledgers"
                );
            }
            $initialInventory->sku_product_initialized_at = null;
            $initialInventory->fifo_layer_id = null;
            $initialInventory->save();
            $fifoLayer->delete();
            if ($fnskuProduct)
            {
                $fnskuProduct->reconciled_quantity = 0;
                $fnskuProduct->save();
            }
        });
        return $initialInventory;
    }
}
